use anyhow::Context;
use forgejo_api::structs::Repository;

use crate::actions::GeneralArgs;
use crate::render::json::JsonToStdout;
use crate::render::option::option_display;
use crate::render::spinner::spin_until_ready;
use crate::types::context::BergContext;
use crate::types::git::OwnerRepo;
use crate::types::output::OutputMode;

use super::parse_owner_and_repo;

use clap::Parser;

/// Display short summary of the current repository
#[derive(Parser, Debug)]
pub struct RepoInfoArgs {
    /// Repository to be queried for information, defaults to repo in $pwd if there is any
    #[arg(value_name = "OWNER/REPO")]
    pub owner_and_repo: Option<String>,
}

impl RepoInfoArgs {
    pub async fn run(self, general_args: GeneralArgs) -> anyhow::Result<()> {
        let ctx = BergContext::new(self, general_args).await?;

        let (owner, repo) = ctx
            .args
            .owner_and_repo
            .as_ref()
            .and_then(|s| parse_owner_and_repo(s).ok())
            .unwrap_or(
                ctx.owner_repo()
                    .map(|OwnerRepo { repo, owner }| (owner, repo))?,
            );
        let repo_data = spin_until_ready(ctx.client.repo_get(owner.as_str(), repo.as_str()))
            .await
            .context("Current repo not found on this instance.")?;

        match general_args.output_mode {
            OutputMode::Pretty => {
                present_repo_info(&ctx, repo_data);
            }
            OutputMode::Json => repo_data.print_json()?,
        }

        Ok(())
    }
}

fn present_repo_info(ctx: &BergContext<RepoInfoArgs>, repo_data: Repository) {
    let mut table = ctx.make_table();

    table
        .set_header(vec!["Repository Info"])
        .add_row(vec![
            String::from("Repository Name"),
            option_display(&repo_data.name),
        ])
        .add_row(vec![
            String::from("Repository Owner"),
            option_display(
                &repo_data
                    .owner
                    .as_ref()
                    .and_then(|user| user.login.as_ref()),
            ),
        ])
        .add_row(vec![
            String::from("Stars"),
            format!("{}★", option_display(&repo_data.stars_count)),
        ]);

    println!("{table}", table = table.show());
}
