{
  flake.testLib =
    {
      self',
      lib,
      pkgs,
    }:
    let
      scopes =
        [
          "activitypub"
          "issue"
          "notification"
          "organization"
          "package"
          "repository"
          "user"
        ]
        |> builtins.map (scope: "write:${scope}")
        |> lib.concatStringsSep ",";
    in
    rec {
      testRepo = "testingRepo";
      user = "adminx";
      email = "admin@example.com";

      berg_cmd = "BERG_BASE_URL=forgejo:3000 BERG_PROTOCOL=http berg";

      berg_run =
        {
          cmd,
          withTestRepo ? false,
          success ? true,
          timeout ? 3,
        }:
        let
          # python linter says fstrings without any parameters are an error so we have to correct the cmd
          fstringRequired = (builtins.match ".*\\\{.*" cmd) != null;
          prefix = lib.optionalString fstringRequired "f";
          fixedCmd = lib.escape [ ''"'' ] cmd;

          cdTestRepo = lib.optionalString withTestRepo "cd ${testRepo}; ";
        in
        [
          "forgejo."
          (if success then "wait_until_succeeds" else "fail")
          "("
          prefix
          "\""
          cdTestRepo
          berg_cmd
          " --non-interactive"
          " ${fixedCmd}"
          "\""
          ",timeout=${builtins.toString timeout}"
          ")"
        ]
        |> builtins.concatStringsSep "";

      berg_run_json =
        {
          cmd,
          withTestRepo ? false,
          success ? true,
          timeout ? 3,
        }:
        ''json.loads(${
          berg_run {
            inherit withTestRepo success timeout;
            cmd = "--output-mode json ${cmd}";
          }
        })'';

      # forgejo CLI with all the necessary arguments
      forgejo_run = "su -- forgejo gitea admin user --work-path /var/lib/forgejo";

      forgjeoTest =
        {
          name,
          test,
          withRepoSetup ? false,
          withAuth ? false,
        }:
        pkgs.testers.runNixOSTest {
          name = "berg-forgejo-integration-test-${name}";
          nodes = {
            forgejo = {
              environment.systemPackages = [
                pkgs.curl
                pkgs.forgejo
                pkgs.git
                self'.packages.berg
              ];

              services.forgejo = {
                enable = true;
                package = pkgs.forgejo;
                settings = {
                  default = {
                    RUN_MODE = "dev";
                  };
                  server = {
                    ROOT_URL = "forgejo:3000";
                  };
                };
              };

              system.stateVersion = "25.05";
            };
          };

          testScript =
            [
              # imports
              ''
                import json
                def printj(val):
                  print(json.dumps(val, indent=2))
              ''
              # forgejo setup
              ''

                forgejo.wait_for_open_port(3000)
                forgejo.succeed("${forgejo_run} create --email ${email} --username ${user} --password pass")

                # generate an access token
                token = forgejo.succeed("${forgejo_run} generate-access-token --username ${user} --scopes '${scopes}' --raw")
                print(token)
              ''
              # auth login test
              (lib.optionalString withAuth ''
                auth = ${berg_run { cmd = "auth login --token {token}"; }}
                print(auth)
              '')
              # create a repo
              (lib.optionalString withRepoSetup ''
                ${berg_run { cmd = ''repo create --name "${testRepo}"''; }}
                print("Successfully created repo ${testRepo}")

                ${berg_run { cmd = ''repo clone ${user}/${testRepo}''; }}
                print("Successfully cloned ${user}/${testRepo}")
              '')
              # the actual test case
              test
              # Final message
              ''
                print("Success")
              ''
            ]
            |> builtins.concatStringsSep "\n"
            |> lib.traceVal;
        };
      embeddInFlake =
        testAttrs:
        testAttrs
        |> builtins.mapAttrs (testName: testConfig: testConfig // { name = testName; })
        |> builtins.mapAttrs (_: forgjeoTest);
    };
}
