use std::fmt;
use std::str::FromStr;
use structopt::clap::AppSettings;
use structopt::StructOpt;

#[derive(Debug)]
pub enum SubcommandError {
    Unknown,
}

impl fmt::Display for SubcommandError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            SubcommandError::Unknown => write!(f, "Unknown flag."),
        }
    }
}

#[derive(StructOpt, Debug, Clone)]
pub enum Subcommand {
    #[structopt(
        visible_alias = "B",
        about =  "Show a backlight notification", 
        settings = &[AppSettings::ColoredHelp, AppSettings::DisableHelpFlags, AppSettings::DisableVersion]
    )]
    Backlight,
    #[structopt(
        visible_alias = "b",
        about = "Show a battery notification", 
        settings = &[AppSettings::ColoredHelp, AppSettings::DisableHelpFlags, AppSettings::DisableVersion]
    )]
    Battery,
    #[structopt(
        visible_alias = "n",
        about = "Show a network notification",
        settings = &[AppSettings::ColoredHelp, AppSettings::DisableHelpFlags, AppSettings::DisableVersion]
    )]
    Network,
    #[structopt(
        visible_alias = "u",
        about = "Show an uptime notification",
        settings = &[AppSettings::ColoredHelp, AppSettings::DisableHelpFlags, AppSettings::DisableVersion]
    )]
    Uptime,
    #[structopt(
        visible_alias = "m",
        about = "Show a memory notification",
        settings = &[AppSettings::ColoredHelp, AppSettings::DisableHelpFlags, AppSettings::DisableVersion]
    )]
    Memory,
    #[structopt(
        visible_alias = "d",
        about = "Show a date notification",
        settings = &[AppSettings::ColoredHelp, AppSettings::DisableHelpFlags, AppSettings::DisableVersion]
    )]
    Date,
}

impl fmt::Display for Subcommand {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            Subcommand::Backlight => write!(f, "Backlight"),
            Subcommand::Battery => write!(f, "Battery"),
            Subcommand::Network => write!(f, "Network"),
            Subcommand::Uptime => write!(f, "Uptime"),
            Subcommand::Memory => write!(f, "Memory"),
            Subcommand::Date => write!(f, "Date"),
        }
    }
}

impl FromStr for Subcommand {
    type Err = SubcommandError;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match &s.to_lowercase()[..] {
            "backlight" => Ok(Subcommand::Backlight),
            "battery" => Ok(Subcommand::Battery),
            "network" => Ok(Subcommand::Network),
            "uptime" => Ok(Subcommand::Uptime),
            "memory" => Ok(Subcommand::Memory),
            "date" => Ok(Subcommand::Date),
            _ => Err(SubcommandError::Unknown),
        }
    }
}

#[derive(Debug, StructOpt, Default)]
#[structopt(settings = &[AppSettings::ArgsNegateSubcommands, AppSettings::ArgRequiredElseHelp, AppSettings::DisableHelpSubcommand])]
pub struct Options {
    /// Notifies you about your system's uptime.
    #[structopt(subcommand, help = "Send a notification with uptime information")]
    pub cmd: Option<Subcommand>,

    /// Lists the available icon themes.
    #[structopt(short = "L", help = "Returns a list of the available icon themes")]
    pub icon_themes: bool,
}
