use crate::fmt::date::{Day, Meridiem, Month};
use crate::fmt::{Format, Iconize};
use std::fmt;
use time::Weekday;

#[derive(Debug, PartialEq)]
pub struct DateFormatter {
    hour: u8,
    minute: u8,
    day: u8,
    month: Month,
    railway: bool,
    weekday: Day,
    meridiem: Meridiem,
}

impl DateFormatter {
    pub fn new() -> Self {
        Self {
            hour: 0,
            minute: 0,
            day: 0,
            month: Month::January,
            railway: false,
            meridiem: Meridiem::Ante,
            weekday: Day::Monday,
        }
    }

    pub fn set_railway(&mut self, railway: bool) {
        self.railway = railway;
    }

    pub fn set_hour(&mut self, hour: u8) {
        if self.railway {
            self.hour = hour;
            return;
        }

        if hour == 0 {
            self.hour = hour + 12;
            self.meridiem = Meridiem::Ante;
        } else if hour > 0 && hour < 13 {
            self.hour = hour;
            self.meridiem = Meridiem::Ante;
        } else {
            self.hour = hour - 12;
            self.meridiem = Meridiem::Post;
        }
    }

    pub fn set_minute(&mut self, minute: u8) {
        self.minute = minute;
    }

    pub fn set_day(&mut self, day: u8) {
        self.day = day;
    }

    pub fn set_month(&mut self, month: time::Month) {
        self.month = month.into();
    }

    pub fn set_weekday(&mut self, weekday: Weekday) {
        self.weekday = weekday.into();
    }
}

impl Iconize for DateFormatter {
    fn set_icon(&self) -> &str {
        let night = if self.railway {
            self.hour >= 20 || self.hour <= 6
        } else {
            self.hour >= 8 && self.meridiem == Meridiem::Post
                || self.hour <= 6 && self.meridiem == Meridiem::Ante
        };

        if night {
            "weather-clear-night-symbolic"
        } else {
            "weather-clear-symbolic"
        }
    }
}

impl Format for DateFormatter {
    fn with_format(&self, format: String) -> String {
        format
            .replace("%HH", &format!("{:0>2}", &self.hour))
            .replace("%MM", &format!("{:0>2}", &self.minute))
            .replace("%P", &self.meridiem.to_string())
            .replace("%w", &self.weekday.short().to_string())
            .replace("%W", &self.weekday.long().to_string())
            .replace("%m", &self.month.short().to_string())
            .replace("%M", &self.month.long().to_string())
            .replace("%d", &self.day.to_string())
    }
}

impl fmt::Display for DateFormatter {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if self.railway {
            write!(
                f,
                "{}, {} {:02} at {:02}:{:02}",
                self.weekday.short(),
                self.month.short(),
                self.day,
                self.hour,
                self.minute,
            )
        } else {
            write!(
                f,
                "{}, {} {:02} at {:02}:{:02} {}",
                self.weekday.short(),
                self.month.short(),
                self.day,
                self.hour,
                self.minute,
                self.meridiem,
            )
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn display() {
        let mut fmt = DateFormatter::new();
        fmt.set_weekday(Weekday::Saturday);
        fmt.set_day(1);
        fmt.set_month(time::Month::August);
        fmt.set_hour(14);
        fmt.set_minute(37);
        assert_eq!(fmt.to_string(), "Sat, Aug 01 at 02:37 PM".to_owned());

        fmt.set_railway(true);
        fmt.set_weekday(Weekday::Monday);
        fmt.set_day(13);
        fmt.set_month(time::Month::September);
        fmt.set_hour(16);
        fmt.set_minute(12);
        assert_eq!(fmt.to_string(), "Mon, Sep 13 at 16:12".to_owned());
    }

    #[test]
    fn display_with_format() {
        let mut fmt = DateFormatter::new();

        fmt.set_weekday(Weekday::Monday);
        let f = String::from("%w");
        assert_eq!(fmt.with_format(f), "Mon".to_string());

        fmt.set_weekday(Weekday::Saturday);
        let f = String::from("%W");
        assert_eq!(fmt.with_format(f), "Saturday".to_string());

        fmt.set_month(time::Month::June);
        let f = String::from("%m");
        assert_eq!(fmt.with_format(f), "Jun".to_string());

        fmt.set_month(time::Month::July);
        let f = String::from("%M");
        assert_eq!(fmt.with_format(f), "July".to_string());

        fmt.set_day(12);
        let f = String::from("%d");
        assert_eq!(fmt.with_format(f), "12".to_string());

        fmt.set_hour(12);
        fmt.set_minute(20);
        let f = String::from("%HH:%MM %P");
        assert_eq!(fmt.with_format(f), "12:20 AM".to_string());

        fmt.set_hour(23);
        fmt.set_minute(20);
        let f = String::from("%HH:%MM %P");
        assert_eq!(fmt.with_format(f), "11:20 PM".to_string());

        fmt.set_railway(true);
        fmt.set_hour(20);
        fmt.set_minute(20);
        let f = String::from("%HH:%MM");
        assert_eq!(fmt.with_format(f), "20:20".to_string());
    }

    #[test]
    fn pick_icon() {
        let mut fmt = DateFormatter::new();
        fmt.set_weekday(Weekday::Sunday);
        fmt.set_day(1);
        fmt.set_month(time::Month::August);
        fmt.set_hour(14);
        fmt.set_minute(37);
        assert_eq!(fmt.set_icon(), "weather-clear-symbolic");
    }
}
