use super::{Format, Iconize};
use libmacchina::traits::BatteryState;
use std::fmt;

pub struct BatteryFormatter {
    percentage: Option<u8>,
    state: Option<BatteryState>,
}

impl BatteryFormatter {
    pub fn new() -> Self {
        Self {
            percentage: None,
            state: None,
        }
    }

    pub fn get_state_as_char(&self) -> Option<char> {
        if let Some(s) = self.state.as_ref() {
            match s {
                BatteryState::Charging => Some('+'),
                BatteryState::Discharging => match self.percentage {
                    Some(0..=10) => Some('!'),
                    Some(11..=30) => Some('-'),
                    _ => None,
                },
            }
        } else {
            None
        }
    }

    pub fn set_percentage(&mut self, p: u8) {
        self.percentage = Some(p);
    }

    pub fn get_percentage(&self) -> Option<String> {
        self.percentage.map(|p| format!("{}%", p))
    }

    pub fn get_state(&self) -> Option<String> {
        self.state.as_ref().map(|s| s.to_string())
    }

    pub fn set_state(&mut self, s: BatteryState) {
        self.state = Some(s);
    }
}

impl Iconize for BatteryFormatter {
    fn set_icon(&self) -> &str {
        match (self.percentage, &self.state) {
            (Some(0..=9), Some(BatteryState::Charging)) => "battery-level-0-charging-symbolic",
            (Some(0..=9), Some(BatteryState::Discharging)) => "battery-level-0-symbolic",
            (Some(10..=19), Some(BatteryState::Charging)) => "battery-level-10-charging-symbolic",
            (Some(10..=19), Some(BatteryState::Discharging)) => "battery-level-10-symbolic",
            (Some(20..=29), Some(BatteryState::Charging)) => "battery-level-20-charging-symbolic",
            (Some(20..=29), Some(BatteryState::Discharging)) => "battery-level-20-symbolic",
            (Some(30..=39), Some(BatteryState::Charging)) => "battery-level-30-charging-symbolic",
            (Some(30..=39), Some(BatteryState::Discharging)) => "battery-level-30-symbolic",
            (Some(40..=49), Some(BatteryState::Charging)) => "battery-level-40-charging-symbolic",
            (Some(40..=49), Some(BatteryState::Discharging)) => "battery-level-40-symbolic",
            (Some(50..=59), Some(BatteryState::Charging)) => "battery-level-50-charging-symbolic",
            (Some(50..=59), Some(BatteryState::Discharging)) => "battery-level-50-symbolic",
            (Some(60..=69), Some(BatteryState::Charging)) => "battery-level-60-charging-symbolic",
            (Some(60..=69), Some(BatteryState::Discharging)) => "battery-level-60-symbolic",
            (Some(70..=79), Some(BatteryState::Charging)) => "battery-level-70-charging-symbolic",
            (Some(70..=79), Some(BatteryState::Discharging)) => "battery-level-70-symbolic",
            (Some(80..=89), Some(BatteryState::Charging)) => "battery-level-80-charging-symbolic",
            (Some(80..=89), Some(BatteryState::Discharging)) => "battery-level-80-symbolic",
            (Some(90..=100), Some(BatteryState::Charging)) => "battery-level-90-charging-symbolic",
            (Some(90..=100), Some(BatteryState::Discharging)) => "battery-level-90-symbolic",
            (Some(_), Some(_)) => "battery-full-symbolic",
            _ => "battery-missing-symbolic",
        }
    }
}

impl Format for BatteryFormatter {
    fn with_format(&self, format: String) -> String {
        if self.percentage.is_none() {
            return String::new();
        }

        let mut format = format
            .replace("%p", &self.get_percentage().unwrap_or_default())
            .replace("%s", &self.get_state().unwrap_or_default());

        if format.contains("%b") {
            if let Some(symbol) = self.get_state_as_char() {
                format = format.replace("%b", &symbol.to_string());
            } else {
                format = format.replace("%b", "");
            }
        }

        format
    }
}

impl fmt::Display for BatteryFormatter {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if let Some(p) = self.percentage {
            if p == 100 {
                return write!(f, "State: Full");
            }

            writeln!(f, "Percentage: {}%", p);

            if let Some(s) = &self.state {
                return write!(f, "State: {}", s);
            }

            return write!(f, "State: Unknown");
        }

        return write!(f, "No battery detected.");
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn display() {
        let mut fmt = BatteryFormatter::new();
        assert_eq!(fmt.to_string(), "No battery detected.".to_owned());

        fmt.set_percentage(50);
        assert_eq!(
            fmt.to_string(),
            "Percentage: 50%\nState: Unknown".to_owned()
        );

        fmt.set_state(BatteryState::Charging);
        assert_eq!(
            fmt.to_string(),
            "Percentage: 50%\nState: Charging".to_owned()
        );

        fmt.set_percentage(100);
        assert_eq!(fmt.to_string(), "State: Full".to_owned());
    }

    #[test]
    fn disp_percentage_state() {
        let mut fmt = BatteryFormatter::new();
        fmt.set_percentage(80);
        fmt.set_state(BatteryState::Discharging);
        let f = String::from("%p\n%s");
        assert_eq!(fmt.with_format(f), "80%\nDischarging".to_owned());
    }

    #[test]
    fn disp_bp_specf_charging() {
        let mut fmt = BatteryFormatter::new();
        fmt.set_percentage(50);
        fmt.set_state(BatteryState::Charging);
        let f = String::from("%b%p");
        assert_eq!(fmt.with_format(f), "+50%".to_owned());
    }

    #[test]
    fn disp_bp_specf_warning() {
        let mut fmt = BatteryFormatter::new();
        fmt.set_percentage(10);
        fmt.set_state(BatteryState::Discharging);
        let f = String::from("%b%p");
        assert_eq!(fmt.with_format(f), "!10%".to_owned());
    }

    #[test]
    fn disp_bp_specf_discharging() {
        let mut fmt = BatteryFormatter::new();
        fmt.set_percentage(28);
        fmt.set_state(BatteryState::Discharging);
        let f = String::from("%b%p");
        assert_eq!(fmt.with_format(f), "-28%".to_owned());
    }

    #[test]
    fn set_icon_level_80_discharging() {
        let mut fmt = BatteryFormatter::new();
        fmt.set_percentage(85);
        fmt.set_state(BatteryState::Discharging);
        assert_eq!(fmt.set_icon(), "battery-level-80-symbolic");
    }

    #[test]
    fn set_icon_level_20_charging() {
        let mut fmt = BatteryFormatter::new();
        fmt.set_percentage(20);
        fmt.set_state(BatteryState::Charging);
        assert_eq!(fmt.set_icon(), "battery-level-20-charging-symbolic");
    }
}
