/*
 * CHEST, chess analyst.  For Copyright notice read file "COPYRIGHT".
 *
 * $Source: /home/heiner/ca/chest/RCS/cost.h,v $
 * $Id: cost.h,v 1.3 1999/07/20 21:18:13 heiner Exp $
 *
 *	Cost of (sub-) computations & value of stored entries
 */

#ifndef CHEST_cost_h_INCLUDED
#define CHEST_cost_h_INCLUDED

#include "bastyp.h"

/* FFS: PROD_LEV */

/*
 * We use "cost" to approximate the amount of computation that has
 * been used to derive the result of some sub-computation.
 * This is especially the "value" of ACM-nodes.
 * Basically we count certain procedure calls.
 *
 * We do this, because measuring the true time would involve
 * system calls, which is (or may be) far too expensive.
 *
 * We use the approximate computation cost to guide the cache replacement
 * policy, and to compute an approximate "speed" value.
 *
 * However, 32-bit quantities are somewhat limited.
 * Some statistics: on a P-166 we do
 * ca 80000 cost         per second
 * ca  8500 ACM searches per second
 * Wrapping an 32-bit unsigned (4*10^9):
 * 8*10^4/s == 4*10^9/(5*10^4 s)
 * 50000 secs = 833 mins = 13.8 hours
 * 138 hours = 5.75  days
 * Even the counters for ACM searches may wrap after some days.
 *
 * Operations on "cost" values:
 * - clear (set zero) [=0]
 * - test for zero value
 * - assignment (copy) [=]
 * - increment (by one) [++]
 * - add positive int (answer sorting cost) [+=]
 * - add cost to cost [+=]
 * - subtract smaller from larger value, yielding a delta cost [-=]
 * - divide cost by counter or cost [/]
 * - print value
 *
 * Usage outline:
 * - "apx_cost" and "apx_ovhd" are implemented in "analyse.c".
 *   They are increased in "analyse.c", only.
 *   They are printed from "acm.c" (total summary) and "trace.c",
 *   where move tracing saves an old value and prints a difference.
 * - "apx_saved" is local to "acm.c", and is increased by acm_used().
 * - sum_icost/cnt_icost in "acm.c" is tracking the summed cost per info.
 *   It is updated by acm_note() and used in acm_used().  Thus ACM nodes
 *   need not each a cost, just an average per info.
 *   At least sum_icost shall be larger than 32 bit.
 * - acm_note() gets a cost as argument.  Should be of type AnaCost.
 *   Called from "analyse.c", only.
 *
 * Possible implementations:
 * (a) 32-bit unsigned: this is too small
 * (b) 64-bit ints (long long): are not always accessible
 * (c) doubles (53 bit mantissa): may be slow
 * (d) 32-bit[2] in struct: may be slower than other implementations
 * The struct approach (d) and the double approach (c) are both very portable.
 *
 * How to decide for an implementation:
 * - look at assembler code
 * - perform profiling/timing
 */

#ifndef  COST_AS_DOUBLE
# define COST_AS_DOUBLE		1	/* default: as double */
#endif /* COST_AS_DOUBLE */

#if COST_AS_DOUBLE
  typedef double	 AnaCost;
  typedef double	rAnaCost;
#else	/* ! COST_AS_DOUBLE */
  typedef uint32	 AnaCost;
  typedef uint32	rAnaCost;
#endif	/* ! COST_AS_DOUBLE */

Extern AnaCost	apx_cost;	/* approximate cost of analysis */
Extern AnaCost	apx_ovhd;	/* approximate overhead cost of memory */

#endif	/* ndef CHEST_cost_h_INCLUDED */
