#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <stdbool.h>
#include <pwd.h>
#include <grp.h>
#include <err.h>

#include <sys/param.h>

#include <openssl/ssl.h>
#include <openssl/asn1.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>
#include <openssl/err.h>
#include <openssl/ossl_typ.h>
#include <openssl/ocsp.h>

/* begin copied from openssl/crypto/ocsp/ocsp_local.h */

struct ocsp_response_st {   
    ASN1_ENUMERATED *responseStatus;
    OCSP_RESPBYTES *responseBytes;
};

struct ocsp_resp_bytes_st { 
    ASN1_OBJECT *responseType;
    ASN1_OCTET_STRING *response;
}; 

struct ocsp_responder_id_st {
    int type;
    union {
        X509_NAME *byName;  
        ASN1_OCTET_STRING *byKey;
    } value;
};  

struct ocsp_response_data_st {
    ASN1_INTEGER *version;  
    OCSP_RESPID responderId;
    ASN1_GENERALIZEDTIME *producedAt;
    STACK_OF(OCSP_SINGLERESP) *responses;
    STACK_OF(X509_EXTENSION) *responseExtensions;
};  
struct ocsp_basic_response_st {
    OCSP_RESPDATA tbsResponseData;
    X509_ALGOR signatureAlgorithm;
    ASN1_BIT_STRING *signature;
    STACK_OF(X509) *certs;
};

struct ocsp_single_response_st {
    OCSP_CERTID *certId;    
    OCSP_CERTSTATUS *certStatus;
    ASN1_GENERALIZEDTIME *thisUpdate;
    ASN1_GENERALIZEDTIME *nextUpdate;
    STACK_OF(X509_EXTENSION) *singleExtensions;
};  

struct ocsp_cert_status_st {
    int type;
    union {
        ASN1_NULL *good;    
        OCSP_REVOKEDINFO *revoked;
        ASN1_NULL *unknown;
    } value;
};  

/* end of copied from openssl/crypto/ocsp/ocsp_local.h */


#define NAGIOS_OK	0
#define NAGIOS_WARN	1 
#define NAGIOS_CRIT	2
#define NAGIOS_UNKNOWN	3

#define DEFAULT_CRIT 3
#define DEFAULT_WARN 7
#define DEFAULT_CMD "/usr/pkg/sbin/httpd -S -E /dev/null"

#define MSG_MAXLEN	256

struct conf {
	int warn;
	int crit;
	int verbose;
	int req_ocsp_stapling;
	char *user;
	char *cmd;
};

struct ocsp_stapling_arg {
	int status;
	char *msg;
	size_t msglen;
};

/* From curl's hostcheck.c */
bool Curl_cert_hostcheck(const char *match, size_t matchlen,
                         const char *hostname, size_t hostlen);

static int
ocsp_stapling_cb(SSL *ssl, void *arg)
{
	struct ocsp_stapling_arg *osa = (struct ocsp_stapling_arg *)arg;
	int len;
	const unsigned char *buf = NULL;
	OCSP_RESPONSE *or = NULL;
	long ors;
	OCSP_RESPBYTES *orb;
	OCSP_BASICRESP *obr;
	OCSP_RESPDATA *ord;
	int i;

	len = SSL_get_tlsext_status_ocsp_resp(ssl, &buf);
	if (buf == NULL) {
		osa->status = NAGIOS_CRIT;
		snprintf(osa->msg, osa->msglen, "no response");
		goto out;
	}
	
	or = d2i_OCSP_RESPONSE(NULL, &buf, len);
	if (or == NULL)
		goto out;

	ors = ASN1_ENUMERATED_get(or->responseStatus);
	if (ors != OCSP_RESPONSE_STATUS_SUCCESSFUL) {
		osa->status = NAGIOS_CRIT;
		snprintf(osa->msg, osa->msglen, "response status %s",
			 OCSP_response_status_str(ors));
		goto out;
	}

	orb = or->responseBytes;
	if (i2a_ASN1_OBJECT(NULL, orb->responseType) <= 0)
		goto out;

	if (OBJ_obj2nid(orb->responseType) != NID_id_pkix_OCSP_basic)
		goto out;

	if ((obr = OCSP_response_get1_basic(or)) == NULL)
		goto out;

	ord = &obr->tbsResponseData;
	for (i = 0; i < sk_OCSP_SINGLERESP_num(ord->responses); i++) {
		OCSP_SINGLERESP *osr;

		osr = sk_OCSP_SINGLERESP_value(ord->responses, i);
		if (osr == NULL)
			continue;

		switch (osr->certStatus->type) {
		case V_OCSP_CERTSTATUS_GOOD:
			break;
		case V_OCSP_CERTSTATUS_REVOKED:
			osa->status = NAGIOS_CRIT;
			break;
		case V_OCSP_CERTSTATUS_UNKNOWN: /* FALLTHROUGH */
		default:
			osa->status = NAGIOS_UNKNOWN;
			break;
		}

		snprintf(osa->msg, osa->msglen, "cert status %s",
			 OCSP_cert_status_str(osr->certStatus->type));
		break;
	}

	osa->status = NAGIOS_OK;

out:
	if (osa->status == NAGIOS_UNKNOWN) {
		snprintf(osa->msg, osa->msglen, "OpenSSL error %s",
			 ERR_error_string(ERR_get_error(), NULL));
	}

	if (or)
		OCSP_RESPONSE_free(or);

	return 1;
}

int
check_subject(char *host, X509 *cert)
{
	int ret = 0;
	X509_NAME *subject;
	const X509_NAME_ENTRY *cn;
	const ASN1_STRING *asn1str;
	const char *str;
	int index;
	int len;

	if ((subject = X509_get_subject_name(cert)) == NULL)
		goto out;

	index = X509_NAME_get_index_by_NID(subject, NID_commonName, -1);
	if ((cn = X509_NAME_get_entry(subject, index)) == NULL)
		goto out;

	if ((asn1str = X509_NAME_ENTRY_get_data(cn)) == NULL)
		goto out;

	if ((str = (char *)ASN1_STRING_get0_data(asn1str)) == NULL)
		goto out;

	len = ASN1_STRING_length(asn1str);
	if (Curl_cert_hostcheck(str, len, host, strlen(host)))
		ret = 1;
out:
	return ret;
}

int
check_san(char *host, X509 *cert)
{
	int ret = 0;
	size_t hostlen;
	X509_EXTENSION *san;
	GENERAL_NAMES *names = NULL;
	const char *str;
	int index;
	int len;

	hostlen = strlen(host);

	index = X509_get_ext_by_NID(cert, NID_subject_alt_name, -1);
	if ((san = X509_get_ext(cert, index)) == NULL)
		goto out;

	if ((names = X509V3_EXT_d2i(san)) == NULL)
		goto out;

	for (index = 0; index < sk_GENERAL_NAME_num(names); index++) {
		const GENERAL_NAME *name;

		name = sk_GENERAL_NAME_value(names, index);
		if (name->type != GEN_DNS)
			continue;

		str = (char *)ASN1_STRING_get0_data(name->d.dNSName);
		if (str == NULL)
			continue;

		len = ASN1_STRING_length(name->d.dNSName);
		if (Curl_cert_hostcheck(str, len, host, hostlen)) {
			ret = 1;
			goto out;
		}
			
	}
out:
	if (names)
		GENERAL_NAMES_free(names);

	return ret;
}

int
tls_handshake(char *host, int port, struct conf *conf, char *msg, size_t msglen)
{
	char hostname[MAXHOSTNAMELEN + 7]; /* 7 for :12345\0 */
	const SSL_METHOD* method = TLS_client_method();
	SSL_CTX* ctx = NULL;
	BIO *bio = NULL;
	SSL *ssl = NULL;
	X509 *cert = NULL;
	struct ocsp_stapling_arg osa;
	char ocsp_msg[256];
	const ASN1_TIME *notAfter;
	int days;
	int secs;
	int ret = NAGIOS_UNKNOWN;

	if ((ctx = SSL_CTX_new(method)) == NULL)
		goto out;

	if ((bio = BIO_new_ssl_connect(ctx)) == NULL)
		goto out;

	(void)snprintf(hostname, sizeof(hostname), "%s:%d", host, port);
	if (BIO_set_conn_hostname(bio, hostname) != 1)
		goto out;
	
	if (BIO_get_ssl(bio, &ssl) != 1 || ssl == NULL)
		goto out;

	if (SSL_set_tlsext_host_name(ssl, host) != 1)
		goto out;

	if (conf->req_ocsp_stapling) {
		osa.status = NAGIOS_UNKNOWN;
		osa.msg = ocsp_msg;
		osa.msglen = sizeof(ocsp_msg);

		SSL_set_tlsext_status_type(ssl, TLSEXT_STATUSTYPE_ocsp);
		SSL_CTX_set_tlsext_status_cb(ctx, ocsp_stapling_cb);
		SSL_CTX_set_tlsext_status_arg(ctx, &osa);
	}

	if (BIO_do_connect(bio) != 1)
		goto out;

	if (BIO_do_handshake(bio) != 1)
		goto out;

	if ((cert = SSL_get_peer_certificate(ssl)) == NULL)
		goto out;
	
	if (check_subject(host, cert) != 1 && 
	    check_san(host, cert) != 1) {
		ret = NAGIOS_CRIT;
		snprintf(msg, msglen,
			 "%s does not match subject nor subjectAltName",
			 host);
		goto out;
	}

	if ((notAfter = X509_get0_notAfter(cert)) == NULL)
		goto out;

	if (ASN1_TIME_diff(&days, &secs, NULL, notAfter) != 1)
		goto out;

	BIO_closesocket(SSL_get_fd(ssl));

	snprintf(msg, msglen, "%s expires in %d days", host, days);

	ret = NAGIOS_OK;
	if (days < conf->warn)
		ret = NAGIOS_WARN;
	if (days < conf->crit)
		ret = NAGIOS_CRIT;

out:
	if (ret == NAGIOS_UNKNOWN) {
		snprintf(msg, msglen, "%s %s",
			 host, ERR_error_string(ERR_get_error(), NULL));
	}

	if (conf->req_ocsp_stapling) {
		snprintf(msg, msglen, "%s (OCSP %s)", msg, osa.msg);
		if (osa.status > ret)
			ret = osa.status;
	}

	if (conf->verbose)
		printf("%s\n", msg);

	if (ssl)
		SSL_free(ssl); /* also frees bio? */
	if (ctx)
		SSL_CTX_free(ctx);
	if (cert)
		X509_free(cert);

	return ret;
}

void
drop_privs(struct conf *conf)
{
	struct passwd *pw;

	if ((pw = getpwnam(conf->user)) == NULL)
		err(NAGIOS_UNKNOWN, "inexistant user %s", conf->user);

	if (setgroups(1, &(pw->pw_gid)))
		err(NAGIOS_UNKNOWN, "setgroups(1, %d) failed", pw->pw_gid);

	if (setgid(pw->pw_gid))
		err(NAGIOS_UNKNOWN, "setgid(%d) failed", pw->pw_gid);

	if (setuid(pw->pw_uid))
		err(NAGIOS_UNKNOWN, "setuid(%d) failed", pw->pw_uid);
	
	return;
}

char *
append_str(char *base, char *append)
{
	size_t base_len;
	size_t append_len;
	size_t new_len;
	char *new;

	if (append == NULL)
		return base;

	base_len = base ? strlen(base) : 0;
	append_len = strlen(append);

	if (base_len + append_len > MSG_MAXLEN) {
		char dots[] = " (...)";

		if (base == NULL ||
		    strcmp(base + base_len - sizeof(dots) + 1, dots) != 0) {
			new_len = base_len + sizeof(dots);
			new = realloc(base, new_len);
			sprintf(new + base_len, dots);
		} else {
			new = base;
		}
			
		return new;
	}

	new_len = base_len + (base_len ? 2 : 0) + append_len + 1;
	new = realloc(base, new_len);
	sprintf(new + base_len, "%s%s", base_len ? ", " : "", append);

	return new;
}

int
main(int argc, char **argv)
{
	struct conf conf = {
		DEFAULT_WARN,
		DEFAULT_CRIT,
		0,		/* verbose */
		0,		/* OCSP stapling */
		"nagios",	/* user */
		DEFAULT_CMD,
	};
	char *progname = argv[0];
	int ch;
	FILE *f;
	char *line;
	char linebuf[512];
	int status = NAGIOS_OK;
	char *ok_str = NULL;
	char *warn_str = NULL;
	char *crit_str = NULL;
	char *unknown_str = NULL;

	while ((ch = getopt(argc, argv, "c:e:su:vw:")) != -1) {
        	switch (ch) {
		case 'c':
			conf.crit = atoi(optarg);
			break;
		case 'e':
			conf.cmd = optarg;
			break;
		case 's':
			conf.req_ocsp_stapling = 1;
			break;
		case 'u':
			conf.user = optarg;
			break;
		case 'v':
			conf.verbose = 1;
			break;
		case 'w':
			conf.warn = atoi(optarg);
			break;
		default:
			errx(NAGIOS_UNKNOWN,
			     "usage: %s [-u user][-v][-s][-w warn][-c crit]",
			     progname);
			break;
		}
	}
	argc -= optind;
	argv += optind;

	(void)setuid(0);

	if ((f = popen(conf.cmd, "r")) == NULL)
		err(NAGIOS_UNKNOWN, "popen(%s) failed", conf.cmd);

	if (conf.user)
		drop_privs(&conf);

	SSL_library_init();

	while ((line = fgets(linebuf, sizeof(linebuf), f)) != NULL) {
		char msg[256];
		char search_host[] = ":443";
		char search_vhost[] = "port 443 namevhost";
		char *cp;

		if ((cp = strstr(line, search_vhost)) != NULL) {
			cp += sizeof(search_vhost);
		} else {
			if ((cp = strstr(line, search_host)) != NULL &&
			    strstr(cp, "is a NameVirtualHost") == NULL)
				cp += sizeof(search_host);
			else
				continue;
		}

		switch(tls_handshake(strtok(cp, " "), 443, 
				     &conf, msg, sizeof(msg))) {
		case NAGIOS_OK:
			ok_str = append_str(ok_str, msg);
			break;
		case NAGIOS_WARN:
			if (status != NAGIOS_CRIT)
				status = NAGIOS_WARN;
			warn_str = append_str(warn_str, msg);
			break;
		case NAGIOS_CRIT:
			status = NAGIOS_CRIT;
			crit_str = append_str(crit_str, msg);
			break;
		default:
			if (status == NAGIOS_OK)
				status = NAGIOS_UNKNOWN;
			unknown_str = append_str(unknown_str, msg);
			break;
		}
	}

	if (status == NAGIOS_OK && (ok_str == NULL || ok_str[0] == '\0')) {
		status = NAGIOS_UNKNOWN;
		unknown_str = "no data";
	}


	if (ferror(f))
		err(NAGIOS_UNKNOWN, "popen(/usr/pkg/sbin/httpd -S) failed");

	fclose(f);

	switch (status) {
	case NAGIOS_OK:
		printf("OK %s\n", ok_str);
		break;
	case NAGIOS_WARN:
		printf("WARNING %s\n", warn_str);
		break;
	case NAGIOS_CRIT:
		printf("CRITICAL %s\n", crit_str);
		break;
	default:
		printf("UNKNOWN %s\n", unknown_str);
		break;
	}

	return status;
}
