/* ccgo: view_game.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "view_game.hh"
#include "settings.hh"
#include <sstream>

using namespace go;

void ViewGame::on_realize()
{
	Gtk::DrawingArea::on_realize();
	win = get_window();
	style = get_style();
	if (use_bg_pixmap) win->set_back_pixmap(board_pixmap, false); // might not need to do it here
}

void ViewGame::on_unrealize()
{
	Gtk::DrawingArea::on_unrealize();
}

bool ViewGame::on_configure_event(GdkEventConfigure * e)
{
	win_w = e->width;
	win_h = e->height;
	do_width_height();
	return true;
}

bool ViewGame::on_expose_event(GdkEventExpose * e)
{
	// std::cerr << "ViewGame exposed" << std::endl;
	if (e->count > 0) return true;
	int z = get_board_size();
	if (coordinate_on) {
		for (int x = 0; x < z; ++ x) {
			std::string s = "";
			s += (x < 8 ? char('A' + x) : char ('B' + x));
			Glib::RefPtr<Pango::Layout> tx = create_pango_layout(s);
			int w;
			int h;
			tx->get_pixel_size(w, h);
			win->draw_layout(style->get_fg_gc(get_state()), pad_w + cell_w * x - w / 2, 0, tx);
			win->draw_layout(style->get_fg_gc(get_state()), pad_w + cell_w * x - w / 2, win_h - h, tx);
		}
		for (int y = 0; y < z; ++ y) {
			std::ostringstream s;
			s << y + 1;
			Glib::RefPtr<Pango::Layout> tx = create_pango_layout(s.str());
			int w;
			int h;
			tx->get_pixel_size(w, h);
			win->draw_layout(style->get_fg_gc(get_state()), (pad_w - cell_w / 2 - w) / 2, pad_h + cell_h * (z - 1 - y) - h / 2, tx);
			win->draw_layout(style->get_fg_gc(get_state()), win_w - (pad_w - cell_w / 2 + w) / 2, pad_h + cell_h * (z - 1 - y) - h / 2, tx);
		}
	}
	for (int x = 0; x < z; ++ x) for (int y = 0; y < z; ++ y) draw_loc(Loc(x, y));
	return true;
}

void ViewGame::on_style_changed(const Glib::RefPtr<Gtk::Style> & s)
{
	Widget::on_style_changed(s);
	if (win && use_bg_pixmap) win->set_back_pixmap(board_pixmap, false);
// 	std::cerr << "style changed" << std::endl;
}

void ViewGame::do_width_height()
{
	int z = get_board_size();
	if (z) {
		if (coordinate_on) {
			int w;
			int h;
			create_pango_layout("A")->get_pixel_size(w, h);
			cell_h = (win_h - 2 * h) / z;
			create_pango_layout("99")->get_pixel_size(w, h);
			cell_w = (win_w - 2 * w) / z;
		} else {
			cell_w = (win_w - 2) / z;
			cell_h = (win_h - 2) / z;
		}
		pad_w = (win_w - cell_w * (z - 1)) / 2;
		pad_h = (win_h - cell_h * ( z - 1)) / 2;
		if (stone_image_ok && cell_w > 0 && cell_h > 0) {
			scaled_w_stone = w_stone_image->scale_simple(cell_w, cell_h, Gdk::INTERP_HYPER);
			scaled_b_stone = b_stone_image->scale_simple(cell_w, cell_h, Gdk::INTERP_HYPER);
			scaled_ok = true;
		} else scaled_ok = false;
	} else {
		cell_w = 0;
		cell_h = 0;
	}
}

void ViewGame::draw_loc_at(const Loc & l, int cx, int cy)
{
	int z = get_board_size();
	int x = l.get_x();
	int y = l.get_y();
// 	win->draw_rectangle(style->get_bg_gc(get_state()), true, cx - cell_w / 2, cy - cell_h / 2, cell_w, cell_h);
	win->clear_area(cx - cell_w / 2, cy - cell_h / 2, cell_w, cell_h);
	// grids on the board
	win->draw_line(style->get_fg_gc(get_state()), (x == 0 ? cx : cx - cell_w / 2), cy, (x == z - 1 ? cx : cx + cell_w - cell_w / 2 - 1), cy);
	win->draw_line(style->get_fg_gc(get_state()), cx, (y == z - 1 ? cy : cy - cell_h / 2), cx, (y == 0 ? cy : cy + cell_h - cell_h / 2 - 1));
	if ((z == 19 && (x == 3 || x == 9 || x == 15) && (y == 3 || y == 9 || y == 15)) || (z == 13 && (x == 3 || x == 6 || x == 9) && (y == 3 || y == 6 || y == 9)) || (z == 9 && ((x == 4 && y == 4) || ((x == 2 || x == 6) && (y == 2 || y == 6))))) win->draw_arc(style->get_fg_gc(get_state()), true, cx - cell_w / 6, cy - cell_h / 6, cell_w / 3 - 1, cell_h / 3 - 1, 0, 64 * 360); // star positions
	switch (get_site(l)) {
	case STATE_WHITE:
		if (scaled_ok) {
			scaled_w_stone->render_to_drawable_alpha(win, 0, 0, cx - cell_w / 2, cy - cell_h / 2, cell_w, cell_h, Gdk::PIXBUF_ALPHA_FULL, 0, Gdk::RGB_DITHER_NORMAL, 0, 0);
		} else {
			win->draw_arc(style->get_white_gc(), true, cx - cell_w / 2, cy - cell_h / 2, cell_w - 1, cell_h - 1, 0, 64 * 360);
			win->draw_arc(style->get_black_gc(), false, cx - cell_w / 2, cy - cell_h / 2, cell_w - 1, cell_h - 1, 0, 64 * 360);
		}
		break;
	case STATE_BLACK:
		if (scaled_ok) {
			scaled_b_stone->render_to_drawable_alpha(win, 0, 0, cx - cell_w / 2, cy - cell_h / 2, cell_w, cell_h, Gdk::PIXBUF_ALPHA_FULL, 0, Gdk::RGB_DITHER_NORMAL, 0, 0);
		} else {
			win->draw_arc(style->get_black_gc(), true, cx - cell_w / 2, cy - cell_h / 2, cell_w - 1, cell_h - 1, 0, 64 * 360);
			win->draw_arc(style->get_white_gc(), false, cx - cell_w / 2, cy - cell_h / 2, cell_w - 1, cell_h - 1, 0, 64 * 360);
		}
		break;
	case STATE_EMPTY:
		break;
	}
	if (l == get_ko()) {
		win->draw_line(style->get_black_gc(), cx + cell_w / 4, cy - cell_h / 4, cx - cell_w / 4, cy + cell_h / 4);
		win->draw_line(style->get_black_gc(), cx - cell_w / 4, cy - cell_h / 4, cx + cell_w / 4, cy + cell_h / 4);
	}
	if (l == get_last_move()) {
		switch (get_site(l)) {
		case STATE_WHITE:
			win->draw_arc(style->get_black_gc(), false, cx - cell_w / 4, cy - cell_h / 4, cell_w / 2 - 1, cell_h / 2 - 1, 0, 64 * 360);
			break;
		case STATE_BLACK:
			win->draw_arc(style->get_white_gc(), false, cx - cell_w / 4, cy - cell_h / 4, cell_w / 2 - 1, cell_h / 2 - 1, 0, 64 * 360);
			break;
		case STATE_EMPTY:
			win->draw_arc(style->get_fg_gc(get_state()), false, cx - cell_w / 4, cy - cell_h / 4, cell_w / 2 - 1, cell_h / 2 - 1, 0, 64 * 360);
			break;
		}
	}
}

void ViewGame::draw_loc(const Loc & l)
{
	if (! is_realized() || l == Loc()) return;
	if (cell_w == 0 || cell_h == 0) return;
	int cx;
	int cy;
	trans_loc(l, cx, cy);
	draw_loc_at(l, cx, cy);
}

void ViewGame::trans_loc(const Loc & l, int & cx, int & cy)
{
	int z = get_board_size();
// 	cx = (2 * l.get_x() + 1) * win_w / z / 2;
// 	cy = (2 * (z - 1 - l.get_y()) + 1) * win_h / z / 2;
	cx = pad_w + cell_w * l.get_x();
	cy = pad_h + cell_h * (z - 1 - l.get_y());
}

Loc ViewGame::find_loc(int xx, int yy)
{
	if (cell_w == 0 || cell_h == 0) return Loc();
	int z = get_board_size();
	int x = (xx - pad_w + cell_w / 2) / cell_w;
	int y = z - 1 - (yy - pad_h + cell_h / 2) / cell_h;
	Loc l(x, y);
	if (! l.in_range(z)) l = Loc();
	return l;
}

ViewGame::ViewGame() :
	Game()
{
	std::string ip = go::settings.get_image_path();
	if (go::settings.get_option(go::Settings::OPT_PICTURE_STONE) && Glib::file_test(ip + "/white_stone.png", Glib::FILE_TEST_EXISTS) && Glib::file_test(ip + "/black_stone.png", Glib::FILE_TEST_EXISTS)) {
		w_stone_image = Gdk::Pixbuf::create_from_file(ip + "/white_stone.png");
		b_stone_image = Gdk::Pixbuf::create_from_file(ip + "/black_stone.png");
		stone_image_ok = true;
	} else stone_image_ok = false;
	scaled_ok = false;
	coordinate_on = go::settings.get_option(go::Settings::OPT_SHOW_COORDINATE);
	if (go::settings.get_option(go::Settings::OPT_PICTURE_BOARD) && Glib::file_test(ip + "/board.png", Glib::FILE_TEST_EXISTS)) {
		Glib::RefPtr<Gdk::Bitmap> bm;
		Gdk::Pixbuf::create_from_file(ip + "/board.png")->render_pixmap_and_mask(board_pixmap, bm, 0);
		use_bg_pixmap = true;
	} else use_bg_pixmap = false;
}

ViewGame::~ViewGame()
{
	// std::cerr << "deleting ViewGame" << std::endl;
}

void ViewGame::set_board_size(unsigned z)
{
	Game::set_board_size(z);
	if (is_realized()) do_width_height();
	else {
		cell_w = 0;
		cell_h = 0;
	}
}

void ViewGame::set_ko(const Loc & l)
{
	Loc old_ko = get_ko();
	Game::set_ko(l);
	draw_loc(get_ko());
	draw_loc(old_ko);
}

void ViewGame::set_last_move(const Loc & l)
{
	Loc old_last_move = get_last_move();
	Game::set_last_move(l);
	draw_loc(get_last_move());
	draw_loc(old_last_move);
}

void ViewGame::set_site(const Loc & l, State s)
{
	Game::set_site(l, s);
	draw_loc(l);
}
