/* ccgo: ui/editor.cc
 * 
 * Copyright (C) 2006 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
#include <ui/editor.hh>
#include <go/game.hh>
#include <debug.hh>
#include <iostream>
#include <sstream>
using namespace ui;
using namespace go;
using namespace std;

bool Editor::is_movable() const
{
	return mode != SETUP_MODE && mode != SCORE_MODE;
}

void Editor::new_size()
{
	if (! face) return;
	if (where()) face->resize(get_size()); // get the size ready
	// other information will be taken care of by 'new_state' and 'moved_from'
	else {
		assert(get_size() == 0); // not attached, face should act independently
		face->clear(); // however, clear the board
	}
}

void Editor::new_state(Loc loc)
{
	if (face) face->set_state(loc, state(loc));
}

void Editor::moved_from(Node * old)
{
	PathWalk::moved_from(old);
	bscore = 0;
	wscore = 0;
	if (where()) {
		mode = NORMAL_MODE;
		// calculate scores
		const std::map<Loc, Mark> & m = where()->get_marking();
		for (std::map<Loc, Mark>::const_iterator i = m.begin(); i != m.end(); i ++) switch (i->second) {
		case MARK_B_TERRITORY:
			bscore ++; // black territory
			if (state(i->first) == White) bscore ++; // zombie white
			break;
		case MARK_W_TERRITORY:
			wscore ++; // white territory
			if (state(i->first) == Black) wscore ++; // zombie black
			break;
		default:
			;
		}
		attached = true;
	}
	else attached = false;

	if (face && old) {
		face->clear_marks();
		face->clear_labels();
		markup_face();
		if (attached) face->move();
		else face->edit();
	}

	if (map) map->update();

	new_node();
}

void Editor::new_path()
{
	if (map) map->update();
}

void Editor::face_play(Loc loc)
{
	assert(is_movable() && attached);
	Node * n = add_move(Move(loc));
	n->link();
	move_to(n);
}

void Editor::face_pick(Loc loc)
{
	assert(mode == SCORE_MODE);
	vector<Loc> g;
	dead_group(loc, g);
	State t;
	if (terri[loc] == Empty) t = flip(state(loc)); // marking dead
	else t = Empty; // revive
	for (vector<Loc>::iterator i = g.begin(); i != g.end(); i ++) terri[* i] = t;
	scoring_update();
}

void Editor::face_knock(Loc loc)
{
	// change properties at loc
	assert(face && attached);
	Mark mk = face->get_mark(loc);
	string lb = face->get_label(loc);
	Mark omk = mk;
	change_site(loc, mk, lb);
	if (mk != omk) { // update scoring
		if (omk == MARK_B_TERRITORY) {
			bscore --;
			if (state(loc) == White) bscore --;
		}
		else if (omk == MARK_W_TERRITORY) {
			wscore --;
			if (state(loc) == Black) wscore --;
		}
		if (mk == MARK_B_TERRITORY) {
			bscore ++;
			if (state(loc) == White) bscore ++;
		}
		else if (mk == MARK_W_TERRITORY) {
			wscore ++;
			if (state(loc) == Black) wscore ++;
		}
	}
	where()->set_mark(loc, mk);
	where()->set_label(loc, lb);
	face->set_mark(loc, mk);
	face->set_label(loc, lb);
}

void Editor::face_track(Loc loc)
{
	view_site(loc);
}

void Editor::scoring_update()
{
	assert(terri);
	score_terri(terri);
	// scores follow face in score mode
	bscore = 0;
	wscore = 0;
	for (Loc l = 0; l < range(); l ++) {
		if (terri[l]) {
			if (face) face->set_mark(l, terri[l] == Black ? MARK_B_TERRITORY : MARK_W_TERRITORY);
			if (terri[l] == Black) {
				bscore ++;
				if (state(l) == White) bscore ++;
			}
			else {
				wscore ++;
				if (state(l) == Black) wscore ++;
			}
		}
		else if (face) face->set_mark(l, MARK_NONE);
	}
}

void Editor::markup_face()
{
	assert(face);
	if (where()) {
		// update markings
		const std::map<Loc, Mark> & m = where()->get_marking();
		for (std::map<Loc, Mark>::const_iterator i = m.begin(); i != m.end(); i ++) {
			face->set_mark(i->first, i->second);
		}
		// update label
		const std::map<Loc, string> & l = where()->get_labels();
		for (std::map<Loc, string>::const_iterator i = l.begin(); i != l.end(); i ++) {
			face->set_label(i->first, i->second);
		}
		face->set_turn(where()->get_turn());
		face->set_move(where()->get_move());
		face->set_ko(where()->get_ko());
	}
	else {
		face->set_turn(Black);
		face->set_move(MoveNone);
		face->set_ko(set<Loc>());
		face->clear();
	}
}

void Editor::view_site(Loc loc)
{
}

void Editor::new_node()
{
}

void Editor::change_site(Loc loc, Mark & mk, string & lb)
{
}

Editor::Editor() :
	face(0),
	map(0),
	mode(NORMAL_MODE),
	attached(false),
	terri(0)
{
}

Editor::~Editor()
{
	if (terri) delete [] terri;
}

void Editor::set_face(Face * f)
{
	if (! f) {
		if (face) face->set_editor(0);
		face = 0;
		return;
	}
	if (face) { // old face exists? copy things over
		f->copy_face(* face);
		face->set_editor(0);
		face = f;
	}
	else { // markup the face...
		face = f;
		face->set_editor(this);
		face->clear_marks();
		face->clear_labels();
		markup_face();
	}
	switch (mode) {
	case SETUP_MODE:
		assert(attached);
		face->edit();
		break;
	case SCORE_MODE:
		assert(attached);
		face->pick();
		scoring_update();
		break;
	default:
		if (attached) face->move();
		else face->edit();
	}
}

void Editor::set_map(Map * m)
{
	map = m;
	m->set_editor(this);
}

void Editor::set_node(Node * node)
{
	attach(node);
}

void Editor::attach(Node * node)
{
	assert(is_movable());
	move_to(node);
}

void Editor::down()
{
	assert(can_crawl());
	Node * n = get_path();
	if (n) move_to(n);
}

void Editor::up()
{
	assert(can_crawl());
	Node * n = where()->up();
	if (n) move_to(n);
}

void Editor::hop()
{
	assert(can_crawl());
	Node * n = where()->up();
	if (n) {
		unsigned r = where()->get_order() + 1;
		if (r >= n->down().size()) r = 0;
		n = n->down()[r];
		move_to(n);
	}
}

void Editor::hop(unsigned alt)
{
	assert(can_crawl());
	Node * n = where()->up();
	if (n) {
		if (alt >= n->down().size()) return;
		n = n->down()[alt];
		move_to(n);
	}
}

void Editor::setup()
{
	assert(is_movable() && attached);
	mode = SETUP_MODE;
	face->edit();
	// clear markings
	face->set_move(MoveNone);
	face->set_ko(set<Loc>());
	face->clear_marks();
	face->clear_labels();
}

void Editor::score()
{
	assert(is_movable() && attached);
	mode = SCORE_MODE;
	// initialize terri
	if (terri) delete [] terri;
	terri = new State [range()];
	for (Loc l = 0; l < range(); l ++) terri[l] = Empty;
	assert(where());
	for (std::map<Loc, Mark>::const_iterator i = where()->get_marking().begin(); i != where()->get_marking().end(); i ++) switch (i->second) {
	case MARK_W_TERRITORY:
		terri[i->first] = White;
		break;
	case MARK_B_TERRITORY:
		terri[i->first] = Black;
		break;
	default:
		;
	}
	face->pick();
	scoring_update();
}

void Editor::done_setup()
{
	assert(mode == SETUP_MODE);
	Turn turn = face->get_turn();
	unsigned capb = where()->get_capb();
	unsigned capw = where()->get_capw();
	set<Loc> ko = where()->get_ko();
	Node * n = add_setup(* face, turn, capb, capw, ko);
	n->link();
	face->copy(* this); // restore face
	move_to(n);
}

void Editor::done_score()
{
	assert(mode == SCORE_MODE);
	// change scoring marks
	std::map<go::Loc, go::Mark> m = where()->get_marking();
	for (Loc l = 0; l < range(); l ++) switch (terri[l]) {
	case Black:
		m[l] = go::MARK_B_TERRITORY;
		break;
	case White:
		m[l] = go::MARK_W_TERRITORY;
		break;
	default:
		if (m.find(l) != m.end() && (m[l] == go::MARK_B_TERRITORY || m[l] == go::MARK_W_TERRITORY)) m.erase(l);
	}
	where()->set_marking(m);
	// some markings could be erased on face, restore them
	for (std::map<Loc, Mark>::const_iterator i = m.begin(); i != m.end(); i ++) {
		face->set_mark(i->first, i->second);
	}
	mode = NORMAL_MODE;
	face->move();
}

void Editor::cancel_setup()
{
	assert(mode == SETUP_MODE);
	mode = NORMAL_MODE;
	face->copy(* this);
	markup_face(); // re-markup the face
	face->move();
}

void Editor::cancel_score()
{
	assert(mode == SCORE_MODE);
	mode = NORMAL_MODE;
	// restore marks
	face->clear_marks();
	const std::map<Loc, Mark> & m = where()->get_marking();
	bscore = 0;
	wscore = 0;
	for (std::map<Loc, Mark>::const_iterator i = m.begin(); i != m.end(); i ++) {
		face->set_mark(i->first, i->second);
		switch (i->second) {
		case MARK_B_TERRITORY:
			bscore ++; // black territory
			if (state(i->first) == White) bscore ++; // zombie white
			break;
		case MARK_W_TERRITORY:
			wscore ++; // white territory
			if (state(i->first) == Black) wscore ++; // zombie black
			break;
		default:
			;
		}
	}
	face->move();
}

bool Editor::can_init()
{
	return face && ! attached; // face is required for initialization
}

void Editor::init_size(unsigned sz)
{
	assert(can_init());
	face->resize(sz);
}

void Editor::init_handi(unsigned h)
{
	assert(can_init());
	face->clear();
	vector<Loc> hs = face->handi_sites(h);
	if (hs.size()) {
		for (vector<Loc>::iterator i = hs.begin(); i != hs.end(); i ++) face->set_state(* i, Black);
		face->set_turn(White);
	}
	else face->set_turn(Black); // no handicap
}

void Editor::init_board(const Board & board)
{
	assert(can_init());
	face->copy(board);
}

void Editor::init_turn(Turn t)
{
	assert(can_init());
	face->set_turn(t);
}

void Editor::init_create()
{
	assert(can_init());
	Game * g = new Game(* face, face->get_move(), face->get_turn(), 0, 0, face->get_ko());
	face->copy(* this); // clear board
	move_to(g);
}

void Editor::setup_turn(Turn t)
{
	assert(mode == SETUP_MODE);
	face->set_turn(t);
}

bool Editor::can_play() const
{
	return is_movable() && attached;
}

bool Editor::can_crawl() const
{
	return is_movable() && attached;
}

bool Editor::doing_setup() const
{
	return mode == SETUP_MODE;
}

bool Editor::is_scoring() const
{
	return mode == SCORE_MODE;
}

void Editor::pass()
{
	assert(can_play());
	Node * n = add_move(MovePass);
	n->link();
	move_to(n);
}

bool Editor::is_attached()
{
	return attached;
}

unsigned Editor::get_move_num()
{
	assert(where());
	return where()->get_move_num();
}

Turn Editor::get_turn()
{
	assert(where());
	return where()->get_turn();
}

unsigned Editor::get_level()
{
	assert(where());
	return where()->get_level();
}

const string & Editor::get_text()
{
	assert(where());
	return where()->get_text();
}

void Editor::set_text(const string & text)
{
	assert(where());
	where()->set_text(text);
}

unsigned Editor::get_alt_num()
{
	assert(where());
	if (where()->up()) return where()->up()->down().size();
	return 1;
}

unsigned Editor::get_alt()
{
	assert(where());
	if (! where()->up()) return 0;
	return where()->get_order();
}

double Editor::get_b_score()
{
	assert(attached);
	double sc = where()->get_game()->get_komi();
	if (sc < 0) sc = 0;
	sc += bscore;
	sc += where()->get_capb();
	return sc;
}

double Editor::get_w_score()
{
	assert(attached);
	double sc = - where()->get_game()->get_komi(); // reverse komi
	if (sc < 0) sc = 0;
	sc += wscore;
	sc += where()->get_capw();
	return sc;
}

bool Editor::can_del_node()
{
	assert(where());
	return where()->can_unlink();
}

void Editor::del_node()
{
	delete_node();
}
