/* ccgo: time.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "time.hh"
#include <sstream>
#include <glibmm.h>
extern "C" {
#include <sys/time.h>
#include <time.h>
}

using namespace go;

void Time::mark_time()
{
	timeval tv;
	gettimeofday(& tv, 0);
	mark_sec = tv.tv_sec;
	mark_usec = tv.tv_usec;
}

long Time::get_time()
{
	timeval tv;
	gettimeofday(& tv, 0);
	if (tv.tv_usec < mark_usec) tv.tv_sec --;
	return tv.tv_sec - mark_sec;
}

bool Time::tick()
{
	if (! run) return true;

	if (get_time() <= tick_time) return true;
	tick_time ++;

	counter ++;
	if (turn == TURN_BLACK) {
		if (b_time > 0) {
			b_time --;
			if (b_time == 0) {
				if (b_move < 0) {
					b_time = byo_time;
					b_move = byo_move;
				} else {
					timeup(turn);
				}
			}
			update_timing();
		}
	} else {
		if (w_time > 0) {
			w_time --;
			if (w_time == 0) {
				if (w_move < 0) {
					w_time = byo_time;
					w_move = byo_move;
				} else {
					timeup(turn);
				}
			}
			update_timing();
		}
	}
	return true;
}

void Time::update_timing()
{
	std::ostringstream s;
	s << "W:" << w_time / 60 << ":" << (w_time % 60) / 10 << w_time % 10 << "/";
	if (w_move < 10) s << ' ';
	if (w_move >= 0) s << w_move;
	else s << "-";
	w_timing.set_text(s.str());
	s.str("");
	s << "B:" << b_time / 60 << ":" << (b_time % 60) / 10 << b_time % 10 << "/";
	if (b_move < 10) s << ' ';
	if (b_move >= 0) s << b_move;
	else s << "-";
	b_timing.set_text(s.str());
}

void Time::adjust_sizes()
{
	int w;
	int h;
	w_timing.create_pango_layout("W:44:44/44")->get_pixel_size(w, h);
	w_timing.set_size_request(w, - 1);
	b_timing.create_pango_layout("W:44:44/44")->get_pixel_size(w, h);
	b_timing.set_size_request(w, - 1);
}

void Time::on_realize()
{
	Gtk::VBox::on_realize();
	adjust_sizes();
}

void Time::on_style_changed(const Glib::RefPtr<Gtk::Style> & ps)
{
	Gtk::VBox::on_style_changed(ps);
	adjust_sizes();
}

Time::Time()
{
	run = false;
	turn = TURN_BLACK;
	pack_start(w_timing, Gtk::PACK_SHRINK);
	pack_start(b_timing, Gtk::PACK_SHRINK);
	time_con = Glib::signal_timeout().connect(mem_fun(* this, & Time::tick), 100);
}

Time::~Time()
{
	time_con.disconnect();
}

void Time::setup(int t, int yt, int ym)
{
	time = t;
	byo_time = yt;
	byo_move = ym;
	w_time = time;
	b_time = time;
	w_move = -1;
	b_move = -1;
	run = false;
	update_timing();
	set_sensitive(false);
}

void Time::calibrate(int wt, int wm, int bt, int bm)
{
	w_time = wt;
	if (w_time < 0) w_time = 0;
	w_move = wm;
	b_time = bt;
	if (b_time < 0) b_time = 0;
	b_move = bm;
	mark_time();
	tick_time = 0;
	update_timing();
}

void Time::start()
{
	run = true;
	set_sensitive(true);
	mark_time();
	tick_time = 0;
	counter = 0;
}

void Time::stop()
{
	run = false;
	set_sensitive(false);
}

void Time::toggle()
{
	if (turn == TURN_BLACK) {
		if (b_move > 0) b_move --;
		turn = TURN_WHITE;
		if (w_move == 0) {
			w_time = byo_time;
			w_move = byo_move;
		}
	} else {
		if (w_move > 0) w_move --;
		turn = TURN_BLACK;
		if (b_move == 0) {
			b_time = byo_time;
			b_move = byo_move;
		}
	}
	counter = 0;
}

unsigned Time::get_counter()
{
	return counter;
}
