/* ccgo: proper_game.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "config.hh"
#include "gettext.h"
#include "proper_game.hh"
#include "modifi_proper.hh"
#include "root_proper.hh"
#include "score_proper.hh"
#include "setup.hh"
#include "move.hh"
#include <sstream>
#include <cstdio>
#define _(String) Glib::locale_to_utf8(gettext(String))

using namespace go;

bool ProperGame::on_enter_notify_event(GdkEventCrossing * e)
{
	int z = get_board_size();
	if (! z) return true;
// 	int x = int(e->x * z / win_w);
// 	int y = z - int(e->y * z / win_h) - 1;
	Loc l = find_loc(int(e->x), int(e->y));
	if (track == l) return true;
	track = l;
	if (track && label != NULL && label[track.expand(z)] != NULL) label_port.set_text(_("label: ") + * label[track.expand(z)]);
	else label_port.set_text(track.to_string());
	on_track_change();
	return true;
}

bool ProperGame::on_leave_notify_event(GdkEventCrossing * e)
{
	if (! get_board_size()) return true;
	track = Loc();
	label_port.set_text("--");
	on_track_change();
	return true;
}

bool ProperGame::on_motion_notify_event(GdkEventMotion * e)
{
	int z = get_board_size();
	if (! z) return true;
	Loc l = find_loc(int(e->x), int(e->y));
	if (track != l) {
		track = l;
		if (label && track && label[track.expand(z)]) label_port.set_text(_("label: ") + * label[track.expand(z)]);
		else label_port.set_text(track.to_string());
		on_track_change();
	}
	return true;
}

void ProperGame::update_score()
{
	std::ostringstream s;
	s << get_komi() + get_cap_w() + score_modify_w;
	w_score_port.set_text(s.str());
	s.str("");
	s << get_cap_b() + score_modify_b;
	b_score_port.set_text(s.str());
}

void ProperGame::draw_loc_at(const Loc & l, int cx, int cy)
{
	ViewGame::draw_loc_at(l, cx, cy);
	if (label == NULL) return;
	int z = get_board_size();
	if (label[l.expand(z)] != NULL) {
		Glib::RefPtr<Pango::Layout> t = create_pango_layout(* label[l.expand(z)]);
		int w;
		int h;
		t->get_pixel_size(w, h);
		if (w > cell_w || h > cell_h) {
			t->set_text("#");
			t->get_pixel_size(w, h);
		}
		switch (get_site(l)) {
		case STATE_WHITE:
			win->draw_layout(style->get_black_gc(), cx - w / 2, cy - h / 2, t);
			break;
		case STATE_BLACK:
			win->draw_layout(style->get_white_gc(), cx - w / 2, cy - h / 2, t);
			break;
		case STATE_EMPTY:
			win->clear_area(cx - cell_w / 2, cy - cell_h / 2, cell_w, cell_h);
			win->draw_layout(style->get_fg_gc(get_state()), cx - w / 2, cy - h / 2, t);
		}
	}
	Glib::RefPtr<const Gdk::GC> gc;
	switch (get_site(l)) {
	case STATE_WHITE:
		gc = style->get_black_gc();
		break;
	case STATE_BLACK:
		gc = style->get_white_gc();
		break;
	case STATE_EMPTY:
		gc = style->get_fg_gc(get_state());
		break;
	}
	switch (mark[l.expand(z)]) {
	case MARK_NONE:
		break;
	case MARK_CIRCLE:
		if (get_site(l) == STATE_EMPTY) win->clear_area(cx - cell_w * 3 / 8, cy - cell_h * 3 / 8, cell_w * 3 / 4 - 1, cell_h * 3 / 4 - 1);
		win->draw_arc(gc, false, cx - cell_w * 3 / 8, cy - cell_h * 3 / 8, cell_w * 3 / 4 - 1, cell_h * 3 / 4 - 1, 0, 64 * 360);
		break;
	case MARK_TRIANGLE:
		if (get_site(l) == STATE_EMPTY) win->clear_area(cx - cell_w * 29 / 100, cy - cell_h * 49 / 100, cell_w * 58 / 100, cell_h * 73 / 100 - 1);
		win->draw_line(gc, cx - cell_w * 43 / 100, cy + cell_h * 6 / 25 - 1, cx, cy - cell_h * 49 / 100);
		win->draw_line(gc, cx + cell_w * 43 / 100, cy + cell_h * 6 / 25 - 1, cx, cy - cell_h * 49 / 100);
		win->draw_line(gc, cx - cell_w * 43 / 100, cy + cell_h * 6 / 25 - 1, cx + cell_w * 43 / 100 - 1, cy + cell_h * 6 / 25 - 1);
		break;
	case MARK_SQUARE:
		if (get_site(l) == STATE_EMPTY) win->clear_area(cx - cell_w * 34 / 100, cy - cell_h * 34 / 100, cell_w * 68 / 100 - 1, cell_h * 68 / 100 - 1);
		win->draw_rectangle(gc, false, cx - cell_w * 34 / 100, cy - cell_h * 34 / 100, cell_w * 68 / 100 - 1, cell_h * 68 / 100 - 1);
		break;
	case MARK_CROSS:
		win->draw_line(gc, cx - cell_w * 34 / 100, cy - cell_h * 34 / 100, cx + cell_w * 34 / 100 - 1, cy + cell_h * 34 / 100 - 1);
		win->draw_line(gc, cx + cell_w * 34 / 100 - 1, cy - cell_h * 34 / 100, cx - cell_w * 34 / 100, cy + cell_h * 34 / 100 - 1);
		break;
	case MARK_SELECT:
		win->draw_rectangle(gc, true, cx - cell_w * 2 / 10, cy - cell_h * 2 / 10, cell_w * 4 / 10 - 1, cell_h * 4 / 10 - 1);
		break;
	case MARK_W_TERRITORY:
		win->draw_arc(style->get_white_gc(), true, cx - cell_w / 4, cy - cell_h / 4, cell_w / 2 - 1, cell_h / 2 - 1, 0, 64 * 360);
		win->draw_arc(style->get_black_gc(), false, cx - cell_w / 4, cy - cell_h / 4, cell_w / 2 - 1, cell_h / 2 - 1, 0, 64 * 360);
		break;
	case MARK_B_TERRITORY:
		win->draw_arc(style->get_black_gc(), true, cx - cell_w / 4, cy - cell_h / 4, cell_w / 2 - 1, cell_h / 2 - 1, 0, 64 * 360);
		win->draw_arc(style->get_white_gc(), false, cx - cell_w / 4, cy - cell_h / 4, cell_w / 2 - 1, cell_h / 2 - 1, 0, 64 * 360);
		break;
	}
}

void ProperGame::on_track_change()
{
}

const Loc & ProperGame::get_track() const
{
	return track;
}

ProperGame::ProperGame() :
	b_score_port("0"),
	w_score_port("0"),
	modifi_port(_("start")),
	label_port("--"),
	score_modify_w(0),
	score_modify_b(0)
{
	unsigned z = get_board_size();
	if (z) {
		label = new std::string * [z * z];
		mark = new Mark [z * z];
		for (unsigned i = 0; i < z * z; i ++) {
			label[i] = NULL;
			mark[i] = MARK_NONE;
		}
	} else {
		label = NULL;
		mark = NULL;
	}
	text_port.set_editable(false);
	text_port.set_cursor_visible(false);
	text_port.set_wrap_mode(Gtk::WRAP_WORD);
	add_events(Gdk::POINTER_MOTION_MASK | Gdk::ENTER_NOTIFY_MASK | Gdk::LEAVE_NOTIFY_MASK);
}

ProperGame::~ProperGame()
{
	unsigned z = get_board_size();
	z *= z;
	if (label != NULL) {
		for (unsigned i = 0; i < z; i ++) if (label[i] != NULL) delete label[i];
		delete[] label;
		delete[] mark;
	}
	// std::cerr << "ProperGame deleted" << std::endl;
}

void ProperGame::set_board_size(unsigned s)
{
	Game::set_board_size(s);
	s *= s;
	label = new std::string * [s];
	mark = new Mark [s];
	for (unsigned i = 0; i < s; i ++) {
		label[i] = NULL;
		mark[i] = MARK_NONE;
	}
}

void ProperGame::clear_proper()
{
	unsigned s = get_board_size();
	for (unsigned i = 0; i < s * s; i ++) {
		bool c = false;
		if (label[i] != NULL) {
			delete label[i];
			label[i] = NULL;
			c = true;
		}
		if (mark[i] != MARK_NONE) {
			mark[i] = MARK_NONE;
			c = true;
		}
		if (c) draw_loc(Loc::distribute(s, i));
	}
	score_modify_w = 0;
	score_modify_b = 0;
	update_score();
}

void ProperGame::set_proper(const Proper & p)
{
	unsigned s = get_board_size();
	const std::vector<Proper::Label> & l = p.get_labels();
	for (std::vector<Proper::Label>::const_iterator i = l.begin(); i != l.end(); i ++) {
		if (label[i->loc.expand(s)]  != NULL) continue;
		if (i->label == "") continue;
		label[i->loc.expand(s)] = new std::string(i->label);
		draw_loc(i->loc);
	}
	const std::vector<Proper::Marking> & m = p.get_marks();
	for (std::vector<Proper::Marking>::const_iterator i = m.begin(); i != m.end(); i ++) {
		mark[i->loc.expand(s)] = i->mark;
		draw_loc(i->loc);
	}
	Glib::RefPtr<Gtk::TextBuffer> tb = text_port.get_buffer();
	tb->erase(tb->begin(), tb->end());
	if (p.get_name() != "") tb->insert(tb->begin(), std::string(_("Node: ")) + p.get_name() + "\n");
	tb->insert(tb->end(), p.get_comment());
	// modifi port
	modifi_port.set_text(_("node"));
	if (dynamic_cast<const RootProper *>(& p)) modifi_port.set_text(_("start"));
	const ModifiProper *ap = dynamic_cast<const ModifiProper *>(& p);
	if (ap) {
		if (dynamic_cast<const Setup *>(ap->get_modifi()) != NULL) modifi_port.set_text(_("setup"));
		const Move * m = dynamic_cast<const Move *>(ap->get_modifi());
		if (m) modifi_port.set_text(m->as_string());
	}
	const ScoreProper * sp = dynamic_cast<const ScoreProper *>(& p);
	if (sp) {
		// std::cerr << "score proper" << std::endl;
		score_modify_w = sp->get_terri_score_w();
		score_modify_b = sp->get_terri_score_b();
		double w = get_komi() + get_cap_w() + score_modify_w;
		double b = get_cap_b() + score_modify_b;
		char buf[100];
		if (w >= b) snprintf(buf, 100, _("Score: W+%g").c_str(), w - b);
		else snprintf(buf, 100, _("Score: B+%g").c_str(), b - w);
		modifi_port.set_text(buf);
	}
	update_score();
}

Gtk::Label & ProperGame::w_score_widget()
{
	return w_score_port;
}

Gtk::Label & ProperGame::b_score_widget()
{
	return b_score_port;
}

Gtk::TextView & ProperGame::text_widget()
{
	return text_port;
}

Gtk::Label & ProperGame::modifi_widget()
{
	return modifi_port;
}

Gtk::Label & ProperGame::label_widget()
{
	return label_port;
}
