/* ccgo: play_game.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "config.hh"
#include "gettext.h"
#include "play_game.hh"
#include "score_proper.hh"

#define _(String) Glib::locale_to_utf8(gettext(String))

using namespace go;

void PlayGame::do_track()
{
	if (! track_mode) return;
	if (! get_track()) return;
	switch (track_mode) {
	case TRACK_NONE:
		return;
	case TRACK_SCORE:
	case TRACK_MY_SCORE:
		if (! get_site(get_track())) return;
		break;
	case TRACK_PLAY:
		if (get_site(get_track()) && ! get_rule().overlap_stone) return;
		if (get_track() == ko && ! get_rule().force_ko) return;
		break;
	}
	track_on = get_track();
	mark_down = false;
	int cx;
	int cy;
	trans_loc(track_on, cx, cy);
	if (track_mode != TRACK_PLAY) {
		Glib::RefPtr<const Gdk::GC> gc = (get_site(track_on) == STATE_WHITE ? style->get_black_gc() : style->get_white_gc());
		win->draw_arc(gc, false, cx - cell_w / 4, cy - cell_h / 4, cell_w / 2, cell_h / 2, 0, 64 * 360);
	} else {
		Glib::RefPtr<const Gdk::GC> gc = (get_turn() == TURN_BLACK ? style->get_black_gc() : style->get_white_gc());
		win->draw_arc(gc, true, cx - cell_w / 2, cy - cell_h / 2, cell_w - 1, cell_h - 1, 0, 64 * 360);
	}
}

void PlayGame::clear_track()
{
	if (! track_on) return;
	draw_loc(track_on);
	track_on = Loc();
	mark_down = false;
}

void PlayGame::mark_track()
{
	if (! track_on || ! track_mode) {
		return;
	}
	int cx;
	int cy;
	trans_loc(track_on, cx, cy);
	if (track_mode != TRACK_PLAY) {
		Glib::RefPtr<const Gdk::GC> gc = (get_site(track_on) == STATE_BLACK ? style->get_white_gc() : style->get_black_gc());
		win->draw_arc(gc, true, cx - cell_w / 4, cy - cell_h / 4, cell_w / 2, cell_h / 2, 0, 64 * 360);
	} else {
		Glib::RefPtr<const Gdk::GC> gc = (get_turn() == TURN_BLACK ? style->get_white_gc() : style->get_black_gc());
		win->draw_arc(gc, false, cx - cell_w / 2, cy - cell_h / 2, cell_w - 1, cell_h - 1, 0, 64 * 360);
	}
	mark_down = true;
}

void PlayGame::on_track_change()
{
	if (track_on) clear_track();
	if (! track_mode) return;
	if (get_track()) do_track();
	return;
}

void PlayGame::pass_clicked()
{
	if (player_input) {
		if (playing) arena->request_pass(this);
		else if (scoring) arena->request_done(this);
	}
}

void PlayGame::undo_clicked()
{
	if (player_input) {
		if (playing) arena->request_undo(this);
		else if (scoring) arena->request_reset(this);
	}
}

void PlayGame::handi_clicked()
{
	if (player_input) {
		if (playing) arena->request_handicap(this, unsigned(h_adj->get_value()));
	}
}

bool PlayGame::on_button_press_event(GdkEventButton * e)
{
	if (! track_mode) {
		if (track_on) clear_track();
		return true;
	}
	if (track_on) mark_track();
	return true;
}

bool PlayGame::on_button_release_event(GdkEventButton * e)
{
	if (track_on && mark_down) {
		switch (track_mode) {
		case TRACK_NONE:
			break;
		case TRACK_PLAY:
			if (playing && player_input) arena->request_put(this, track_on);
			break;
		case TRACK_SCORE:
			if (scoring && player_input) {
				if (arena->request_terri(this, track_on)) claim_terri(track_on);
			}
			break;
		case TRACK_MY_SCORE:
			claim_terri(track_on);
			break;
		}
		clear_track();
	}
	return true;
}

void PlayGame::show_hide_buttons()
{
	if (get_move_num()) {
		handi_control.hide();
		undo_button.show();
	} else {
		undo_button.hide();
		if (get_board_size() == 19 && handicap == 0) handi_control.show();
		else handi_control.hide();
	}
}

void PlayGame::update_terri_scoring()
{
	if ((track_mode != TRACK_SCORE && track_mode != TRACK_MY_SCORE) || ! terri) {
		clear_proper();
		return;
	}
	unsigned z = get_board_size();
	score_terri(terri);
	Terri * t = new Terri[z * z];
	for (unsigned i = 0; i < z * z; i ++) t[i] = terri[i];
	ScoreProper sp;
	sp.set_terri(* this, t);
	clear_proper();
	set_proper(sp);
}

void PlayGame::score_toggle()
{
	if (setting_track_mode) return;
	if (score_bt.get_active()) {
		set_track_mode(TRACK_MY_SCORE);
	} else {
		end_my_score();
	}
}

void PlayGame::end_my_score()
{
}

PlayGame::PlayGame() :
	track_mode(TRACK_NONE),
	setting_track_mode(false),
// 	tracking(false),
// 	track_scoring(false),
	player_input(false),
	terri(0),
	terri_score_w(0),
	terri_score_b(0),
	track_on(),
	mark_down(false),
	pass_button(_("Pass")),
	undo_button(_("Undo")),
	h_adj(Gtk::manage(new Gtk::Adjustment(2, 2, 9))),
	score_bt(_("Score"))
{
	control_box.pack_start(pass_button, Gtk::PACK_SHRINK);
	pass_button.signal_clicked().connect(mem_fun(* this, & PlayGame::pass_clicked));
	control_box.pack_start(undo_button, Gtk::PACK_SHRINK);
	undo_button.signal_clicked().connect(mem_fun(* this, & PlayGame::undo_clicked));
	Gtk::Button * bt = Gtk::manage(new Gtk::Button(_("Handi")));
	bt->signal_clicked().connect(mem_fun(* this, & PlayGame::handi_clicked));
	handi_control.pack_start(* bt, Gtk::PACK_SHRINK);
	Gtk::SpinButton * sb = Gtk::manage(new Gtk::SpinButton(* h_adj));
	handi_control.pack_start(* sb, Gtk::PACK_SHRINK);
	control_box.pack_start(handi_control, Gtk::PACK_SHRINK);
	control_box.set_sensitive(false);
	score_bt.signal_clicked().connect(mem_fun(* this, & PlayGame::score_toggle));
	add_events(Gdk::BUTTON_PRESS_MASK | Gdk::BUTTON_RELEASE_MASK);
	show_hide_buttons();
}

PlayGame::~PlayGame()
{
	if (terri) delete[] terri;
}

void PlayGame::set_board_size(unsigned z)
{
	ProperGame::set_board_size(z);
	if (terri) delete[] terri;
	terri = new Terri[z * z];
}

Gtk::Widget * PlayGame::get_control()
{
	return & control_box;
}

Gtk::Widget * PlayGame::get_score_bt()
{
	return & score_bt;
}

void PlayGame::set_move_num(unsigned n)
{
	unsigned o = get_move_num();
	Game::set_move_num(n);
	if (bool(o) == bool(n)) return;
	show_hide_buttons();
}

void PlayGame::set_track_mode(Track tm)
{
	setting_track_mode = true;
	track_mode = tm;
	clear_track();
	terri_clear();
	switch (track_mode) {
	case TRACK_NONE:
		control_box.set_sensitive(false);
		score_bt.set_active(false);
		score_bt.set_sensitive(true);
		break;
	case TRACK_PLAY:
		pass_button.set_label(_("Pass"));
		undo_button.set_label(_("Undo"));
		control_box.set_sensitive(true);
		score_bt.set_active(false);
		score_bt.set_sensitive(true);
		break;
	case TRACK_SCORE:
		pass_button.set_label(_("Done"));
		undo_button.set_label(_("Reset"));
		control_box.set_sensitive(true);
		update_terri_scoring();
		score_bt.set_sensitive(false);
		break;
	case TRACK_MY_SCORE:
		control_box.set_sensitive(false);
		score_bt.set_sensitive(true);
		update_terri_scoring();
		break;
	}
	show_hide_buttons();
	setting_track_mode = false;
	do_track();
}

PlayGame::Track PlayGame::get_track_mode() const
{
	return track_mode;
}

void PlayGame::set_player_input(bool b)
{
	player_input = b;
}

void PlayGame::claim_terri(const Loc & l)
{
	if (! get_site(l)) return;
	std::vector<Loc> g;
	get_group(l, g);
	terri_mark(g);
	update_terri_scoring();
}

void PlayGame::terri_mark(const std::vector<Loc> & t)
{
	unsigned z = get_board_size();
	bool marked = true;
	for (std::vector<Loc>::const_iterator i = t.begin(); i != t.end(); i ++) {
		Terri t = get_site(* i) == STATE_WHITE ? TERRI_BLACK : TERRI_WHITE;
		if (terri[(* i).expand(z)] != t) {
			terri[(* i).expand(z)] = t;
			marked = false;
		}
	}
	if (marked) for (std::vector<Loc>::const_iterator i = t.begin(); i != t.end(); i ++) {
		terri[(* i).expand(z)] = TERRI_NEUTRAL;
	}
}

void PlayGame::terri_clear()
{
	unsigned z = get_board_size();
	z *= z;
	for (unsigned i = 0; i < z; i ++) terri[i] = TERRI_NEUTRAL;
	// update_terri_scoring();
}

void PlayGame::set_terri(Terri * t)
{
	if (! terri) return;
	unsigned z = get_board_size();
	z *= z;
	for (unsigned i = 0; i < z; i ++) terri[i] = t[i];
	update_terri_scoring();
}

void PlayGame::enable_player()
{
	Player::enable_player();
}

void PlayGame::disable_player()
{
	Player::disable_player();
}

void PlayGame::enter_scoring()
{
	Player::enter_scoring();
}

void PlayGame::done_scoring()
{
	Player::done_scoring();
}
