/* ccgo: part.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "config.hh"
#include "gettext.h"
#include "part.hh"
#include "root_proper.hh"
#include "put_move.hh"
#include "handicap.hh"
#include "part_win.hh"
#include <gtkmm/spinbutton.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/box.h>
#include <gtkmm/alignment.h>
#include <sstream>
#include <gtkmm/togglebutton.h>
#include <gtkmm/tooltips.h>

#define _(String) Glib::locale_to_utf8(gettext(String))

using namespace go;

void Part::change_position()
{
	unsigned p = unsigned(p_adj->get_value());
	while (walk.pos_num() < p) forward();
	while (walk.pos_num() > p) backward();
}

void Part::find_range()
{
	while (tail.down());
	int r = tail.pos_num();
	p_adj->set_upper(r);
	std::ostringstream s;
	s << r;
	n_move.set_text(s.str());
	p_adj->set_value(walk.pos_num());
}

void Part::duplicate()
{
	new PartWin(* this);
}

void Part::init()
{
	walk.start(); // setup the boards;
	tail.start();
	walk.set_play_node(& tail);

	// GUI setup
	Gtk::Tooltips * tt = Gtk::manage(new Gtk::Tooltips);

	button_area = Gtk::manage(new Gtk::VBox);
	text_sw.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_ALWAYS);
	text_sw.add(walk.text_widget());

	Gtk::SpinButton * sb = Gtk::manage(new Gtk::SpinButton(* p_adj, 0, 0));
	// button_area->pack_start(* Gtk::manage(new Gtk::Label("Node")), Gtk::PACK_SHRINK);
	button_area->pack_start(n_move, Gtk::PACK_SHRINK);
	button_area->pack_start(* sb, Gtk::PACK_SHRINK);
	Gtk::Alignment * al;
	al = Gtk::manage(new Gtk::Alignment(0.5, 0.5, 0, 0));
	button_area->pack_start(* al, Gtk::PACK_SHRINK);
	Gtk::HBox * hb;
	hb = Gtk::manage(new Gtk::HBox);
	al->add(* hb);
	al = Gtk::manage(new Gtk::Alignment(0.5, 0.5, 0, 0));
	button_area->pack_start(* al, Gtk::PACK_SHRINK);
	hb = Gtk::manage(new Gtk::HBox);
	al->add(* hb);
	hb->pack_start(* Gtk::manage(new Gtk::Label(_("W:"))), Gtk::PACK_SHRINK);
	hb->pack_start(walk.w_score_widget(), Gtk::PACK_SHRINK);
	al = Gtk::manage(new Gtk::Alignment(0.5, 0.5, 0, 0));
	button_area->pack_start(* al, Gtk::PACK_SHRINK);
	hb = Gtk::manage(new Gtk::HBox);
	al->add(* hb);
	hb->pack_start(* Gtk::manage(new Gtk::Label(_("B:"))), Gtk::PACK_SHRINK);
	hb->pack_start(walk.b_score_widget(), Gtk::PACK_SHRINK);
	p_adj->signal_value_changed().connect(sigc::mem_fun(* this, & Part::change_position));
	button_area->pack_start(* walk.get_control(), Gtk::PACK_SHRINK);
	button_area->pack_start(time);
	time.setup(60, 300, 25);
	button_area->pack_end(* walk.get_score_bt(), Gtk::PACK_SHRINK);
	tt->set_tip(* walk.get_score_bt(), _("Temporary score mode"));

	Gtk::Button * bt = Gtk::manage(new Gtk::Button(Gtk::Stock::COPY));
	button_area->pack_end(* bt, Gtk::PACK_SHRINK);
	bt->signal_clicked().connect(sigc::mem_fun(* this, & Part::duplicate));
	tt->set_tip(* bt, _("Duplicate the game"));

	find_range();
	walk.update_proper();
	no_play = false;
	hold_action = false;
}

Part::Part(unsigned z, double k, unsigned h) :
	root(new ProperNode),
	walk(root),
	tail(root),
	p_adj(Gtk::manage(new Gtk::Adjustment(0, 0, 0))),
	n_move("0"),
	terri(0)
{
	RootProper * rp = new RootProper;
	rp->set_size(z);
	rp->set_komi(k);
	rp->set_handicap(h);
	if (h > 1 && z == 19) { // set handicap, assume japanese rule
		std::vector<Loc> handi = Handicap(h).get_handi();
		Board b(z);
		for (std::vector<Loc>::iterator i = handi.begin(); i != handi.end(); i ++) b.set_site(* i, STATE_BLACK);
		rp->set_init(& b);
	} else rp->set_handicap(1); // assume no handicap otherwise
	root->set_proper(walk, rp);
	init();
}

Part::Part(const Part & p) :
	root(dynamic_cast<go::ProperNode *>(p.root->dup())),
	walk(root),
	tail(root),
	p_adj(Gtk::manage(new Gtk::Adjustment(0, 0, 0))),
	n_move("0"),
	terri(0)
{
	init();
	while (walk != tail) forward();
	walk.do_player();
}

Part::~Part()
{
	// std::cerr << "deleting Part" << std::endl;
	Arena::stop_action();
	time.stop();
	if (terri) delete[] terri;
}

void Part::forward()
{
	walk.down();
	walk.update_proper();
	score_bt.set_active(false);
	p_adj->set_value(walk.pos_num());
	walk.do_player();
}

void Part::backward()
{
	walk.up();
	walk.update_proper();
	score_bt.set_active(false);
	p_adj->set_value(walk.pos_num());
	walk.do_player();
}

const Game * Part::get_game() const
{
	return & tail;
}

void Part::set_players(Player * w, Player * b)
{
	set_player(TURN_WHITE, w ? w : & walk);
	set_player(TURN_BLACK, b ? b : & walk);
	Arena::start_action();
	// time.start();
	walk.do_player();
}

void Part::enter_score()
{
	Arena::enter_score();
	unsigned z = tail.get_board_size();
	if (terri) delete[] terri;
	terri = new Terri[z * z];
	for (unsigned i = 0; i < z * z; i ++) terri[i] = TERRI_NEUTRAL;
	time.stop();
}

void Part::done_score()
{
	Arena::done_score();
}

bool Part::action_handicap(unsigned n)
{
	if (hold_action) {
		held_handicap(n);
		return false;
	}
	if (! tail.add_move(new Handicap(n))) return false;
	post_action();
	return true;
}

bool Part::action_put(const Loc & l)
{
	if (hold_action) {
		held_put(l, time.get_counter());
		return false;
	}
	if (! tail.add_move(new PutMove(tail.get_turn(), l))) return false;
	post_action();
	return true;
}

bool Part::action_pass()
{
	if (hold_action) {
		held_pass(time.get_counter());
		return false;
	}
	if (! tail.add_move(new Move(tail.get_turn()))) return false;
	post_action();
	return true;
}

bool Part::action_terri(const Loc & l)
{
	if (hold_action) {
		held_terri(l);
		return false;
	}
	if (! tail.get_site(l)) return false;
	std::vector<Loc> g;
	tail.get_group(l, g);
	unsigned z = tail.get_board_size();
	for (std::vector<Loc>::const_iterator i = g.begin(); i != g.end(); i ++) {
		terri[(* i).expand(z)] = tail.get_site(* i) == STATE_WHITE ? TERRI_BLACK : TERRI_WHITE;
	}
	return true;
}

bool Part::action_reset()
{
	if (hold_action) {
		held_reset();
		return false;
	}
	unsigned z = tail.get_board_size();
	for (unsigned i = 0; i < z * z; i ++) terri[i] = TERRI_NEUTRAL;
	return true;
}

bool Part::action_done()
{
	if (hold_action) {
		held_done();
		return false;
	}
	tail.score_terri(terri);
	tail.add_score(terri);
	terri = 0;
	done_score();
	post_action();
	return true;
}

void Part::ask_terri(Terri * t)
{
	unsigned z = tail.get_board_size();
	if (terri) for (unsigned i = 0; i < z * z; i ++) t[i] = terri[i];
	else {
// 		std::cerr << "NO terri" << std::endl;
		for (unsigned i = 0; i < z * z; i ++) t[i] = TERRI_NEUTRAL;
	}
}

void Part::post_action()
{
	bool a = walk == tail && walk.get_track_mode() != PlayGame::TRACK_MY_SCORE;
	find_range();
	if (a) {
		forward(); // walk follows tail
		walk.do_player();
	}
	time.toggle();
}

unsigned Part::get_time_counter()
{
	return time.get_counter();
}

bool Part::action_undo()
{
	if (hold_action) {
		held_undo();
		return false;
	}
	if (! tail.pos_num()) return false; // no move to undo
	if (walk == tail) backward();
	tail.up();
	tail.get_pos()->delete_branch();
	find_range();
	if (walk == tail) {
		walk.update_proper();
		p_adj->set_value(walk.pos_num());
		walk.do_player();
	}
	return true;
}

void Part::set_komi(double k)
{
	RootProper * rp = dynamic_cast<RootProper *>(root->get_proper());
	if (rp) {
		rp->set_komi(k);
	}
	walk.set_komi(k);
	tail.set_komi(k);
}

bool Part::add_move(Move * m)
{
	if (! tail.add_move(m)) return false;
	bool a = walk == tail && walk.get_track_mode() != PlayGame::TRACK_MY_SCORE;
	find_range();
	if (a) {
		forward(); // walk follows tail
		walk.do_player();
	}
	time.toggle();
	return true;
}

bool Part::undo_move()
{
	if (! tail.pos_num()) return false;
	if (walk == tail) backward();
	tail.up();
	tail.get_pos()->delete_branch();
	find_range();
	if (walk == tail) {
		walk.update_proper();
		p_adj->set_value(walk.pos_num());
		walk.do_player();
	}
	return true;
}


bool Part::terri_request(const go::Loc & l)
{
	if (! tail.get_site(l)) return false;
	std::vector<Loc> g;
	tail.get_group(l, g);
	unsigned z = tail.get_board_size();
	for (std::vector<Loc>::const_iterator i = g.begin(); i != g.end(); i ++) {
		terri[(* i).expand(z)] = tail.get_site(* i) == STATE_WHITE ? TERRI_BLACK : TERRI_WHITE;
	}
	walk.do_player();
	return true;
}

bool Part::restore_board()
{
	if (! terri) return false;
// 	std::cerr << "restoring board" << std::endl;
	unsigned z = tail.get_board_size();
	for (unsigned i = 0; i < z * z; i ++) terri[i] = TERRI_NEUTRAL;
	walk.do_player();
	return true;
}

void Part::start_time(int t, int bt, int bm)
{
	time.setup(t, bt, bm);
	time.start();
}

void Part::time_update(int wt, int wm, int bt, int bm)
{
	time.calibrate(wt, wm, bt, bm);
}

void Part::set_no_play(bool b)
{
	no_play = b;
	if (b) walk.get_control()->hide();
	else walk.get_control()->show();
}

void Part::set_hold_action(bool b)
{
	hold_action = b;
}

void Part::stop_game()
{
	time.stop();
	Arena::stop_action();
	walk.do_player();
}

void Part::add_score(Terri * t)
{
	tail.add_score(t);
	bool a = walk == tail && walk.get_track_mode() != PlayGame::TRACK_MY_SCORE;
	find_range();
	if (a) {
		forward();
		walk.do_player();
	}
}

Gtk::Widget & Part::get_board_area()
{
	return walk;
}

Gtk::Widget & Part::get_button_area()
{
	return * button_area;
}

Gtk::Widget & Part::get_text_area()
{
	return text_sw;
}

Gtk::Widget & Part::get_modifi_area()
{
	return walk.modifi_widget();
}

Gtk::Widget & Part::get_label_area()
{
	return walk.label_widget();
}
