/* ccgo: nc/nc.cc
 *
 * Copyright (C) 2006 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
#include <nc/nc.hh>
#include <iostream>
extern "C" {
#include <ncurses.h>
}

using namespace nc;
using namespace std;

void Nc::new_window_size(unsigned w, unsigned h)
{
}

void Nc::prompt()
{
	move_cursor_to(0, 0);
}

void Nc::process_key(int c)
{
	clear_message();
}

void Nc::draw_message(const string & str)
{
	draw_text(0, 0, str);
}

void Nc::clear_message()
{
	move_cursor_to(0, 0);
	clear_to_eol();
}

void Nc::stop()
{
	running = false;
}

Nc::Nc() :
	running(false)
{
}

Nc::~Nc()
{
}

void Nc::run()
{
	initscr();
	nonl();
	cbreak();
	noecho();
	keypad(stdscr, true);
	// define addition keys...
	// gnome-terminal
	define_key("\033O5B", KeyCtrlDown);
	define_key("\033O5A", KeyCtrlUp);
	define_key("\033O5D", KeyCtrlLeft);
	define_key("\033O5C", KeyCtrlRight);
	// xterm
	define_key("\033[1;5B", KeyCtrlDown);
	define_key("\033[1;5A", KeyCtrlUp);
	define_key("\033[1;5D", KeyCtrlLeft);
	define_key("\033[1;5C", KeyCtrlRight);
	if (has_colors()) {
		start_color();
		use_default_colors();
	}
	bkgd_char = getbkgd(stdscr);
	running = true;
	int w;
	int h;
	getmaxyx(stdscr, h, w);
	new_window_size(w, h);
	do {
		prompt();
		int c = getch();
		clear_message();
		if (c == KEY_RESIZE) {
			getmaxyx(stdscr, h, w);
			clear();
			new_window_size(w, h);
			continue;
		}
		if (c == ERR) {
			continue;
		}
		process_key(c);
	} while (running);
	endwin();
}

void Nc::move_cursor_to(unsigned x, unsigned y)
{
	move(y, x);
}

void Nc::draw_char(char c)
{
	addch(c);
}

void Nc::draw_char(unsigned x, unsigned y, char c)
{
	mvaddch(y, x, c);
}

void Nc::draw_text(std::string str)
{
	addstr(str.c_str());
}

void Nc::draw_text(unsigned x, unsigned y, std::string str)
{
	mvaddstr(y, x, str.c_str());
}

void Nc::clear_to_eol()
{
	clrtoeol();
}

void Nc::set_reverse()
{
	attron(A_REVERSE);
	bkgd_char |= A_REVERSE;
	bkgdset(bkgd_char);
}

void Nc::unset_reverse()
{
	attroff(A_REVERSE);
	bkgd_char &= ~ A_REVERSE;
	bkgdset(bkgd_char);
}

void Nc::set_bold()
{
	attron(A_BOLD);
	bkgd_char |= A_BOLD;
	bkgdset(bkgd_char);
}

void Nc::unset_bold()
{
	attroff(A_BOLD);
	bkgd_char &= ~ A_BOLD;
	bkgdset(bkgd_char);
}

ColorPair Nc::color_pairs()
{
	return COLOR_PAIRS;
}

void Nc::make_color(ColorPair pair, Color fg, Color bg)
{
	init_pair(pair, fg, bg);
}

void Nc::set_color(ColorPair pair)
{
	attron(COLOR_PAIR(pair));
	bkgd_char |= COLOR_PAIR(pair);
	bkgdset(bkgd_char);
}

void Nc::unset_color(ColorPair pair)
{
	attroff(COLOR_PAIR(pair));
	bkgd_char &= ~ COLOR_PAIR(pair);
	bkgdset(bkgd_char);
}

int Nc::get_key()
{
	return getch();
}

string Nc::read_line()
{
	char buf[256];
	echo();
	getnstr(buf, 256);
	noecho();
	return buf;
}

void Nc::message(string str)
{
	draw_message(str);
}

void Nc::reconfigure()
{
	if (! running) return;
	int h;
	int w;
	getmaxyx(stdscr, h, w);
	clear();
	new_window_size(w, h);
}
