/* ccgo: nc/face.cc
 *
 * Copyright (C) 2006 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
#include <nc/face.hh>
#include <debug.hh>
#include <iostream>
#include <sstream>
#include <iomanip>

using namespace nc;
using namespace std;

void Face::new_face_size()
{
	size = get_size();
	cur_x = size / 2;
	cur_y = size / 2;
	if (nc) nc->reconfigure();
}

void Face::new_state(go::Loc loc)
{
	draw_site(loc);
}

void Face::new_mode()
{
	erase_cursor();
	switch (get_mode()) {
	case VIEW_MODE:
		cur_left = '{';
		cur_right = '}';
		break;
	case EDIT_MODE:
		cur_left = '(';
		cur_right = ')';
		break;
	case MOVE_MODE:
		cur_left = '[';
		cur_right = ']';
		break;
	case PICK_MODE:
		cur_left = '>';
		cur_right = '<';
		break;
	}
	draw_cursor();
}

void Face::new_move_from(go::Move old)
{
	if (get_move() > go::MovePass) {
		move_cursor(go::Loc(get_move()));
		draw_site(go::Loc(get_move()));
	}
	if (old > go::MovePass) draw_site(go::Loc(old));
}

void Face::new_ko_from(const std::set<go::Loc> & old)
{
	std::set<go::Loc> k = get_ko();
	if (old.size()) k.insert(old.begin(), old.end());
	for (std::set<go::Loc>::iterator i = k.begin(); i != k.end(); i ++) draw_site(* i);
}

void Face::new_mark(go::Loc loc)
{
	draw_site(loc);
}

void Face::new_label(go::Loc loc)
{
	draw_site(loc);
}

Face::Face() :
	nc(0),
	shown(false),
	active(false),
	size(get_size())
{
	new_mode();
}

void Face::set_nc(Nc * n)
{
	nc = n;
}

unsigned Face::get_max_w() const
{
	return size * 2 + 5;
}

unsigned Face::get_max_h() const
{
	return size + 2;
}

void Face::set_window(int x, int y, int w, int h)
{
	assert(nc);
	win_x = x;
	win_y = y;
	win_w = w;
	win_h = h;
	shown = w >= size * 2 + 5 && h >= size + 2;
	c_move = 1;
	c_mark = 2;
	c_label = 3;
	assert(nc->color_pairs() > 3);
	nc->make_color(c_move, ColorRed, ColorDefault);
	nc->make_color(c_mark, ColorGreen, ColorDefault);
	nc->make_color(c_label, ColorCyan, ColorDefault);
	draw();
}

void Face::set_active(bool act)
{
	erase_cursor();
	active = act;
	draw_cursor();
}

bool Face::get_active() const
{
	return active;
}

void Face::do_up()
{
	erase_cursor();
	if (cur_y < size - 1) cur_y ++;
	draw_cursor();
}

void Face::do_down()
{
	erase_cursor();
	if (cur_y > 0) cur_y --;
	draw_cursor();
}

void Face::do_left()
{
	erase_cursor();
	if (cur_x > 0) cur_x --;
	draw_cursor();
}

void Face::do_right()
{
	erase_cursor();
	if (cur_x < size - 1) cur_x ++;
	draw_cursor();
}

void Face::do_enter()
{
	go::Loc loc = cur_y * size + cur_x;
	switch (get_mode()) {
	case VIEW_MODE:
		break;
	case EDIT_MODE:
		set_state(loc, go::Loc((state(loc) + 1) % 3));
		break;
	case MOVE_MODE:
		if (state(loc) == go::Empty && ! loc_is_ko(loc)) play_move(loc);
		else nc->message("can't place stone here!");
		break;
	case PICK_MODE:
		if (state(loc) != go::Empty) pick_stone(loc);
		break;
	}
}

void Face::do_knock()
{
	go::Loc loc = cur_y * size + cur_x;
	assert(get_mode() != EDIT_MODE);
	knock_site(loc);
}

void Face::do_track()
{
	if (active && size) track(go::Loc(cur_y * size + cur_x));
}

void Face::move_cursor(go::Loc loc)
{
	erase_cursor();
	cur_x = loc % size;
	cur_y = loc / size;
	draw_cursor();
}

void Face::draw_cursor()
{
	if (! active || ! size) return;
	unsigned wx = win_x + 2 + cur_x * 2;
	unsigned wy = win_y + size - cur_y;
	nc->move_cursor_to(wx, wy);
	nc->draw_char(cur_left);
	wx += 2;
	nc->move_cursor_to(wx, wy);
	nc->draw_char(cur_right);
}

void Face::erase_cursor()
{
	if (! active || ! size) return;
	unsigned wx = win_x + 2 + cur_x * 2;
	unsigned wy = win_y + size - cur_y;
	nc->move_cursor_to(wx, wy);
	nc->draw_char(' ');
	wx += 2;
	nc->move_cursor_to(wx, wy);
	nc->draw_char(' ');
}

void Face::draw_site(go::Loc loc)
{
	if (! shown) return;
	int x = loc % size;
	int y = loc / size;

	nc->move_cursor_to(win_x + 3 + x * 2, win_y + size - y);
	Color c = 0;
	if (get_mark(loc) != go::MARK_NONE) c = c_mark;
	else if (get_label(loc).size()) c = c_label;
	else if (loc == go::Loc(get_move())) c = c_move;
	if (c) nc->set_color(c);
	switch (state(loc)) {
	case go::Empty:
		if (loc_is_ko(loc)) nc->draw_char('x');
		else if (loc_is_star(loc)) nc->draw_char('+');
		else if (get_mark(loc) == go::MARK_W_TERRITORY || get_mark(loc) == go::MARK_B_TERRITORY) nc->draw_char('-');
		else nc->draw_char('.');
		break;
	case go::Black:
		nc->draw_char('#');
		break;
	case go::White:
		nc->draw_char('O');
		break;
	}
	if (c) nc->unset_color(c);
}

void Face::draw()
{
	if (! shown) return;
	for (go::Loc i = 0; i < range(); i ++) draw_site(i);
	for (int i = 0; i < size; i ++) {
		char c = 'a' + i;
		if (c >= 'i') c ++; // skip 'i'
		nc->draw_char(win_x + 3 + i * 2, win_y, c);
		nc->draw_char(win_x + 3 + i * 2, win_y + size + 1, c);
		c = '0' + (i + 1) % 10;
		nc->draw_char(win_x + 1, win_y + size - i, c);
		nc->draw_char(win_x + size * 2 + 3 + (i + 1) / 10, win_y + size - i, c);
		if (i > 8) {
			char c = '0' + (i + 1) / 10;
			nc->draw_char(win_x, win_y + size - i, c);
			nc->draw_char(win_x + size * 2 + 3, win_y + size - i, c);
		}
	}
	draw_cursor();
}
