/* ccgo: nc/entry.cc
 *
 * Copyright (C) 2006 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
#include <nc/entry.hh>
#include <debug.hh>
#include <sstream>
#include <iomanip>

using namespace nc;
using namespace std;

Entry::Entry() :
	nc(0),
	shown(false),
	entx(0),
	enty(0),
	entw(0),
	enth(0),
	entl(0),
	entc(0),
	curl(0),
	curc(0),
	top(0),
	topc(0),
	show_guide(false)
{
	buf.push_back("");
}

Entry::~Entry()
{
}

void Entry::set_nc(Nc * n)
{
	nc = n;
	shown = false;
}

void Entry::set_window(unsigned x, unsigned y, unsigned width, unsigned height)
{
	assert(nc);
	entx = x;
	enty = y;
	entw = width;
	enth = height;
	entl = 0;
	entc = 0;
	curl = 0;
	curc = 0;
	top = 0;
	topc = 0;
	shown = width > 2 && height > 0;
	draw_buf();
}

bool Entry::is_shown() const
{
	return shown;
}

void Entry::draw_buf()
{
	if (! shown) return;
	unsigned bl = top; // start from top line
	unsigned bc = topc; // column offset of top line
	for (int l = 0; l < enth; l ++) {
		int c = 0; // current char
		nc->move_cursor_to(entx, enty + l);
		if (show_guide) {
			if (bc == 0) {
				if (bl < buf.size()) nc->draw_char(']');
				else nc->draw_char('~'); // out of buffer range
			} else nc->draw_char(' '); // continuing line
			c ++;
		}
		if (bl < buf.size()) {
			string s = buf[bl].substr(bc, entw - c);
			nc->draw_text(s);
			if (int(s.length()) < entw - c) { // whole buffer line used
				bl ++; // next buffer line
				bc = 0; // first column
				c += s.length();
			} else {
				bc += s.length();
				c = entw;
			}
		}
		while (c < entw) {
			nc->draw_char(' ');
			c ++;
		}
	}
}

void Entry::set_cursor()
{
	if (! shown) return;
	// count lines
	int wid = entw;
	if (show_guide) wid --;
	entl = 0;
	for (int i = top; i < curl; i ++) {
		entl += buf[i].length() / wid;
		entl ++;
	}
	entl += curc / wid;
	entl -= topc / wid;
	entc = curc % wid;
	if (show_guide) entc ++;
	nc->move_cursor_to(entx + entc, enty + entl);
}

string Entry::get_text()
{
	string t;
	for (vector<string>::const_iterator i = buf.begin(); i != buf.end(); i ++) {
		if (i != buf.begin()) t += '\n';
		t += * i;		
	}
	return t;
}

void Entry::set_text(const string & text)
{
	buf.clear();
	string::size_type i = 0;
	string::size_type j;
	while ((j = text.find('\n', i)) != string::npos) {
		buf.push_back(text.substr(i, j - i));
		i = j + 1;
	};
	buf.push_back(text.substr(i));
	entl = 0;
	entc = 0;
	curl = 0;
	curc = 0;
	top = 0;
	topc = 0;
	draw_buf();
}

void Entry::set_guide(bool guide_shown)
{
	if (guide_shown == show_guide) return;
	show_guide = guide_shown;
	draw_buf();
}

bool Entry::get_guide() const
{
	return show_guide;
}

void Entry::do_up()
{
	if (! shown) return;
	int wid = show_guide ? entw - 1 : entw; // text width
	if (curc >= wid) {
		curc -= wid;
		if (curl == top && curc < topc) {
			topc -= wid;
			draw_buf();
		}
	} else if (curl > 0) {
		curl --;
		curc += wid * (buf[curl].length() / wid);
		if (curc > (int) buf[curl].length()) {
			curc = buf[curl].length();
		}
		if (curl < top) {
			top = curl;
			topc = wid * (buf[top].length() / wid);
			draw_buf();
		}
	}
}

void Entry::do_down()
{
	if (! shown) return;
	int wid = show_guide ? entw - 1 : entw; // text width
	curc += wid;
	if (curc / wid > int(buf[curl].length()) / wid) {
		if (curl >= (int) buf.size() - 1) { // no more lines
			curc -= wid;
			return;
		}
		curc = curc % wid;
		curl ++;
	}
	if (curc > (int) buf[curl].length()) curc = buf[curl].length();
	if (entl == enth - 1) { // need to scroll
		topc += wid;
		if (topc > (int) buf[top].length()) {
			topc = 0;
			top ++;
		}
		draw_buf();
	}
}

void Entry::do_left()
{
	if (! shown) return;
	int wid = show_guide ? entw - 1 : entw; // text width
	if (curc == 0) return;
	curc --;
	if (curl == top && curc < topc) { // need to scroll?
		topc -= wid;
		draw_buf();
	}
}

void Entry::do_right()
{
	if (! shown) return;
	int wid = show_guide ? entw - 1 : entw; // text width
	if (curc >= (int) buf[curl].length()) return;
	curc ++;
	if (curc % wid == 0 && entl == enth - 1) { // need to scroll?
		topc += wid;
		if (topc > int(buf[top].length())) {
			topc = 0;
			top ++;
		}
		draw_buf();
	}
}

void Entry::do_home()
{
	if (! shown) return;
	curc = 0;
	if (curl == top && topc > 0) {
		topc = 0;
		draw_buf();
	}
}

void Entry::do_end()
{
	if (! shown) return;
	int wid = show_guide ? entw - 1 : entw; // text width
	int i = buf[curl].length() / entw - curc / wid;
	if (i + entl >= enth) { // need to scroll
		i = i + entl - enth + 1;
		while (i > 0) {
			topc += wid;
			if (topc > (int) buf[top].length()) {
				top ++;
				topc = 0;
			}
			i --;
		}
		draw_buf();
	}
	curc = buf[curl].length();
}

void Entry::do_enter()
{
	if (! shown) return;
	int wid = show_guide ? entw - 1 : entw; // text width
	buf.insert(buf.begin() + curl + 1, buf[curl].substr(curc));
	buf[curl].erase(curc);
	curl ++;
	curc = 0;
	if (entl == enth - 1) { // need to scrool
		topc += wid;
		if (topc > (int) buf[top].length()) {
			topc = 0;
			top ++;
		}
	}
	draw_buf();
}

void Entry::do_back()
{
	if (! shown) return;
	int wid = show_guide ? entw - 1 : entw; // text width
	if (curc > 0) {
		curc --;
		buf[curl].erase(curc, 1);
		if (curl == top && curc < topc) {
			topc --;
		}
		draw_buf();
	} else if (curl > 0) {
		curl --;
		curc = buf[curl].length();
		buf[curl].append(buf[curl + 1]);
		buf.erase(buf.begin() + curl + 1);
		if (curl < top) {
			top = curl;
			topc = wid * (curc / wid);
		}
		draw_buf();
	}
}

void Entry::do_delete()
{
	if (! shown) return;
	if (curc < (int) buf[curl].length()) {
		buf[curl].erase(curc, 1);
		draw_buf();
	} else if (curl < (int) buf.size() - 1) {
		buf[curl].append(buf[curl + 1]);
		buf.erase(buf.begin() + curl + 1);
		draw_buf();
	}
}

void Entry::do_page_up()
{
	if (! shown) return;
	int wid = show_guide ? entw - 1 : entw; // text width
	bool need_redraw = false;
	int i = enth - 2;
	while (i > 0) {
		if (topc > 0) topc -= wid;
		else if (top > 0) {
			top --;
			topc = wid * (buf[top].length() / wid);
		} else break;
		i --;
		entl ++;
		need_redraw = true;
	}
	while (entl >= enth) {
		if (curc >= wid) curc -= wid;
		else {
			curl --;
			curc = wid * (buf[curl].length() / wid) + curc % wid;
		}
		entl --;
	}
	if (curc > (int) buf[curl].length()) curc = buf[curl].length();
	if (need_redraw) draw_buf();
}

void Entry::do_page_down()
{
	if (! shown) return;
	int wid = show_guide ? entw - 1 : entw; // text width
	bool need_redraw = false;
	int i = enth - 2;
	while (i > 0) {
		if (topc + wid <= (int) buf[top].length()) topc += wid;
		else if (top < (int) buf.size() - 1) {
			top ++;
			topc = 0;
		} else break;
		i --;
		need_redraw = true;
	}
	while (curl < top) {
		curl ++;
		curc = curc % wid;
	}
	while (curc < topc) curc += wid;
	if (curc > (int) buf[curl].length()) curc = buf[curl].length();
	if (need_redraw) draw_buf();
}

void Entry::do_insert(int c)
{
	if (! shown) return;
	int wid = show_guide ? entw - 1 : entw; // text width
	if (c >= 32 && c < 128) { // normal key?
		char b[2];
		b[1] = 0;
		b[0] = c;
		buf[curl].insert(curc, b);
		curc ++;
		if (curc % wid == 0 && entl == enth - 1) { // need to scroll?
			topc += wid;
			if (topc > (int) buf[top].length()) {
				topc = 0;
				top ++;
			}
		}
		draw_buf();
	}
}
