/* ccgo: nc/editor.cc
 *
 * Copyright (C) 2006 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
#include <nc/editor.hh>
#include <nc/misc.hh>
#include <go/sgf.hh>
#include <debug.hh>
#include <fstream>
#include <sstream>
#include <iomanip>

using namespace nc;
using namespace std;

void Editor::view_site(go::Loc loc)
{
	ostringstream s;
	unsigned sz = face.get_size();
	assert(sz);
	s << xy_to_str(loc % sz, loc / sz) << '(';
	switch (face.state(loc)) {
	case go::Black:
		s << '#';
		break;
	case go::White:
		s << 'O';
		break;
	default:
		s << '.';
		break;
	}
	s << "):";
	if (face.get_mark(loc)) {
		s << '[' << mark_to_str(face.get_mark(loc)) << ']';
	}
	s << face.get_label(loc);
	message(s.str());
}

void Editor::new_node()
{
	if (is_attached()) text.set_text(get_text());
	else {
		text.set_text("");
		message("unattached to any game");
	}
	draw_status();
	edit_text = false;
	text.set_guide(false);
	face.do_track();
}

void Editor::change_site(go::Loc loc, go::Mark & mk, string & lb)
{
	if (! changer) return;
	if (changer == ChangeMark) {
		mk = go::Mark((mk + 1) % go::MARK_NUMBER);
		return;
	}
	assert(changer == ChangeLabel);
	ostringstream s;
	unsigned sz = face.get_size();
	s << "Change label at " << xy_to_str(loc % sz, loc / sz) << " [";
	s << lb << "]=>";
	message(s.str());
	lb = read_line();
}

void Editor::new_window_size(unsigned w, unsigned h)
{
	win_w = w;
	win_h = h;

	int upper_h = face.get_max_h(); // height of upper part is usually the face height
	if (upper_h < int(status_h + 1)) upper_h = status_h + 1; // but, will want to fit status and one map line
	if (upper_h > win_h - 1) upper_h = win_h - 1; // constrained by window height
	info_y = upper_h; // infomation line

	int text_h = h - upper_h - 1;
	if (text_h < 0) text_h = 0;
	text_shown = text_h > 0;

	int face_w = face.get_max_w();
	if (face_w > win_w) face_w = win_w;
	int right_w = win_w - face_w;
	status_shown = right_w >= int(map.get_width()) && upper_h >= int(status_h);

	int map_h = upper_h - status_h;
	int map_w = right_w;
	if (map_h < 0) map_h = 0;
	map_shown = map_h >= int(map.get_min_h()) && right_w >= int(map.get_width());

	// setting window sizes
	face.set_window(0, 0, face_w, upper_h);
	map.set_window(face_w, 0, map_w, map_h);
	text.set_window(0, upper_h + 1, win_w, text_h);
	status_x = face_w;

	// drawing things
	draw_status();
	if (! text_shown) {
		edit_text = false;
		text.set_guide(false);
	}
	face.do_track();
}

void Editor::prompt()
{
	if (edit_text) {
		text.set_cursor();
		return;
	}
	move_cursor_to(win_w - 1, win_h - 1);
}

void Editor::process_key(int c)
{
	if (edit_text) { // editing text, send commands & keys to text entry
		switch(c) {
		case nc::KeyUp:
			text.do_up();
			break;
		case nc::KeyLeft:
			text.do_left();
			break;
		case nc::KeyRight:
			text.do_right();
			break;
		case nc::KeyDown:
			text.do_down();
			break;
		case 'a' & 0x1f: // ctrl-a
		case nc::KeyHome:
			text.do_home();
			break;
		case 'c' & 0x1f: // ctrl-c, abort editing
			edit_text = false;
			text.set_text(get_text()); // restore text
			text.set_guide(false);
			break;
		case 'e' & 0x1f: // ctrl-e
		case nc::KeyEnd:
			text.do_end();
			break;
		case '\r':
			text.do_enter();
			break;
		case 'h' & 0x1f: // ctrl-h
		case nc::KeyBackspace:
			text.do_back();
			break;
		case nc::KeyDC:
		case 'd' & 0x1f: // ctrl-d
			text.do_delete();
			break;
		case nc::KeyPPage:
			text.do_page_up();
			break;
		case nc::KeyNPage:
			text.do_page_down();
			break;
		case '\t':
		case KeyBTab:
		case 24: // ctrl-x
			edit_text = false;
			if (text.get_text() != get_text()) set_text(text.get_text());
			text.set_guide(false);
			break;
		case 'g' & 0x1f: // ctrl-g
			text.set_guide(! text.get_guide());
			break;
		default:
			if (c >= 32 && c < 128) text.do_insert(c);
			else message("unknown key");
		}
		return;
	}
	if (face.get_active()) {
		bool face_key = true;
		switch (c) {
		case KeyUp: // move up
			face.do_up();
			face.do_track();
			break;
		case KeyDown: // move down
			face.do_down();
			face.do_track();
			break;
		case KeyRight: // move right
			face.do_right();
			face.do_track();
			break;
		case KeyLeft: // move left
			face.do_left();
			face.do_track();
			break;
		case ' ': // space
		case '\r': // enter
			face.do_enter();
			face.do_track();
			if (is_scoring()) draw_status(); // scoring could be changed
			break;
		default:
			face_key = false;
		}
		if (face_key) return;
	}
	if (can_init()) { // can initialize
		bool init_key = true;
		ostringstream s;
		int i;
		string n;
		switch (c) {
		case 'h': // setup handicap
			message("handicap (0,2-9):");
			i = get_key();
			if (i != Err) {
				int h = i - '0';
				if (h && (h < 2 || h > 9)) {
					message("invalid handicap");
					break;
				}
				s << h;
				draw_text(s.str());
				init_handi(h);
				draw_status();
			}
			break;
		case 'n': // new game
			init_create();
			message("new game created!");
			break;
		case 't': // toggle turn to play
			init_turn(go::flip(face.get_turn()));
			s << (face.get_turn() == go::White ? "white" : "black");
			s << " to play";
			draw_status();
			message(s.str());
			break;
		case 'z': // change size, odd number 5~19
			init_size(5 + (face.get_size() + 2 - 5) % 16);
			s << "board size changed to " << face.get_size();
			message(s.str());
			break;
		case 'l': // load game
			message("load sgf file: ");
			n = read_line();
			if (n.size()) {
				std::fstream f(n.c_str(), ios::in);
				if (! f) {
					s << "can not open file " << n << '!';
					message(s.str());
					break;
				}
				go::Game * g = go::read_sgf(f);
				attach(g);
				s << "file " << n << " loaded!";
				message(s.str());
			}
			break;
		default:
			init_key = false;
		}
		if (init_key) return;
	}
	if (doing_setup()) {
		bool setup_key = true;
		ostringstream s;
		int i;
		switch (c) {
		case 't': // toggle turn
			setup_turn(go::flip(face.get_turn()));
			s << (face.get_turn() == go::White ? "white" : "black");
			s << " to play";
			draw_status();
			message(s.str());
			break;
		case 'd': // done
			done_setup();
			break;
		case 'q': // cancel
			message("cancel setup?");
			i = get_key();
			if (i == 'y' || i == 'Y') {
				draw_char(i);
				cancel_setup();
				draw_status();
			}
			face.do_track();
			break;
		default:
			setup_key = false;
		}
		if (setup_key) return;
	}
	if (is_scoring()) {
		bool score_key = true;
		ostringstream s;
		int i;
		switch (c) {
		case 'd': // done
			done_score();
			draw_status();
			break;
		case 'q': // cancel
			message("cancel scoring?");
			i = get_key();
			if (i == 'y' || i == 'Y') {
				draw_char(i);
				cancel_score();
				draw_status();
			}
			face.do_track();
			break;
		default:
			score_key = false;
		}
		if (score_key) return;
	}
	if (can_play()) {
		bool play_key = true;
		switch (c) {
		case 'p':
			pass();
			break;
		case 's': // setup
			setup();
			draw_status();
			message("setup the board");
			break;
		case 'S': // score
			score();
			draw_status();
			message("score the board");
			break;
		case '\t':
		case KeyBTab:
			if (text_shown && is_attached()) {
				edit_text = true;
				text.set_guide(true);
				text.set_cursor();
			}
			break;
		case 'm': // change mark
			if (face.get_active()) {
				changer = ChangeMark;
				face.do_knock();
				face.do_track();
				draw_status(); // scoring could be changed
			}
			else play_key = false;
			break;
		case 'l': // change label
			if (face.get_active()) {
				changer = ChangeLabel;
				face.do_knock();
				face.do_track();
			}
			else play_key = false;
			break;
		default:
			play_key = false;
		}
		if (play_key) return;
	}
	if (can_crawl()) {
		bool crawl_key = true;
		switch (c) {
		case 'd' & 0x1f: // delete a node ctrl-d
			if (can_del_node()) del_node();
			else message("node can not be deleted!");
			break;
		case KeyCtrlUp:
		case 'u':
			up();
			break;
		case KeyCtrlDown:
		case 'd':
			down();
			break;
		case '/': // hop...
			hop();
			break;
		case KeyCtrlLeft:
		case ',':
		case '<':
			if (get_alt() > 0) {
				hop(get_alt() - 1);
			}
			break;
		case KeyCtrlRight:
		case '.':
		case '>':
			if (get_alt() < get_alt_num() - 1) {
				hop(get_alt() + 1);
			}
			break;
		default:
			crawl_key = false;
		}
		if (crawl_key) return;
	}
	ostringstream s;
	switch (c) {
	case 'l' & 0x1f: // refresh on ctrl-l
		reconfigure();
		break;
	case 'q': // quit on q
	//case 27:
		stop();
		break;
	default:
		s << "unrecognized key! (" << c << ')';
		msg(DBG_DEBUG) << s.str() << '\n';
		message(s.str());
		break;
	}
}

void Editor::draw_message(const std::string & msg)
{
	set_reverse();
	draw_text(0, info_y, msg);
	unset_reverse();
	clear_to_eol();
}

void Editor::clear_message()
{
	move_cursor_to(0, info_y);
	clear_to_eol();
}

Editor::Editor() :
	changer(ChangeNone),
	status_shown(false),
	map_shown(false),
	text_shown(false),
	edit_text(false)
{
	text.set_nc(this);
	set_face(& face);
	face.set_nc(this);
	face.set_active(true);
	set_map(& map);
	map.set_nc(this);

	init_size(9); // initialize with 9x9
}

Editor::~Editor()
{
}

void Editor::draw_status()
{
	if (! status_shown) return;
	string sep;
	if (can_init()) sep = "=[init]";
	else if (doing_setup()) sep = "=[setup]";
	else if (is_scoring()) sep = "=[score]";
	sep.resize(map.get_width(), '=');
	unsigned y = info_y - status_h;
	draw_text(status_x, y, sep);

	ostringstream s;
	s << "Move " << setw(3);
	if (is_attached()) s << left << get_move_num();
	else s << "---";
	y ++;
	draw_text(status_x, y, s.str());

	s.str("");
	s << "Turn: ";
	s << (face.get_turn() == go::Black ? '#' : 'O');
	y ++;
	draw_text(status_x, y, s.str());

	s.str("");
	s << "Alt: ";
	if (is_attached()) s << get_alt() + 1 << '/' << get_alt_num();
	y ++;
	draw_text(status_x, y, s.str());

	// scores
	s.str("");
	s << "B: ";
	if (is_attached()) s << setw(5) << get_b_score();
	else s << "     ";
	y ++;
	draw_text(status_x, y, s.str());
	s.str("");
	s << "W: ";
	if (is_attached()) s << setw(5) << get_w_score();
	else s << "     ";
	y ++;
	draw_text(status_x, y, s.str());
}
