/* ccgo: go/nc.cc
 *
 * Copyright (C) 2004 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "nc.hh"
extern "C" {
#include <ncurses.h>
}

#include <sstream>
#include <iomanip>
#include <iostream>

using namespace Nc;
using namespace std;

namespace {
	string loc_to_str(int x, int y)
	{
		char c = 'a' + x;
		if (c >= 'i') c ++;
		ostringstream s;
		s << c << y + 1;
		return s.str();
	}
}

void Face::new_size()
{
	size = get_size();
	editor->win_resize();
}

void Face::new_state(go::Loc loc)
{
}

void Face::new_mode()
{
}

void Face::new_info()
{
}

Face::Face(Editor * ed) :
	editor(ed),
	shown(false),
	active(false)
{
	set_editor(ed);
}

int Face::get_max_width() const
{
	return size * 3 + 5;
}

int Face::get_max_height() const
{
	return size + 2;
}

void Face::set_window(int x, int y, int w, int h)
{
	win_x = x;
	win_y = y;
	win_w = w;
	win_h = h;
	shown = w >= size * 3 + 5 && h >= size + 2;
	redraw();
}

void Face::set_active(bool act)
{
	active = act;
	if (act) show_cursor();
	else hide_cursor();
}

void Face::cursor_up()
{
}

void Face::cursor_down()
{
}

void Face::cursor_left()
{
}

void Face::cursor_right()
{
}

void Face::enter()
{
}


void Face::show_cursor()
{
}

void Face::hide_cursor()
{
}

void Face::redraw()
{
}
// members of class Entry
Entry::Entry() :
	shown(false),
	entx(0),
	enty(0),
	entw(0),
	enth(0),
	entl(0),
	entc(0),
	curl(0),
	curc(0),
	top(0),
	topc(0),
	show_guide(false)
{
	buf.push_back("");
}

Entry::~Entry()
{
}

void Entry::set_window(int x, int y, int width, int height)
{
	entx = x;
	enty = y;
	entw = width;
	enth = height;
	entl = 0;
	entc = 0;
	curl = 0;
	curc = 0;
	top = 0;
	topc = 0;
	shown = true;
	draw_buf();
}

void Entry::draw_buf()
{
	if (! shown) return;
	unsigned bl = top;
	unsigned bc = topc;
	for (int l = 0; l < enth; l ++) {
		int c = 0;
		move(enty + l, entx - 1);
		if (show_guide) {
			if (bc == 0) {
				if (bl < buf.size()) addch(' ');
				else addch('~');
			} else addch('>');
		} else addch(' ');
		if (bl < buf.size()) {
			string s = buf[bl].substr(bc, entw);
			addstr(s.c_str());
			if (int(s.length()) < entw) {
				bl ++;
				bc = 0;
				c += s.length();
			} else {
				bc += s.length();
				c = entw;
			}
		}
		while (c < entw) {
			addch(' ');
			c ++;
		}
	}
}

void Entry::set_cursor()
{
	if (! shown) return;
	// count lines
	entl = 0;
	for (int i = top; i < curl; i ++) {
		entl += buf[i].length() / entw;
		entl ++;
	}
	entl += curc / entw;
	entl -= topc / entw;
	entc = curc % entw;
	move(enty + entl, entx + entc);
}

bool Entry::process_key(int c)
{
	int i;
	bool need_redraw = false;
	switch(c) {
	case KEY_UP:
		if (curc >= entw) {
			curc -= entw;
			if (curl == top && curc < topc) {
				topc -= entw;
				need_redraw = true;
			}
		} else if (curl > 0) {
			curl --;
			curc += entw * (buf[curl].length() / entw);
			if (curc > (int) buf[curl].length()) {
				curc = buf[curl].length();
			}
			if (curl < top) {
				top = curl;
				need_redraw = true;
			}
		}
		break;
	case KEY_LEFT:
		if (curc == 0) break;
		curc --;
		if (curl == top && curc < topc) { // need to scroll?
			topc -= entw;
			need_redraw = true;
		}
		break;
	case KEY_RIGHT:
		if (curc >= (int) buf[curl].length()) break;
		curc ++;
		if (curc % entw == 0 && entl == enth - 1) { // need to scroll?
			topc += entw;
			if (topc > int(buf[top].length())) {
				topc = 0;
				top ++;
			}
			need_redraw = true;
		}
		break;
	case KEY_DOWN:
		curc += entw;
		if (curc >= (int) buf[curl].length()) {
			if (curl >= (int) buf.size() - 1) { // no more lines
				curc -= entw;
				break;
			}
			curc = curc % entw;
			curl ++;
			if (curc > (int) buf[curl].length()) {
				curc = buf[curl].length();
			}
		}
		if (entl == enth - 1) { // need to scroll
			topc += entw;
			if (topc > (int) buf[top].length()) {
				topc = 0;
				top ++;
			}
			need_redraw = true;
		}
		break;
	case 1: // control-a
	case KEY_HOME:
		curc = 0;
		if (curl == top && topc > 0) {
			topc = 0;
			need_redraw = true;
		}
		break;
	case 5: // control-e
	case KEY_END:
		i = buf[curl].length() / entw - curc / entw;
		if (i + entl >= enth) { // need to scroll
			i = i + entl - enth + 1;
			while (i > 0) {
				topc += entw;
				if (topc > (int) buf[top].length()) {
					top ++;
					topc = 0;
				}
				i --;
			}
			need_redraw = true;
		}
		curc = buf[curl].length();
		break;
	case '\r':
		buf.insert(buf.begin() + curl + 1, buf[curl].substr(curc));
		buf[curl].erase(curc);
		curl ++;
		curc = 0;
		if (entl == enth - 1) { // need to scrool
			topc += entw;
			if (topc > (int) buf[top].length()) {
				topc = 0;
				top ++;
			}
		}
		need_redraw = true;
		break;
	case 8: // control-h
	case KEY_BACKSPACE:
		if (curc > 0) {
			curc --;
			buf[curl].erase(curc, 1);
			if (curl == top && curc < topc) {
				topc --;
			}
			need_redraw = true;
		} else if (curl > 0) {
			curl --;
			curc = buf[curl].length();
			buf[curl].append(buf[curl + 1]);
			buf.erase(buf.begin() + curl + 1);
			if (curl < top) {
				top = curl;
				topc = entw * (curc / entw);
			}
			need_redraw = true;
		}
		break;
	case KEY_DC:
	case 4: // control-d
		if (curc < (int) buf[curl].length()) {
			buf[curl].erase(curc, 1);
			need_redraw = true;
		} else if (curl < (int) buf.size() - 1) {
			buf[curl].append(buf[curl + 1]);
			buf.erase(buf.begin() + curl + 1);
			need_redraw = true;
		}
		break;
	case KEY_PPAGE:
		i = enth - 2;
		while (i > 0) {
			if (topc > 0) topc -= entw;
			else if (top > 0) {
				top --;
				topc = entw * (buf[top].length() / entw);
			} else break;
			i --;
			entl ++;
			need_redraw = true;
		}
		while (entl >= enth) {
			if (curc >= entw) curc -= entw;
			else {
				curl --;
				curc = entw * (buf[curl].length() / entw) + curc % entw;
			}
			entl --;
		}
		if (curc > (int) buf[curl].length()) curc = buf[curl].length();
		break;
	case KEY_NPAGE:
		i = enth - 2;
		while (i > 0) {
			if (topc + entw <= (int) buf[top].length()) topc += entw;
			else if (top < (int) buf.size() - 1) {
				top ++;
				topc = 0;
			} else break;
			i --;
			need_redraw = true;
		}
		while (curl < top) {
			curl ++;
			curc = curc % entw;
		}
		while (curc < topc) curc += entw;
		if (curc > (int) buf[curl].length()) curc = buf[curl].length();
		break;
	default:
		if (c >= 32 && c < 128) { // normal key?
			char b[2];
			b[1] = 0;
			b[0] = c;
			buf[curl].insert(curc, b);
			curc ++;
			if (curc % entw == 0 && entl == enth - 1) { // need to scroll?
				topc += entw;
				if (topc > (int) buf[top].length()) {
					topc = 0;
					top ++;
				}
			}
			need_redraw = true;
			break;
		}
		return false;
	}
	if (need_redraw) draw_buf();
	return true;
}

string Entry::get_text()
{
	string t;
	for (vector<string>::const_iterator i = buf.begin(); i != buf.end(); i ++) {
		if (i != buf.begin()) t += '\n';
		t += * i;		
	}
	return t;
}

void Entry::set_text(const string & text)
{
	buf.clear();
	string::size_type i = 0;
	string::size_type j;
	while ((j = text.find('\n', i)) != string::npos) {
		buf.push_back(text.substr(i, j - i));
		i = j + 1;
	};
	buf.push_back(text.substr(i));
	entl = 0;
	entc = 0;
	curl = 0;
	curc = 0;
	top = 0;
	topc = 0;
}

void Entry::set_guide(bool guide_shown)
{
	show_guide = guide_shown;
	draw_buf();
}

// members of class Editor
void Editor::view_site(go::Loc loc)
{
}

void Editor::new_node()
{
}

int Editor::cnt = 0;

Editor::Editor() :
	face(this),
	map_shown(false),
	text_shown(false),
	edit_text(false)
{
	set_face(& face);

	if (cnt == 0) { // first instance, initialize ncurses
		initscr();
		keypad(stdscr, true);
		nonl();
		cbreak();
		noecho();
		if (has_colors()) {
			start_color();
			use_default_colors();
		}
	}
	cnt ++;
	win_resize();
}

Editor::~Editor()
{
	cnt --;
	if (cnt == 0) endwin();
}

void Editor::win_resize()
{
	getmaxyx(stdscr, win_h, win_w);

	map_shown = win_h >= map_min_h && win_w >= face.get_max_width() + map_width;
	text_shown = win_h > face.get_max_height() + 1;
	int f_w = map_shown ? win_w - map_width : win_w;
	int f_x;
	if (f_w >= face.get_max_width()) {
		f_x = (f_w - face.get_max_width()) / 2;
		f_w = face.get_max_width();
	}
	else f_x = 0;
	int f_h = win_h >= face.get_max_height() + 1 ? face.get_max_height() : win_h - 1;
	status_y = f_h + 1;

	::clear(); // clear whole screen

	face.set_window(f_x, 0, f_w, f_h);
	show_status_line();
	show_map();
	if (text_shown) text.set_window(1, f_h + 2, win_w - 1, win_h - f_h - 1);
	else edit_text = false;
}

bool Editor::process_key(int c)
{
	if (c == KEY_RESIZE) {
		win_resize();
		return true;
	}
	show_status_line();
	if (edit_text) { // editing text, send commands & keys to text entry
		switch (c) {
		case 3: // control-c, abort editing
			edit_text = false;
			text.set_text(get_text()); // restore text
			break;
		case '\t':
		case KEY_BTAB:
		case 19: // control-x, finish editing
			edit_text = false;
			if (text.get_text() != get_text()) set_text(text.get_text());
			break;
		default:
			if (! text.process_key(c)) message("unprocessed key!");
		}
		if (! edit_text) text.set_guide(false); // hide guide when not editing
		return true;
	}
	bool ret = true;
	bool node_change = false;
	switch (c) {
	case KEY_UP: // move up
		face.cursor_up();
		break;
	case KEY_DOWN: // move down
		face.cursor_down();
		break;
	case KEY_RIGHT: // move right
		face.cursor_right();
		break;
	case KEY_LEFT: // move left
		face.cursor_left();
		break;
	case '\r': // enter
		face.enter();
		break;
	case ('d' & 0x1f): // delete a node Ctrl-d
		del_node();
		break;
	case 'u':
		up();
		break;
	case 'd':
		down();
		break;
	case '/':
		hop();
		break;
	case 'h': // setup handicap
		if (can_init()) {
			message("handicap (0,2-9):");
			int i = getch();
			if (i != ERR) {
				int h = i - '0';
				if (h && (h < 2 || h > 9)) {
					message("invalid handicap");
					break;
				}
				init_handi(unsigned(h));
			}
		}
		else message("can do setup now!");
		break;
		/*
	case 'p':
		if (board_shown) {
			go::Node * n = make_move(go::MOVE_PASS);
			if (! n) {
				message("Pass not allowed");
				break;
			}
			where()->add_child(n);
			PathWalk::down(n);
			node_change = true;
		}
		break;
	case '\t':
	case KEY_BTAB:
		if (comment_shown) {
			edit_comment = true;
			text.set_guide(true);
		}
		break;
		*/
	case ERR:
		break;
	default:
		ret = false;
	}
	if (node_change) {
		// show_status();
		show_map();
		// show_comment();
		// for (std::set<int>::iterator i = ko.begin(); i != ko.end(); i ++) update_loc(* i);
		// ko = where()->get_ko();
		// for (std::set<int>::iterator i = ko.begin(); i != ko.end(); i ++) update_loc(* i);
	}
	return ret;
}

void Editor::loop()
{
	do {
		if (edit_text) text.set_cursor();
		else move(win_h - 1, 0);

		int c = getch();
		if (process_key(c)) continue;
		if (c == 'q') break;
	} while (true);
}

void Editor::show_status_line()
{
	ostringstream s;
	if (is_attached()) s << '[' << get_level() << ']';
	else s << "[]";
	move(status_y, 1);
	clrtoeol();
	mvaddstr(status_y, 1, s.str().c_str());
}

void Editor::show_map()
{
	if (! map_shown) return;
}

void Editor::message(const std::string & msg)
{
}
