/* ccgo: igs/win.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "../config.hh"
#include "../gettext.h"
#include "win.hh"
#include "tagged_message.hh"
#include "encode.hh"
#include "../settings.hh"
#include <gtkmm/box.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/stock.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/tooltips.h>
#include <sstream>

#define _(String) Glib::locale_to_utf8(gettext(String))

using namespace igs;

void Win::win_size_change(Gtk::Allocation & a)
{
	if (a.get_width() != my_width || a.get_height() != my_height) {
		my_width = a.get_width();
		my_height = a.get_height();
		go::settings.set_int("igs-win-width", my_width);
		go::settings.set_int("igs-win-height", my_height);
	}
}

void Win::finishing()
{
	// if (is_conn()) close();
	delete this;
}

void Win::line_input()
{
	std::string line = line_box.get_line();
	send_line(line);
}

void Win::players_toggle()
{
	if (players_button.get_active()) {
		player_list.show();
	} else {
		player_list.hide();
	}
}

void Win::hide_player_list()
{
	if (players_button.get_active()) players_button.set_active(false);
	win.present();
}

void Win::show_player_list()
{
	if (! players_button.get_active()) players_button.set_active(true);
}

void Win::select_player(const std::string & n)
{
	std::ostringstream s;
	s << "stats " << n;
	send_line(s.str());
}

void Win::refresh_player_list(const std::string & filter)
{
	std::ostringstream s;
	s << "who";
	if (filter != "") {
		s << " " << filter;
	}
	send_line(s.str());
}

void Win::refresh_game_list()
{
	send_line("games");
}

void Win::games_toggle()
{
	if (games_button.get_active()) {
		game_list.show_all();
	} else {
		game_list.hide();
	}
}

void Win::hide_game_list()
{
	if (games_button.get_active()) games_button.set_active(false);
	win.present();
}

void Win::do_bell_toggle()
{
	if (setting_toggle) return;
	send_line(std::string("toggle bell ") + (bell_toggle.get_active() ? "1" : "0"));
}

void Win::do_quiet_toggle()
{
	if (setting_toggle) return;
	send_line(std::string("toggle quiet ") + (quiet_toggle.get_active() ? "1" : "0"));
}

void Win::do_shout_toggle()
{
	if (setting_toggle) return;
	send_line(std::string("toggle shout ") + (shout_toggle.get_active() ? "1" : "0"));
}

void Win::do_automail_toggle()
{
	if (setting_toggle) return;
	send_line(std::string("toggle automail ") + (automail_toggle.get_active() ? "1" : "0"));
}

void Win::do_open_toggle()
{
	if (setting_toggle) return;
	send_line(std::string("toggle open ") + (open_toggle.get_active() ? "1" : "0"));
}

void Win::do_looking_toggle()
{
	if (setting_toggle) return;
	send_line(std::string("toggle looking ") + (looking_toggle.get_active() ? "1" : "0"));
}

void Win::do_kibitz_toggle()
{
	if (setting_toggle) return;
	send_line(std::string("toggle kibitz ") + (kibitz_toggle.get_active() ? "1" : "0"));
}

void Win::do_chatter_toggle()
{
	if (setting_toggle) return;
	send_line(std::string("toggle chatter ") + (chatter_toggle.get_active() ? "1" : "0"));
}

void Win::do_singlegame_toggle()
{
	if (setting_toggle) return;
	send_line(std::string("toggle singlegame ") + (singlegame_toggle.get_active() ? "1" : "0"));
}

void Win::miss_game(int g)
{
	std::ostringstream s;
	s << "games " << g;
	send_line(s.str());
	s.str("");
	s << "refresh " << g;
	send_line(s.str());
}

void Win::skip_move(int g)
{
	std::ostringstream s;
	s << "moves " << g;
	send_line(s.str());
}

void Win::unob_game(int g)
{
	std::ostringstream s;
	s << "unob " << g;
	send_line(s.str());
}

void Win::select_game(int g)
{
	std::ostringstream s;
	s << "ob " << g;
	send_line(s.str());
}

void Win::game_kibitz(const std::string & m, TalkType type, int g)
{
	std::ostringstream s;

	switch (type) {
	case TALK_CHATTER:
		s << "chatter " << g << ' ';
		break;
	case TALK_KIBITZ:
		s << "kibitz " << g << ' ';
		break;
	case TALK_SAY:
		s << "say ";
		break;
	case TALK_TELL:
		s << "tell #" << g << ' ';
		break;
	case TALK_RAW:
		break;
	}

	s << m;
	send_line(s.str());

	// output to my own window
	if (type == TALK_RAW) {
		s.str("");
		s << '[' << m << ']';
		game_list.msg_game(g, s.str());
	} else if (type != TALK_TELL) { // not necessary for tell
		s.str("");
		s << username << " : " << m;
		game_list.msg_game(g, s.str());
	}
}

void Win::send_handicap(unsigned h, int g)
{
	std::ostringstream s;
	s << "handicap " << h;
	send_line(s.str());
}

void Win::send_put(const go::Loc & l, unsigned t, int g)
{
	std::ostringstream s;
	s << l.to_string();
	if (go::settings.get_option(go::Settings::OPT_CLIENT_TIME_CONTROL)) {
		s << ' ' << (t ? t : 1); // min is 1 second
	}
	send_line(s.str());
}

void Win::send_pass(unsigned t, int g)
{
	std::ostringstream s;
	s << "pass";
	if (go::settings.get_option(go::Settings::OPT_CLIENT_TIME_CONTROL)) {
		s << ' ' << (t ? t : 1); // min is 1 second
	}
	send_line(s.str());
}

void Win::send_undo(int g)
{
	send_line("undo");
}

void Win::send_terri(const go::Loc & l, int g)
{
	send_line(l.to_string());
}

void Win::send_reset(int)
{
	std::cerr << "sending undo" << std::endl;
	send_line("undo");
}

void Win::send_done(int)
{
	send_line("done");
}

void Win::send_adjourn(int)
{
	send_line("adjourn");
}

void Win::send_resign(int)
{
	send_line("resign");
}

void Win::send_tell(const std::string & w, const std::string & m)
{
	std::ostringstream o;
	o << "tell " << w << " " << m;
	send_line(o.str());
}

void Win::send_match(const Match & m)
{
	std::ostringstream o;

	if (m.decline) { // declining a match
		o << "decline " << m.opponent;
		send_line(o.str());
		return;
	}

	if (m.get_automatch()) {
		o << "automatch " << m.opponent;
		send_line(o.str());
		return;
	}
	o << "match " << m.opponent;
	if (m.set_option) o << ' ' << (m.my_turn == go::TURN_BLACK ? 'B' : 'W') << ' ' << m.board_size << ' ' << m.time << ' ' << m.byo_time;
	send_line(o.str());
	match_opponent = m.opponent;
}

void Win::send_look(const std::string & s)
{
	std::ostringstream o;
	o << "look " << s;
	send_line(o.str());
}

void Win::send_load(const std::string & s)
{
	std::ostringstream o;
	o << "load " << s;
	send_line(o.str());
}

void Win::send_info(const std::string & s)
{
	std::ostringstream o;
	o << "info ";
	if (s.length()) o << s;
	else o << "<None>";
	send_line(o.str());
}

void Win::exit_igs()
{
	send_line("exit");
}

void Win::conn_lost()
{
// 	std::cerr << "conn lost" << std::endl;
	Control::conn_lost();
	finishing();
}

Win::Win() :
	players_button(_("Players")),
	games_button(_("Games")),
	bell_toggle(_("Bell") + "?"),
	quiet_toggle(_("Quiet") + "?"),
	shout_toggle(_("Shout") + "?"),
	automail_toggle(_("AutoMail") + "?"),
	open_toggle(_("Open") + "?"),
	looking_toggle(_("Looking") + "?"),
	kibitz_toggle(_("Kibitz") + "?"),
	chatter_toggle(_("Chatter") + "?"),
	singlegame_toggle(_("SingleGame") + "?")
{
// 	conn_lost.connect(slot(* this, & Win::finishing));
// 	std::cerr << "new Win" << std::endl;
	std::string ip = go::settings.get_image_path();
	if (Glib::file_test(ip + "/igs_icon.png", Glib::FILE_TEST_EXISTS)) {
		win.set_icon(Gdk::Pixbuf::create_from_file(ip + "/igs_icon.png"));
	}

	open();

	Gtk::Tooltips * tt = Gtk::manage(new Gtk::Tooltips);

	Gtk::HBox * hb = Gtk::manage(new Gtk::HBox);
	win.add(* hb);
	Gtk::VBox * vb = Gtk::manage(new Gtk::VBox);
	hb->pack_start(* vb);
	Gtk::ScrolledWindow * sw = Gtk::manage(new Gtk::ScrolledWindow);
	vb->pack_start(* sw);
	sw->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_ALWAYS);
	sw->add(text_port);
	vaj = sw->get_vadjustment();
	text_port.set_editable(false);
	text_port.set_cursor_visible(false);
	text_port.set_wrap_mode(Gtk::WRAP_WORD);
	Gtk::HBox * hb1 = Gtk::manage(new Gtk::HBox);
	vb->pack_start(* hb1, Gtk::PACK_SHRINK);
	hb1->pack_start(prompt, Gtk::PACK_SHRINK);
	hb1->pack_start(line_box);
	line_box.signal_activate().connect(mem_fun(* this, & Win::line_input));
	line_box.set_sensitive(false);

	tt->set_tip(line_box, _("Enter IGS commands here"));

	vb = Gtk::manage(new Gtk::VBox);
	hb->pack_start(* vb, Gtk::PACK_SHRINK);

	lists_box = Gtk::manage(new Gtk::VBox);
	vb->pack_start(* lists_box, Gtk::PACK_SHRINK);

	lists_box->pack_start(players_button, Gtk::PACK_SHRINK);
	players_button.signal_clicked().connect(mem_fun(* this, & Win::players_toggle));
	player_list.signal_hide().connect(mem_fun(* this, & Win::hide_player_list));
	player_list.signal_show().connect(mem_fun(* this, & Win::show_player_list));

	tt->set_tip(players_button, _("Show/Hide the player list"));

	lists_box->pack_start(games_button, Gtk::PACK_SHRINK);
	games_button.signal_clicked().connect(mem_fun(* this, & Win::games_toggle));
	game_list.signal_hide().connect(mem_fun(* this, & Win::hide_game_list));

	tt->set_tip(games_button, _("Show/Hide the game list"));

	lists_box->set_sensitive(false);

	// toggles
	vb->pack_start(* Gtk::manage(new Gtk::Label(_("My states") + ":")), Gtk::PACK_SHRINK);
	toggles_box = Gtk::manage(new Gtk::VButtonBox);
	vb->pack_start(* toggles_box, Gtk::PACK_SHRINK);

	toggles_box->pack_start(bell_toggle, Gtk::PACK_SHRINK);
	toggles_box->pack_start(quiet_toggle, Gtk::PACK_SHRINK);
	toggles_box->pack_start(shout_toggle, Gtk::PACK_SHRINK);
	toggles_box->pack_start(automail_toggle, Gtk::PACK_SHRINK);
	toggles_box->pack_start(open_toggle, Gtk::PACK_SHRINK);
	toggles_box->pack_start(looking_toggle, Gtk::PACK_SHRINK);
	toggles_box->pack_start(kibitz_toggle, Gtk::PACK_SHRINK);
	toggles_box->pack_start(chatter_toggle, Gtk::PACK_SHRINK);
// 	toggles_box->pack_start(singlegame_toggle, Gtk::PACK_SHRINK);

	toggles_box->set_sensitive(false);

	setting_toggle = false;
	bell_toggle.set_inconsistent();
	quiet_toggle.set_inconsistent();
	shout_toggle.set_inconsistent();
	automail_toggle.set_inconsistent();
	open_toggle.set_inconsistent();
	looking_toggle.set_inconsistent();
	kibitz_toggle.set_inconsistent();
	chatter_toggle.set_inconsistent();

	bell_toggle.signal_toggled().connect(mem_fun(* this, & Win::do_bell_toggle));
	quiet_toggle.signal_toggled().connect(mem_fun(* this, & Win::do_quiet_toggle));
	shout_toggle.signal_toggled().connect(mem_fun(* this, & Win::do_shout_toggle));
	automail_toggle.signal_toggled().connect(mem_fun(* this, & Win::do_automail_toggle));
	open_toggle.signal_toggled().connect(mem_fun(* this, & Win::do_open_toggle));
	looking_toggle.signal_toggled().connect(mem_fun(* this, & Win::do_looking_toggle));
	kibitz_toggle.signal_toggled().connect(mem_fun(* this, & Win::do_kibitz_toggle));
	chatter_toggle.signal_toggled().connect(mem_fun(* this, & Win::do_chatter_toggle));
// 	singlegame_toggle.signal_toggled().connect(mem_fun(* this, & Win::do_singlegame_toggle));

	tt->set_tip(bell_toggle, _("Toggle if to receive beeping"));
	tt->set_tip(quiet_toggle, _("Toggle if to receive unrelated login and game messages"));
	tt->set_tip(shout_toggle, _("Toggle if to receive shout messages"));
	tt->set_tip(automail_toggle, _("Toggle if to receive game records in email"));
	tt->set_tip(open_toggle, _("Toggle if to receive match requests"));
	tt->set_tip(looking_toggle, _("Toggle to show if I am looking for a match"));
	tt->set_tip(kibitz_toggle, _("Toggle if to receive kibitz messages"));
	tt->set_tip(chatter_toggle, _("Toggle if to receive chatter messages"));

	Gtk::Button * bt = Gtk::manage(new Gtk::Button(Gtk::Stock::CLOSE));
	vb->pack_end(* bt, Gtk::PACK_SHRINK);
	bt->signal_clicked().connect(mem_fun(* this, & Win::exit_igs));
	win.signal_hide().connect(mem_fun(* this, & Win::exit_igs));

	player_list.player_selected.connect(mem_fun(* this, & Win::select_player));
	player_list.refresh_list.connect(mem_fun(* this, & Win::refresh_player_list));
	player_list.ask_results.connect(mem_fun(* this, & Win::player_ask_results));
	player_list.ask_odds.connect(mem_fun(* this, & Win::player_ask_odds));
	player_list.ask_stored.connect(mem_fun(* this, & Win::player_ask_stored));
	player_list.select_game.connect(mem_fun(game_list, & GameList::wish_select_game));
	player_list.talk_to_player.connect(mem_fun(* this, & Win::send_tell));
	player_list.enter_match.connect(mem_fun(* this, & Win::send_match));

	player_list.lookup_stored.connect(mem_fun(* this, & Win::send_look));
	player_list.load_stored.connect(mem_fun(* this, & Win::send_load));
	player_list.change_info.connect(mem_fun(* this, & Win::send_info));

	game_list.miss_game.connect(mem_fun(* this, & Win::miss_game));
	game_list.skip_move.connect(mem_fun(* this, & Win::skip_move));
	game_list.close_game.connect(mem_fun(* this, & Win::unob_game));
	game_list.game_selected.connect(mem_fun(* this, & Win::select_game));
	game_list.refresh_list.connect(mem_fun(* this, & Win::refresh_game_list));

	game_list.game_input_kibitz.connect(mem_fun(* this, & Win::game_kibitz));

	game_list.game_input_handicap.connect(mem_fun(* this, & Win::send_handicap));
	game_list.game_input_put.connect(mem_fun(* this, & Win::send_put));
	game_list.game_input_pass.connect(mem_fun(* this, & Win::send_pass));
	game_list.game_input_undo.connect(mem_fun(* this, & Win::send_undo));
	game_list.game_input_terri.connect(mem_fun(* this, & Win::send_terri));
	game_list.game_input_reset.connect(mem_fun(* this, & Win::send_reset));
	game_list.game_input_done.connect(mem_fun(* this, & Win::send_done));

	game_list.game_input_adjourn.connect(mem_fun(* this, & Win::send_adjourn));
	game_list.game_input_resign.connect(mem_fun(* this, & Win::send_resign));

	my_width = go::settings.get_int("igs-win-width");
	if (my_width <= 0) my_width = 300;
	my_height = go::settings.get_int("igs-win-height");
	if (my_height <= 0) my_height = 200;
	win.resize(my_width, my_height);
	win.signal_size_allocate().connect(mem_fun(* this, & Win::win_size_change));

	win.show_all();
}

Win::~Win()
{
	if (is_conn()) close();
}

void Win::cancel_conn()
{
	finishing();
}

void Win::client_ready()
{
	lists_box->set_sensitive(true);
	toggles_box->set_sensitive(true);
	line_box.set_sensitive(true);
	add_message(std::string(_("Logged in as ")) + username);
	send_line("stats"); // to obtain my own states
}

void Win::set_prompt(PromptType p)
{
	Control::set_prompt(p);
	switch (p) {
	case PROMPT_NORMAL:
		prompt.set_text(_("IGS Command:"));
		break;
	case PROMPT_MATCH:
		prompt.set_text(_("IGS (Matching):"));
		break;
	case PROMPT_SCORE:
		prompt.set_text(_("IGS (Scoring):"));
		break;
	case PROMPT_OBSERVE:
		prompt.set_text(_("IGS (Observing):"));
		break;
	}
}

void Win::set_username(const std::string & n)
{
	username = n;
	win.set_title(std::string("igs(") + n + ")");
	game_list.set_name(n);
	player_list.set_name(n);
}


void Win::do_beep()
{
	gdk_beep();
}

void Win::add_message(const std::string & msg)
{
	if (! msg.length()) return;
	Glib::ustring u = msg;
	if (! u.validate()) {
		u = auto_convert(msg);
	}
	bool bot = (vaj->get_value()  + vaj->get_page_size() >= vaj->get_upper());
	text_port.get_buffer()->insert(text_port.get_buffer()->end(), u);
	text_port.get_buffer()->insert(text_port.get_buffer()->end(), "\n");
	if (bot) {
		// std::cerr << "scroll to bottom" << std::endl;
		text_port.scroll_to_mark(text_port.get_buffer()->create_mark("end", text_port.get_buffer()->end()), 0); // any easier way?
	}
}

void Win::add_tagged_message(const std::string & tag, const std::string & text)
{
	// std::cerr << "adding tagged message" << std::endl;
	bool bot = (vaj->get_value()  + vaj->get_page_size() >= vaj->get_upper());
	Gtk::TextIter iter = text_port.get_buffer()->end();
	Glib::RefPtr<Gtk::TextChildAnchor> anchor = text_port.get_buffer()->create_child_anchor(iter);
	TaggedMessage * tm = Gtk::manage(new TaggedMessage(tag, text));
	text_port.add_child_at_anchor(* tm, anchor);
	tm->show();
	// text_port.get_buffer()->insert(text_port.get_buffer()->end(), "\n");
	if (bot) {
		// std::cerr << "scroll to bottom" << std::endl;
		text_port.scroll_to_mark(text_port.get_buffer()->create_mark("end", text_port.get_buffer()->end()), 0); // any easier way?
	}
}

void Win::player_update(const Player & p)
{
	player_list.update(p);
}

void Win::player_stats(const Player & p)
{
	player_list.stats(p);
}

void Win::player_odds(const Odds & o)
{
	player_list.odds(o);
}

void Win::player_end_update()
{
	player_list.end_update();
}

void Win::player_ask_results(const std::string & n)
{
	std::ostringstream s;
	s << "results";
	if (n != "") s << " -" << n;
	send_line(s.str());
}

void Win::player_ask_odds(const std::string & n)
{
	std::ostringstream s;
	s << "prob " << n;
	send_line(s.str());
}

void Win::player_ask_stored(const std::string & n)
{
	std::ostringstream s;
	s << "stored";
	if (n != "") s << " -" << n;
	else s << " -" << username;
	send_line(s.str());
}

void Win::game_result(const std::vector<Result> & r)
{
	player_list.add_result(r);
}

void Win::stored_game(const std::vector<Storage> & d)
{
	player_list.add_stored(d);
}

void Win::game_update(const Game & g)
{
	game_list.update(g);
}

void Win::game_end_update()
{
	game_list.end_update();
}

void Win::game_move(go::Move * mv, int g)
{
	game_list.add_move(g, mv);
}

void Win::game_undo(int g)
{
	game_list.undo_move(g);
}

void Win::game_time(int g, int wt, int wm, int bt, int bm)
{
	game_list.sync_time(g, wt, wm, bt, bm);
}

void Win::game_title(int g, const std::string & t)
{
	game_list.add_title(g, t);
}

void Win::ob_remove(int g)
{
	game_list.unob_game(g);
}

void Win::game_finish(int g, const std::string & s)
{
	game_list.end_game(g, s);
}

void Win::match_score(int g)
{
	game_list.game_score(g);
}

void Win::match_restore(int g)
{
	game_list.game_restore(g);
}

void Win::match_remove(const go::Loc & l, int g)
{
	game_list.game_remove(g, l);
}

void Win::match_adjourn(int g)
{
	game_list.game_adjourn(g);
}

void Win::match_finish(const std::string & s, int g)
{
	game_list.end_game(g, s);
}

void Win::match_message(const std::string & s, int g)
{
	game_list.msg_game(g, s);
}

void Win::match_resign(const std::string & s)
{
	game_list.match_resign(s);
}

void Win::match_timeup(const std::string & s)
{
	game_list.match_timeup(s);
}

void Win::remove_match(const std::string & w, const std::string & b)
{
	game_list.remove_match(w, b);
}

void Win::player_update(const std::string &, const std::string &)
{
}

void Win::player_remove(const std::string &)
{
}

void Win::message_sys(const std::string & s)
{
	add_message(s);
}

void Win::message_tell(const std::string & s)
{
	add_message(s);
	game_list.msg_game(0, s);
}

void Win::message_tell(const std::string & w, const std::string & m)
{
	player_list.msg_tell(w, m);
}

void Win::message_yell(const std::string & s)
{
	add_message(s);
}

void Win::message_kibitz(int g, const std::string & s)
{
	game_list.msg_game(g, s);
}

void Win::message_misc(const std::string & s)
{
	add_message(s);
	win.present(); // could be important
}

void Win::match_request(const Match & m)
{
	player_list.match_request(m);
}

void Win::match_create(int g, const std::string & w)
{
	player_list.match_create(w);
}

void Win::match_decline(const std::string & w)
{
	player_list.match_decline(w);
}

void Win::match_not_online(const std::string & w)
{
	player_list.match_not_online(w);
}

void Win::match_not_open()
{
	player_list.match_not_open(match_opponent);
}

void Win::match_in_match(const std::string & w)
{
	player_list.match_in_match(w);
}

void Win::game_board(const Board & b)
{
	game_list.game_board(b);
}

void Win::toggle_state(ToggleType type, bool state)
{
	setting_toggle = true;
	switch (type) {
	case TOGGLE_BELL:
		if (bell_toggle.get_active() != state) bell_toggle.set_active(state);
		if (state) bell_toggle.set_label(_("Bell") + "+");
		else bell_toggle.set_label(_("Bell") + "-");
		if (bell_toggle.get_inconsistent()) bell_toggle.set_inconsistent(false);
		break;
	case TOGGLE_QUIET:
		if (quiet_toggle.get_active() != state) quiet_toggle.set_active(state);
		if (state) quiet_toggle.set_label(_("Quiet") + "+");
		else quiet_toggle.set_label(_("Quiet") + "-");
		if (quiet_toggle.get_inconsistent()) quiet_toggle.set_inconsistent(false);
		break;
	case TOGGLE_SHOUT:
		if (shout_toggle.get_active() != state) shout_toggle.set_active(state);
		if (state) shout_toggle.set_label(_("Shout") + "+");
		else shout_toggle.set_label(_("Shout") + "-");
		if (shout_toggle.get_inconsistent()) shout_toggle.set_inconsistent(false);
		break;
	case TOGGLE_AUTOMAIL:
		if (automail_toggle.get_active() != state) automail_toggle.set_active(state);
		if (state) automail_toggle.set_label(_("AutoMail") + "+");
		else automail_toggle.set_label(_("AutoMail") + "-");
		if (automail_toggle.get_inconsistent()) automail_toggle.set_inconsistent(false);
		break;
	case TOGGLE_OPEN:
		if (open_toggle.get_active() != state) open_toggle.set_active(state);
		if (state) open_toggle.set_label(_("Open") + "+");
		else open_toggle.set_label(_("Open") + "-");
		if (open_toggle.get_inconsistent()) open_toggle.set_inconsistent(false);
		break;
	case TOGGLE_LOOKING:
		if (looking_toggle.get_active() != state) looking_toggle.set_active(state);
		if (state) looking_toggle.set_label(_("Looking") + "+");
		else looking_toggle.set_label(_("Looking") + "-");
		if (looking_toggle.get_inconsistent()) looking_toggle.set_inconsistent(false);
		break;
	case TOGGLE_KIBITZ:
		if (kibitz_toggle.get_active() != state) kibitz_toggle.set_active(state);
		if (state) kibitz_toggle.set_label(_("Kibitz") + "+");
		else kibitz_toggle.set_label(_("Kibitz") + "-");
		if (kibitz_toggle.get_inconsistent()) kibitz_toggle.set_inconsistent(false);
		break;
	case TOGGLE_CHATTER:
		if (chatter_toggle.get_active() != state) chatter_toggle.set_active(state);
		if (state) chatter_toggle.set_label(_("Chatter") + "+");
		else chatter_toggle.set_label(_("Chatter") + "-");
		if (chatter_toggle.get_inconsistent()) chatter_toggle.set_inconsistent(false);
		break;
	case TOGGLE_SINGLEGAME:
		// don't allow multiple game for now...
		// TODO: multiple game support
		if (! state) {
			send_line("toggle singlegame 1");
		}
// 		singlegame_toggle.set_active(state);
// 		if (state) singlegame_toggle.set_label("SingleGame+");
// 		else singlegame_toggle.set_label("SingleGame-");
		break;
	case TOGGLE_VERBOSE:
		if (state) { // verbose = true -> should fix it!
			send_line("toggle verbose 0");
		}
		break;
	case TOGGLE_CLIENT:
		if (! state) { // client = false -> should fix it!
			send_line("toggle client 1");
		}
		break;
	}
	setting_toggle = false;
}

void Win::file_text(const std::string & text)
{
	add_tagged_message(_("File"), text);
	add_message(".");
}
