/* ccgo: igs/sock.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "sock.hh"
#include <iostream>
extern "C" {
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>
#include <netdb.h>
}
#include <cerrno>

using namespace igs;
using namespace std;

bool Sock::grab_line(Glib::IOCondition)
{
	grab_action = false;
	grab_line_connection = Glib::signal_idle().connect(mem_fun(* this, & Sock::read_buffer));
	return false;
}

bool Sock::read_buffer()
{
	char b[256];
	int s = 0;
	s = recv(sock_id, b, 128, 0); // only process 128 chars at a time, this slows the processing but makes the UI more responsive.
	if (s > 0) {
		idle_time = 0;
		char * sb = b;
		int p = 0;
		int i;
		for (i = 0; p < s; i ++) {
			if (iscntrl(sb[i])) { // a control character
				line_buffer.append(sb, i);
				if (sb[i] == '\n' || sb[i] == '\r') {
					if (line_buffer.length()) { // drop empty lines
						recv_line(line_buffer);
						line_buffer = "";
					}
				} else if (sb[i] == 1) { // control sequence for telnet, drop
					line_buffer = "";
				} else { // append others...
					line_buffer.append(1, sb[i]);
				}
				sb += i + 1;
				i = - 1;
			}
			p ++;
		}
		line_buffer.append(sb, i); // put the rest in the line buffer
		grab_action = true;
		return true;
	}
	if (check_prompt(line_buffer)) {
		recv_line(line_buffer);
		line_buffer = "";
	}
	if (s == 0) { // socket closed on the other end
		close(sock_id);
		sock_id = 0;
		connect_ok = false;
		time_up_connection.disconnect();
		grab_line_connection.disconnect();
		conn_lost();
		return false;
	}
	// reconnect to grab_line
	grab_line_connection = Glib::signal_io().connect(mem_fun(* this, & Sock::grab_line), sock_id, Glib::IO_IN | Glib::IO_HUP);
	return false;
}

bool Sock::time_up()
{
	if (idle_time < 30000) {
		idle_time += 100;
		if (idle_time == 3000 && line_buffer.length()) {
			cerr << "timeout empty buffer" << endl;
			recv_line(line_buffer);
			line_buffer = "";
		}
	}
	return true;
}

Sock::Sock(const string & n, unsigned short p) :
	sock_id(- 1)
{
	open_sock(n, p);
}

Sock::~Sock()
{
	if (sock_id >= 0) {
		grab_line_connection.disconnect();
		time_up_connection.disconnect();
	}
}

bool Sock::open_sock(const string & n, unsigned short p)
{
	connect_ok = false;
	hostent * he;
	he = gethostbyname(n.c_str());
	sockaddr_in se;

	char * sse = (char *) & se;
	for (unsigned i = 0; i < sizeof(se); i ++) sse[i] = 0;
	se.sin_family = AF_INET;
	se.sin_port = htons(p);
	se.sin_addr = * (in_addr *)(he->h_addr_list[0]);

	sock_id = socket(AF_INET, SOCK_STREAM, 0);
	int r = connect(sock_id, (struct sockaddr *) & se, sizeof(se));
	if (r != 0) return false;

	// set nonblock on reading child
	fcntl(sock_id, F_SETFL, O_NONBLOCK);
	grab_line_connection = Glib::signal_io().connect(mem_fun(* this, & Sock::grab_line), sock_id, Glib::IO_IN | Glib::IO_HUP);
	time_up_connection = Glib::signal_timeout().connect(mem_fun(* this, & Sock::time_up), 100);
	connect_ok = true;
	return true;
}

void Sock::close_sock()
{
	if (sock_id >= 0) shutdown(sock_id, SHUT_RDWR);
}

void Sock::send_line(const string & s)
{
	std::string d = s + "\n";
	send(sock_id, d.c_str(), d.length(), 0);
}
