/* ccgo: igs/show.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "../config.hh"
#include "../gettext.h"
#include "show.hh"
#include "../settings.hh"
#include <gtkmm/stock.h>

#define _(String) Glib::locale_to_utf8(gettext(String))

using namespace igs;

TalkSelect::TalkSelect() :
	talk_chatter(_("Chatter")),
	talk_kibitz(_("Kibitz")),
	talk_say(_("Say")),
	talk_tell(_("Tell")),
	talk_raw(_("IGS Command:"))
{
	Gtk::Menu * m = Gtk::manage(new Gtk::Menu);
	m->append(talk_chatter);
	m->append(talk_kibitz);
	m->append(talk_say);
	m->append(talk_tell);
	m->append(talk_raw);
	set_menu(* m);
}

TalkType TalkSelect::get_type()
{
	TalkType t;

	switch (get_history()) {
	case 0:
		t = TALK_CHATTER;
		break;
	case 1:
		t = TALK_KIBITZ;
		break;
	case 2:
		t = TALK_SAY;
		break;
	case 3:
		t = TALK_TELL;
		break;
	case 4:
		t = TALK_RAW;
		break;
	default: // no possible
		t = TALK_CHATTER;
	}
	return t;
}

void TalkSelect::disable_talk(TalkType t)
{
	switch (t) {
	case TALK_CHATTER:
		talk_chatter.set_sensitive(false);
		break;
	case TALK_KIBITZ:
		talk_kibitz.set_sensitive(false);
		break;
	case TALK_SAY:
		talk_say.set_sensitive(false);
		break;
	case TALK_TELL:
		talk_tell.set_sensitive(false);
		break;
	case TALK_RAW:
		talk_raw.set_sensitive(false);
	}
}
//////////////////////////////////////////////////////////////////////////////////////////////
void ResignDialog::on_response(int r)
{
	switch(r) {
	case 0:
		hide();
		break;
	case 1:
		confirmed.emit();
		break;
	default:
		break;
	}
}

ResignDialog::ResignDialog()
{
	get_vbox()->pack_start(* Gtk::manage(new Gtk::Label(_("Resign the Game"))), Gtk::PACK_SHRINK);
	add_button(Gtk::Stock::CANCEL, 0);
	add_button(Gtk::Stock::OK, 1);
	show_all_children();
}
//////////////////////////////////////////////////////////////////////////////////////////////
void Show::line_input()
{
	input_line(line_box.get_line(), talk_select.get_type());
}

void Show::on_hide()
{
	if (! close_button.is_sensitive()) {
		show();
		return;
	}
	show_close(this);
	delete this;
}

void Show::on_size_allocate(Gtk::Allocation & a)
{
	Gtk::Widget::on_size_allocate(a);
	if (a.get_width() != my_width || a.get_height() != my_height) {
		my_width = a.get_width();
		my_height = a.get_height();
		go::settings.set_int("igs-show-width", my_width);
		go::settings.set_int("igs-show-height", my_height);
	}
}

bool Show::handle_move(Gtk::ScrollType)
{
// 	std::cerr << "handle moved = " << vpaned.get_position() << std::endl;
	return true;
}

void Show::resign_hit()
{
	resign_dialog.present();
}

Show::Show(unsigned z) :
	part(z, 5.5, 0),
	adjourn_button(_("Adjourn")),
	resign_button(_("Resign")),
	close_button(Gtk::Stock::CLOSE)
{
	std::string ip = go::settings.get_image_path();
	if (Glib::file_test(ip + "/observing_icon.png", Glib::FILE_TEST_EXISTS)) {
		set_icon(Gdk::Pixbuf::create_from_file(ip + "/observing_icon.png"));
	}

	add(vpaned);
	Gtk::VBox * vb = Gtk::manage(new Gtk::VBox);

	vb->pack_start(header_text, Gtk::PACK_SHRINK);
	header_text.set_size_request(0, -1);

	Gtk::HBox * hb = Gtk::manage(new Gtk::HBox);
	vb->pack_start(* hb);
	hb->pack_start(part.get_board_area());

	Gtk::VBox * vb2 = Gtk::manage(new Gtk::VBox);
	hb->pack_end(* vb2, Gtk::PACK_SHRINK);
	vb2->pack_start(part.get_button_area());
	close_button.signal_clicked().connect(mem_fun(* this, & Window::hide));
	vb2->pack_end(close_button, Gtk::PACK_SHRINK);

	vb2->pack_end(resign_button, Gtk::PACK_SHRINK);
	vb2->pack_end(adjourn_button, Gtk::PACK_SHRINK);

	hb = Gtk::manage(new Gtk::HBox);
	vb->pack_end(* hb, Gtk::PACK_SHRINK);
	hb->pack_start(part.get_modifi_area(), Gtk::PACK_SHRINK);
	hb->pack_end(part.get_label_area(), Gtk::PACK_SHRINK);
	vpaned.pack1(* vb);
	Gtk::ScrolledWindow * sw = Gtk::manage(new Gtk::ScrolledWindow);
	sw->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_ALWAYS);
	sw->add(message_box);
	vb = Gtk::manage(new Gtk::VBox);
	vpaned.pack2(* vb);
	vb->pack_start(* sw);
	hb = Gtk::manage(new Gtk::HBox);
	vb->pack_end(* hb, Gtk::PACK_SHRINK);
	hb->pack_start(talk_select, Gtk::PACK_SHRINK);
	hb->pack_start(line_box);
	line_box.signal_activate().connect(mem_fun(* this, & Show::line_input));
	vaj = sw->get_vadjustment();

	message_box.set_editable(false);
	message_box.set_cursor_visible(false);
	message_box.set_wrap_mode(Gtk::WRAP_WORD);

	my_width = go::settings.get_int("igs-show-width");
	if (my_width <= 0) my_width = 320;
	my_height = go::settings.get_int("igs-show-height");
	if (my_height <= 0) my_height = 360;
	resize(my_width, my_height);
	my_vpaned_position = my_height - 30;
	vpaned.set_position(my_vpaned_position);
	// vpaned.signal_move_handle().connect(mem_fun(* this, & Show::handle_move));
	show_all();
	part.set_no_play();
	part.set_hold_action();
	header_text.hide();

	adjourn_button.hide();
	resign_button.hide();

	adjourn_button.signal_clicked().connect(input_adjourn.make_slot());
	resign_button.signal_clicked().connect(mem_fun(* this, & Show::resign_hit));
	resign_dialog.confirmed.connect(input_resign.make_slot());
}

Show::~Show()
{
}

void Show::set_play(bool w, bool b)
{
	part.set_no_play(! (w || b));
	part.set_players(w ? 0 : this, b ? 0 : this);
	if (w || b) {
		std::string ip = go::settings.get_image_path();
		if (w && b) {
			if (Glib::file_test(ip + "/teaching_icon.png", Glib::FILE_TEST_EXISTS)) {
				set_icon(Gdk::Pixbuf::create_from_file(ip + "/teaching_icon.png"));
			}
		} else {
			if (Glib::file_test(ip + "/matching_icon.png", Glib::FILE_TEST_EXISTS)) {
				set_icon(Gdk::Pixbuf::create_from_file(ip + "/matching_icon.png"));
			}
		}
		part.held_handicap.connect(input_handicap.make_slot());
		part.held_put.connect(input_put.make_slot());
		part.held_pass.connect(input_pass.make_slot());
		part.held_undo.connect(input_undo.make_slot());
		part.held_terri.connect(input_terri.make_slot());
		part.held_reset.connect(input_reset.make_slot());
		part.held_done.connect(input_done.make_slot());
		talk_select.set_history(TALK_SAY);
		if (w && b) talk_select.disable_talk(TALK_TELL);
		adjourn_button.show();
		resign_button.show();
		close_button.set_sensitive(false);
	} else { // no opponent, not playing
		talk_select.disable_talk(TALK_SAY);
	}
}

void Show::set_komi(double k)
{
	part.set_komi(k);
}

int Show::move_num()
{
	return part.get_game()->get_move_num();
}

bool Show::add_move(go::Move * m)
{
	return part.add_move(m);
}

bool Show::undo_move()
{
	return part.undo_move();
}

void Show::hang_up(const std::string & s)
{
	add_message(s);
	part.stop_game();
	line_box.set_sensitive(false);
	close_button.set_sensitive(true);
	adjourn_button.set_sensitive(false);
	resign_button.set_sensitive(false);
	resign_dialog.hide();
}

void Show::start_time(int t, int b, int m)
{
	part.start_time(t, b, m);
}

void Show::sync_time(int wt, int wm, int bt, int bm)
{
	part.time_update(wt, wm, bt, bm);
}

void Show::set_header(const std::string & s)
{
	header_text.set_text(s);
	header_text.show();
}

void Show::add_message(const std::string & msg)
{
	if (! msg.length()) return;
	bool bot = (vaj->get_value()  + vaj->get_page_size() >= vaj->get_upper());
	message_box.get_buffer()->insert(message_box.get_buffer()->end(), msg);
	message_box.get_buffer()->insert(message_box.get_buffer()->end(), "\n");
	if (bot) {
		message_box.scroll_to_mark(message_box.get_buffer()->create_mark("end", message_box.get_buffer()->end()), 0);
	}
}

void Show::show_score()
{
	part.enter_score();
}

void Show::show_restore()
{
	part.restore_board();
}

void Show::show_terri(const go::Loc & l)
{
	part.terri_request(l);
}

// void Show::show_adjourn()
// {
// 	hang_up("game adjourned");
// }

void Show::show_score_board(const Board & b)
{
	go::Terri * t = new go::Terri[b.board_size * b.board_size];
	bool is_score = false;
	for (int i = 0; i < b.board_size * b.board_size; i ++) switch (b.state[i]) {
	case 2: // not a score board!
	case 6:
		is_score = false;
		goto out;
	case 4:
		t[i] = go::TERRI_WHITE;
		is_score = true;
		break;
	case 5:
		t[i] = go::TERRI_BLACK;
		is_score = true;
		break;
	default:
		t[i] = go::TERRI_NEUTRAL;
	}
 out:
	if (is_score) part.add_score(t); // don't delete t
	else delete t;
}
