/* ccgo: igs/player_list.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "../config.hh"
#include "../gettext.h"
#include "player_list.hh"
#include "encode.hh"
#include "../settings.hh"
#include <gtkmm/scrolledwindow.h>
#include <glibmm/listhandle.h>
#include <gtkmm/treeviewcolumn.h>
#include <gtkmm/box.h>
#include <gtkmm/button.h>
#include <gtkmm/paned.h>
#include <gtkmm/stock.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/toolbar.h>
#include <iostream>
#include <cstdio>

#define _(String) Glib::locale_to_utf8(gettext(String))

using namespace igs;

void PlayerList::update_n_players()
{
	Glib::ustring u;
	char buf[100];
	if (list.size() > 1) {
		snprintf(buf, 100, _("%d players").c_str(), list.size());
		u = buf;
	} else if (list.size() == 1) {
		u = _("one player");
	} else {
		u = _("no player");
	}
	n_players.set_text(u);
}

PlayerList::ModelCol::ModelCol()
{
	add(name);
	add(rank);
	add(state);
	add(idle);
}

void PlayerList::sort_by(Gtk::TreeView::Column * c)
{
	int s = c->get_sort_column_id();
	if (c == sort_col) {
		if (sort_type == Gtk::SORT_ASCENDING) sort_type = Gtk::SORT_DESCENDING;
		else sort_type = Gtk::SORT_ASCENDING;
		if (sort_toggle) {
			if (sort_col) sort_col->set_sort_indicator(false);
			store->set_sort_column_id(Gtk::TreeSortable::DEFAULT_SORT_COLUMN_ID, sort_type);
			sort_col = 0;
			return;
		}
		store->set_sort_column_id(s, sort_type);
		c->set_sort_order(sort_type);
		sort_toggle = true;
	} else {
		if (sort_col) sort_col->set_sort_indicator(false);
		sort_col = c;
		store->set_sort_column_id(s, sort_type);
		c->set_sort_indicator(true);
		c->set_sort_order(sort_type);
		sort_toggle = false;
	}
}

void PlayerList::select_player(const Gtk::TreeModel::Path & p, Gtk::TreeViewColumn * c)
{
	std::string n = (* store->get_iter(p))[m_col.name];
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->player.get_name() == n) {
		if (i->stats_box) {
			stats_book.set_current_page(stats_book.page_num(* i->stats_box));
		}
		break;
	}
	player_selected(n);
}

void PlayerList::refresh_request()
{
	refresh_time = time(0);
	refresh_list(refresh_filter.get_text());
	go::settings.set_string("who-filter", refresh_filter.get_text());
}

void PlayerList::page_change(GtkNotebookPage *, int p)
{
	// std::cerr << "page_change" << std::endl;
// 	if (p < 0) {
// 		stats_name.set_text("");
// 		return;
// 	}
// 	StatsBox * s = (StatsBox *) stats_book.get_nth_page(p);
// 	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->stats_box == s) {
// 		stats_name.set_text(i->player.get_name());
// 		break;
// 	}
}

void PlayerList::refresh_stats()
{
	int n = stats_book.get_current_page();
	if (n < 0) return;
	StatsBox * s = (StatsBox *) stats_book.get_nth_page(n);
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) {
		if (i->stats_box == s) {
			player_selected(i->player.get_name());
			break;
		}
	}	
}

void PlayerList::results_stats()
{
	int n = stats_book.get_current_page();
	if (n < 0) {
		ask_results("");
		add_message(Glib::ustring("<") + _("Results of last 23 games") + ">");
		return;
	}
	StatsBox * s = (StatsBox *) stats_book.get_nth_page(n);
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) {
		if (i->stats_box == s) {
			ask_results(i->player.get_name());
			Glib::ustring u = "<";
			char buf[100];
			snprintf(buf, 100, _("Latest results of %s").c_str(), i->player.get_name().c_str());
			u += buf;
			u += ">";
			add_message(u);
			break;
		}
	}	
}

void PlayerList::odds_stats()
{
	int n = stats_book.get_current_page();
	if (n < 0) {
		return;
	}
	StatsBox * s = (StatsBox *) stats_book.get_nth_page(n);
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) {
		if (i->stats_box == s) {
			ask_odds(i->player.get_name());
			break;
		}
	}	
}

void PlayerList::stored_stats()
{
	int n = stats_book.get_current_page();
	if (n < 0) {
		Glib::ustring u = "<";
		ask_stored("");
		u += _("My stored games");
		u += ">";
		add_message(u);
		return;
	}
	StatsBox * s = (StatsBox *) stats_book.get_nth_page(n);
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) {
		if (i->stats_box == s) {
			ask_stored(i->player.get_name());
			Glib::ustring u = "<";
			char buf[100];
			snprintf(buf, 100, _("Stored games of %s").c_str(), i->player.get_name().c_str());
			u += buf;
			u += ">";
			add_message(u);
			break;
		}
	}
}

void PlayerList::match_stats()
{
	int n = stats_book.get_current_page();
	if (n < 0) return;
	StatsBox * s = (StatsBox *) stats_book.get_nth_page(n);
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) {
		if (i->stats_box == s) {
			if (! i->match_box) {
				std::string t = i->player.get_name();
				MatchBox * m = new MatchBox(t);
				if (i->player.get_rank()) t += " " + i->player.get_rank().get_text();
				match_win.add_match(m, t);
				m->match_cancel.connect(mem_fun(* this, & PlayerList::close_match));
				m->match_request.connect(enter_match.make_slot());
				i->match_box = m;
			} else {
				match_win.raise(i->match_box);
			}
			break;
		}
	}	
}

void PlayerList::close_stats()
{
	int n = stats_book.get_current_page();
	if (n < 0) return;
	StatsBox * s = (StatsBox *) stats_book.get_nth_page(stats_book.get_current_page());
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->stats_box == s) {
		i->stats_box = 0;
		if (i->match_box) {
			match_win.del_match(i->match_box);
			delete i->match_box;
			i->match_box = 0;
		}
		break;
	}
	stats_book.remove(* s);
	delete s;
}

void PlayerList::close_match(MatchBox * m)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) {
		if (i->match_box == m) {
			i->match_box = 0;
			break;
		}
	}
	match_win.del_match(m);
	delete m;
}

void PlayerList::add_message(const Glib::ustring & msg)
{
	if (! msg.length()) return;
	bool bot = (vaj->get_value()  + vaj->get_page_size() >= vaj->get_upper());
	text_area.get_buffer()->insert(text_area.get_buffer()->end(), msg);
	text_area.get_buffer()->insert(text_area.get_buffer()->end(), "\n");
	if (bot) {
		text_area.scroll_to_mark(text_area.get_buffer()->create_mark("end", text_area.get_buffer()->end()), 0); // any easier way?
	}
}

void PlayerList::clear_text()
{
	text_area.get_buffer()->erase(text_area.get_buffer()->begin(), text_area.get_buffer()->end());
}

void PlayerList::enter_name()
{
	std::string n = stats_name.get_text();
	stats_name.set_text("");
	ready_stats(n);
}

StatsBox * PlayerList::new_stats_box(const std::string & n)
{
	StatsBox * s = new StatsBox(n);
	s->select_game.connect(select_game.make_slot());
	if (n == name) {
		s->set_info_edit();
		s->info_change.connect(change_info.make_slot());
	}
	stats_book.prepend_page(* s, n);
	stats_book.set_current_page(0);
	return s;
}

void PlayerList::ready_stats(const std::string & n) // make the stats_box ready for the player
{
	// find the player in the list
	Record * r;
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->player.get_name() == n) {
		// found
		r = & (* i);
		goto found_record;
	}

	list.push_back(Record());
	r = & list.back();
	r->iter = store->append();
	(* r->iter)[m_col.name] = n;
	(* r->iter)[m_col.rank] = new Player::Rank();
	(* r->iter)[m_col.state] = new Player::State();
	(* r->iter)[m_col.idle] = new Player::Idle();
	r->player.set_name(n);
	r->stats_box = 0;
	r->match_box = 0;

	update_n_players();

 found_record:
	if (r->stats_box == 0) {
		r->stats_box = new_stats_box(n);
		if (go::settings.get_option(go::Settings::OPT_GET_STATS_ON_POP)) player_selected(n);
	} else stats_book.set_current_page(stats_book.page_num(* r->stats_box));
}

void PlayerList::enter_talk()
{
	int n = stats_book.get_current_page();
	if (n < 0) { // no player to talk to
		add_message(_("no one to tell"));
		return;
	}
	StatsBox * s = (StatsBox *) stats_book.get_nth_page(n);
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) {
		if (i->stats_box == s) {
			talk_to_player(i->player.get_name(), talk_box.get_line());
			Glib::ustring u = name + ">" + i->player.get_name() + ": ";
			u += talk_box.get_line(); // maybe do some translation here...
			add_message(u);
			break;
		}
	}	
}

void PlayerList::open_stored(std::string s)
{
	lookup_stored(s);
}

void PlayerList::execute_stored(std::string s)
{
	load_stored(s);
}

void PlayerList::on_size_allocate(Gtk::Allocation & a)
{
	Gtk::Widget::on_size_allocate(a);
	if (a.get_width() != my_width || a.get_height() != my_height) {
		my_width = a.get_width();
		my_height = a.get_height();
		go::settings.set_int("igs-playerlist-width", my_width);
		go::settings.set_int("igs-playerlist-height", my_height);
	}
}

PlayerList::PlayerList() :
// 	n_players(_("no player")),
	talk_type(_("Tell"))
{
	set_title(_("Player List"));

	std::string ip = go::settings.get_image_path();
	if (Glib::file_test(ip + "/playerlist_icon.png", Glib::FILE_TEST_EXISTS)) {
		set_icon(Gdk::Pixbuf::create_from_file(ip + "/playerlist_icon.png"));
	}

	refresh_filter.set_text(go::settings.get_string("who-filter"));

	Gtk::HPaned * hp = Gtk::manage(new Gtk::HPaned);

	add(* hp);

	Gtk::VBox * vb = Gtk::manage(new Gtk::VBox);
	hp->add1(* vb);

	Gtk::HBox * hb = Gtk::manage(new Gtk::HBox);
	vb->pack_start(* hb, Gtk::PACK_SHRINK);

	Gtk::Image * im = Gtk::manage(new Gtk::Image(Gtk::Stock::REFRESH, Gtk::ICON_SIZE_SMALL_TOOLBAR));
	Gtk::Button * bt = Gtk::manage(new Gtk::Button);
	bt->add(* im);
	hb->pack_start(* bt, Gtk::PACK_SHRINK);
	bt->signal_clicked().connect(mem_fun(* this, & PlayerList::refresh_request));

	hb->pack_start(n_players);

	hb = Gtk::manage(new Gtk::HBox);
	vb->pack_start(* hb, Gtk::PACK_SHRINK);
	hb->pack_start(* Gtk::manage(new Gtk::Label(_("Filter") + ": ")), Gtk::PACK_SHRINK);

	hb->pack_start(refresh_filter);
	refresh_filter.signal_activate().connect(mem_fun(* this, & PlayerList::refresh_request));

	Gtk::ScrolledWindow * sw = Gtk::manage(new Gtk::ScrolledWindow);
	vb->pack_end(* sw);
	sw->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	sw->add(view);
	store = Gtk::ListStore::create(m_col);
	view.set_model(store);

	Gtk::TreeView::Column * v;
	int c;

	v = view.get_column(c = view.append_column(_("name"), m_col.name) - 1);
	v->set_clickable();
	v->set_reorderable();
	v->signal_clicked().connect(bind(mem_fun(* this, & PlayerList::sort_by), v));
	v->set_sort_column_id(c);

	v = view.get_column(c = view.append_column(_("rank"), rank_render) - 1);
	v->add_attribute(rank_render.property_user_data(), m_col.rank);
	v->set_clickable();
	v->set_reorderable();
	v->signal_clicked().connect(bind(mem_fun(* this, & PlayerList::sort_by), v));
	store->set_sort_func(c, sigc::bind(sigc::ptr_fun(& CustomText::compare), m_col.rank));
	v->set_sort_column_id(c);

	v = view.get_column(c = view.append_column(_("state"), state_render) - 1);
	v->add_attribute(state_render.property_user_data(), m_col.state);
	v->set_clickable();
	v->set_reorderable();
	v->signal_clicked().connect(bind(mem_fun(* this, & PlayerList::sort_by), v));
	store->set_sort_func(c, sigc::bind(sigc::ptr_fun(& CustomText::compare), m_col.state));
	v->set_sort_column_id(c);

	v = view.get_column(c = view.append_column(_("idle"), idle_render) - 1);
	v->add_attribute(idle_render.property_user_data(), m_col.idle);
	v->set_clickable();
	v->set_reorderable();
	v->signal_clicked().connect(bind(mem_fun(* this, & PlayerList::sort_by), v));
	store->set_sort_func(c, sigc::bind(sigc::ptr_fun(& CustomText::compare), m_col.idle));
	v->set_sort_column_id(c);

	sort_col = 0;
	sort_type = Gtk::SORT_ASCENDING;
	store->set_default_sort_func(sigc::ptr_fun(& CustomText::null_compare));
	view.signal_row_activated().connect(mem_fun(* this, & PlayerList::select_player));

	refresh_time = time(0);

	vb = Gtk::manage(new Gtk::VBox);
// 	hp->pack2(* vb, Gtk::SHRINK);
	hp->pack2(* vb);

	Gtk::Toolbar * tb = Gtk::manage(new Gtk::Toolbar);
// 	hb = Gtk::manage(new Gtk::HBox);
// 	vb->pack_start(* hb, Gtk::PACK_SHRINK);
// 	hb->pack_start(* tb, Gtk::PACK_SHRINK);
// 	hb->pack_end(stats_name, Gtk::PACK_SHRINK);
// 	stats_name.set_width_chars(10);
// 	stats_name.set_max_length(6);
// 	stats_name.signal_activate().connect(mem_fun(* this, & PlayerList::enter_name));

	vb->pack_start(* tb, Gtk::PACK_SHRINK);

	Gtk::ToolButton * tbn;

	tbn = Gtk::manage(new Gtk::ToolButton(_("Refresh")));
	tbn->set_tooltip_text(_("Refresh player stats"));
	tbn->signal_clicked().connect(mem_fun(* this, & PlayerList::refresh_stats));
	tbn->set_icon_widget(* Gtk::manage(new Gtk::Image(Gtk::Stock::REFRESH, Gtk::ICON_SIZE_LARGE_TOOLBAR)));
	tb->append(* tbn);

	tbn = Gtk::manage(new Gtk::ToolButton(_("Results")));
	tbn->set_tooltip_text(_("Latest game results"));
	tbn->signal_clicked().connect(mem_fun(* this, & PlayerList::results_stats));
	tbn->set_icon_widget(* Gtk::manage(new Gtk::Image(Gtk::Stock::INDEX, Gtk::ICON_SIZE_LARGE_TOOLBAR)));
	tb->append(* tbn);

	tbn = Gtk::manage(new Gtk::ToolButton(_("Odds")));
	tbn->set_tooltip_text(_("My odds against"));
	tbn->signal_clicked().connect(mem_fun(* this, & PlayerList::odds_stats));
	tbn->set_icon_widget(* Gtk::manage(new Gtk::Image(Gtk::Stock::FIND, Gtk::ICON_SIZE_LARGE_TOOLBAR)));
	tb->append(* tbn);

	tbn = Gtk::manage(new Gtk::ToolButton(_("Stored")));
	tbn->set_tooltip_text(_("Stored games"));
	tbn->signal_clicked().connect(mem_fun(* this, & PlayerList::stored_stats));
	tbn->set_icon_widget(* Gtk::manage(new Gtk::Image(Gtk::Stock::SAVE, Gtk::ICON_SIZE_LARGE_TOOLBAR)));
	tb->append(* tbn);

	tbn = Gtk::manage(new Gtk::ToolButton(_("Match")));
	tbn->set_tooltip_text(_("Match player"));
	tbn->signal_clicked().connect(mem_fun(* this, & PlayerList::match_stats));
	tbn->set_icon_widget(* Gtk::manage(new Gtk::Image(Gtk::Stock::EXECUTE, Gtk::ICON_SIZE_LARGE_TOOLBAR)));
	tb->append(* tbn);

	tbn = Gtk::manage(new Gtk::ToolButton(Gtk::Stock::DELETE));
	tbn->set_tooltip_text(_("Delete player stats"));
	tbn->signal_clicked().connect(mem_fun(* this, & PlayerList::close_stats));
	tb->append(* tbn);

	tbn = Gtk::manage(new Gtk::ToolButton(Gtk::Stock::CLEAR));
	tbn->set_tooltip_text(_("Clear messages"));
	tbn->signal_clicked().connect(mem_fun(* this, & PlayerList::clear_text));
	tb->append(* tbn);

	Gtk::VPaned * vp = Gtk::manage(new Gtk::VPaned);
	vb->pack_start(* vp);

	// vp->pack1(stats_book, Gtk::SHRINK);
	sw = Gtk::manage(new Gtk::ScrolledWindow);
// 	vp->pack1(* sw, Gtk::SHRINK);
	vp->pack1(* sw);
	sw->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	sw->add(stats_book);
	stats_book.set_scrollable();
	stats_book.popup_enable();
	stats_book.signal_switch_page().connect(mem_fun(* this, & PlayerList::page_change));

	sw = Gtk::manage(new Gtk::ScrolledWindow);
	vp->pack2(* sw);
	sw->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	sw->add(text_area);
	vaj = sw->get_vadjustment();
	text_area.set_editable(false);
	text_area.set_wrap_mode(Gtk::WRAP_WORD);

	hb = Gtk::manage(new Gtk::HBox);
	vb->pack_end(* hb, Gtk::PACK_SHRINK);

	hb->pack_start(talk_type, Gtk::PACK_SHRINK);
	hb->pack_start(talk_box);
	talk_box.signal_activate().connect(mem_fun(* this, & PlayerList::enter_talk));

	hb->pack_start(* Gtk::manage(new Gtk::Label(Glib::ustring(" ") + _("Who") + ":")), Gtk::PACK_SHRINK);

	hb->pack_end(stats_name, Gtk::PACK_SHRINK);
	stats_name.set_width_chars(6);
	stats_name.set_max_length(10);
	stats_name.signal_activate().connect(mem_fun(* this, & PlayerList::enter_name));

	show_all_children();

	my_width = go::settings.get_int("igs-playerlist-width");
	if (my_width <= 0) my_width = 400;
	my_height = go::settings.get_int("igs-playerlist-height");
	if (my_height <= 0) my_height = 300;
	resize(my_width, my_height);

	update_n_players();
}

PlayerList::~PlayerList()
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) {
		if (i->stats_box) {
			stats_book.remove(* i->stats_box);
			delete i->stats_box;
			i->stats_box = 0;
		}
		if (i->match_box) {
			match_win.del_match(i->match_box);
			delete i->match_box;
			i->match_box = 0;
		}
		void * v;
		if ((v = (* i->iter)[m_col.rank])) delete (Player::Rank *) v;
		if ((v = (* i->iter)[m_col.state])) delete (Player::State *) v;
		if ((v = (* i->iter)[m_col.idle])) delete (Player::Idle *) v;
	}
}

void PlayerList::set_name(const std::string & n)
{
	name = n;
	set_title(Glib::ustring("Igs(") + n + ") " + _("Player List"));
	match_win.set_title(Glib::ustring("Igs(") + n + ") " + _("Match List"));
}

void PlayerList::update(const Player & p)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->player.get_name() == p.get_name()) {
		// std::cerr << "updating player " << p.get_name() << ": state = [" << p.get_state().get_text() << "]\n";
		i->player.update(p);
		if (i->stats_box) i->stats_box->update(p);
		void * v;
		if ((v = (* i->iter)[m_col.rank])) delete (Player::Rank *) v;
		if ((v = (* i->iter)[m_col.state])) delete (Player::State *) v;
		if ((v = (* i->iter)[m_col.idle])) delete (Player::Idle *) v;
		(* i->iter)[m_col.rank] = new Player::Rank(i->player.get_rank());
		(* i->iter)[m_col.state] = new Player::State(i->player.get_state());
		(* i->iter)[m_col.idle] = new Player::Idle(i->player.get_idle());
		return;
	}

	Gtk::TreeModel::Children::iterator i = store->append();
	(* i)[m_col.name] = p.get_name();
	(* i)[m_col.rank] = new Player::Rank(p.get_rank());
	(* i)[m_col.state] = new Player::State(p.get_state());
	(* i)[m_col.idle] = new Player::Idle(p.get_idle());

	Record r;
	r.player = p;
	r.stats_box = 0;
	r.iter = i;
	r.match_box = 0;
	list.push_back(r);

	update_n_players();
}

void PlayerList::stats(const Player & p)
{
	if (go::settings.get_option(go::Settings::OPT_POP_PLAYERS_ON_STATS)) present();

	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->player.get_name() == p.get_name()) {
		i->player.update(p);
		void * v;
		if ((v = (* i->iter)[m_col.rank])) delete (Player::Rank *) v;
		if ((v = (* i->iter)[m_col.state])) delete (Player::State *) v;
		if ((v = (* i->iter)[m_col.idle])) delete (Player::Idle *) v;
		(* i->iter)[m_col.rank] = new Player::Rank(i->player.get_rank());
		(* i->iter)[m_col.state] = new Player::State(i->player.get_state());
		(* i->iter)[m_col.idle] = new Player::Idle(i->player.get_idle());

		if (! i->stats_box) i->stats_box = new_stats_box(i->player.get_name());
		else stats_book.set_current_page(stats_book.page_num(* i->stats_box));
		i->stats_box->update(p);
		return;
	}

	Gtk::TreeModel::Children::iterator i = store->append();
	(* i)[m_col.name] = p.get_name();
	(* i)[m_col.rank] = new Player::Rank(p.get_rank());
	(* i)[m_col.state] = new Player::State(p.get_state());
	(* i)[m_col.idle] = new Player::Idle(p.get_idle());

	Record r;
	r.player = p;
	r.stats_box = new_stats_box(r.player.get_name());
	r.stats_box->update(p);
	r.iter = i;
	r.match_box = 0;
	list.push_back(r);

	update_n_players();
}

void PlayerList::odds(const Odds & o)
{
	int n = stats_book.get_current_page();
	char buf[100];
	if (n < 0) { // no player? what's this?
		snprintf(buf, 100, _("Odds with handicap %g:\n W~B= %g%%~%g%%").c_str(), o.handicap, o.win_as_white, o.win_as_black);
		add_message(buf);
		return;
	}
	StatsBox * s = (StatsBox *) stats_book.get_nth_page(n);
	std::string w;
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->stats_box == s) {
		w = i->player.get_name();
		break;
	}
	snprintf(buf, 100, _("Odds with %s (h:%g):\n W~B= %g%%~%g%%").c_str(), w.c_str(), o.handicap, o.win_as_white, o.win_as_black);
	add_message(buf);
}

void PlayerList::end_update()
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->player.get_stamp() < refresh_time) {
		if (i->stats_box || i->match_box) continue;
		void * v;
		if ((v = (* i->iter)[m_col.rank])) delete (Player::Rank *) v;
		if ((v = (* i->iter)[m_col.state])) delete (Player::State *) v;
		if ((v = (* i->iter)[m_col.idle])) delete (Player::Idle *) v;
		store->erase(i->iter);
		std::vector<Record>::iterator ii = i;
		i --;
		list.erase(ii);
	}

	update_n_players();
}

void PlayerList::add_result(const std::vector<Result> & r)
{
	std::string w;
	int n = stats_book.get_current_page();
	if (n < 0) {
		for (std::vector<Result>::const_iterator j = r.begin(); j != r.end(); j ++) add_message(j->short_text());
		add_message("==");
		return;
	}
	StatsBox * s = (StatsBox *) stats_book.get_nth_page(n);
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->stats_box == s) {
		w = i->player.get_name();
		break;
	}
	int win = 0;
	int loss = 0;
	for (std::vector<Result>::const_iterator j = r.begin(); j != r.end(); j ++) {
		if (j->white == w) {
			if (j->winer == go::TURN_WHITE) win ++;
			else loss ++;
		}
		if (j->black == w) {
			if (j->winer == go::TURN_BLACK) win ++;
			else loss ++;
		}
		add_message(j->short_text(w));
	}

	char buf[100];
	snprintf(buf, 100, _("%s wins %d out of %d games").c_str(), w.c_str(), win, r.size());
	add_message(buf);
}

void PlayerList::add_stored(const std::vector<Storage> & r)
{
	if (! r.size()) {
		add_message(_("no stored game"));
		return;
	}

	bool at_bottom = (vaj->get_value()  + vaj->get_page_size() >= vaj->get_upper());

	for (std::vector<Storage>::const_iterator i = r.begin(); i != r.end(); i ++) {
		if (i != r.begin()) text_area.get_buffer()->insert(text_area.get_buffer()->end(), " ");

		std::string file = i->white + "-" + i->black;
		Gtk::TextIter iter = text_area.get_buffer()->end();
		Glib::RefPtr<Gtk::TextChildAnchor> anchor = text_area.get_buffer()->create_child_anchor(iter);
		Gtk::Button * bt;

		bt = Gtk::manage(new Gtk::Button(file)); // hope bt can be deleted correctly...
		text_area.add_child_at_anchor(* bt, anchor);
		bt->show();
		bt->signal_clicked().connect(bind(mem_fun(* this, & PlayerList::open_stored), file));

		if (i->white == name || i->black == name) {
			iter = text_area.get_buffer()->end();
			anchor = text_area.get_buffer()->create_child_anchor(iter);
			Gtk::Image * im = Gtk::manage(new Gtk::Image(Gtk::Stock::OPEN, Gtk::ICON_SIZE_SMALL_TOOLBAR));
			bt = Gtk::manage(new Gtk::Button);
			bt->add(* im);
			text_area.add_child_at_anchor(* bt, anchor);
			bt->show();
			im->show();
			bt->signal_clicked().connect(bind(mem_fun(* this, & PlayerList::execute_stored), file));
		}

	}
	text_area.get_buffer()->insert(text_area.get_buffer()->end(), "\n");
	char buf[100];
	if (r.size() > 1) snprintf(buf, 100, _("%d stored games").c_str(), r.size());
	else snprintf(buf, 100, "%s", _("one stored game").c_str());
	text_area.get_buffer()->insert(text_area.get_buffer()->end(), buf);
	text_area.get_buffer()->insert(text_area.get_buffer()->end(), "\n");
	if (at_bottom) {
		text_area.scroll_to_mark(text_area.get_buffer()->create_mark("end", text_area.get_buffer()->end()), 0);
		vaj->set_value(vaj->get_upper());
	}
}

void PlayerList::msg_tell(const std::string & w, const std::string & m)
{
	if (go::settings.get_option(go::Settings::OPT_POP_PLAYERS_ON_TELL)) present();
	ready_stats(w);
	Glib::ustring u = w + ": ";
	u += auto_convert(m); // try to convert it to UTF-8
	add_message(u);
}

void PlayerList::match_request(const Match & m)
{
	if (go::settings.get_option(go::Settings::OPT_POP_PLAYERS_ON_MATCH)) present();
	std::vector<Record>::iterator i;
	for (i = list.begin(); i != list.end(); i ++) if (i->player.get_name() == m.opponent) break;
	if (i == list.end()) {
		Record r;
		r.iter = store->append();
		(* r.iter)[m_col.name] = m.opponent;
		(* r.iter)[m_col.rank] = new Player::Rank();
		(* r.iter)[m_col.state] = new Player::State();
		(* r.iter)[m_col.idle] = new Player::Idle();
		r.player.set_name(m.opponent);
		r.stats_box = 0;
		r.match_box = 0;
		list.push_back(r);
		i = list.end() - 1;

		update_n_players();
	}
	if (! i->stats_box) i->stats_box = new_stats_box(m.opponent);
	else stats_book.set_current_page(stats_book.page_num(* i->stats_box));
	if (! i->match_box) {
		i->match_box = new MatchBox(m.opponent);
		match_win.add_match(i->match_box, m.opponent);
		i->match_box->match_cancel.connect(mem_fun(* this, & PlayerList::close_match));
		i->match_box->match_request.connect(enter_match.make_slot());
	} else {
		match_win.raise(i->match_box);
	}
	i->match_box->update(m);
}

void PlayerList::match_create(const std::string & w)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->player.get_name() == w) {
		if (i->match_box) {
			match_win.del_match(i->match_box);
			delete i->match_box;
			i->match_box = 0;
		}
		break;
	}
}

void PlayerList::match_decline(const std::string & w)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->player.get_name() == w) {
		if (i->match_box) {
			i->match_box->decline();
		}
		break;
	}
}

void PlayerList::match_not_online(const std::string & w)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->player.get_name() == w) {
		if (i->match_box) {
			i->match_box->not_online();
		}
		break;
	}
}

void PlayerList::match_not_open(const std::string & w)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->player.get_name() == w) {
		if (i->match_box) {
			i->match_box->not_open();
		}
		break;
	}
}

void PlayerList::match_in_match(const std::string & w)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->player.get_name() == w) {
		if (i->match_box) {
			i->match_box->in_match();
		}
		break;
	}
}
