/* ccgo: igs/player.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "../config.hh"
#include "../gettext.h"
#include "player.hh"
#include <sstream>
#include <cstdio>

#define _(String) Glib::locale_to_utf8(gettext(String))

using namespace igs;

void Player::Rank::make_str()
{
	std::ostringstream s;
	int r = int(strength + 0.5);
	if (r > 40) s << r - 40 << "p";
	else if (r > 28) s << r - 28 << "d";
	else if (r > 0) s << 29 - r << "k";
	else if (strength >= 0) s << "nr";
	else s << "?";
	if (certified) s << "*";
	rank = s.str();
}

Player::Rank::Rank() :
	rank("?")
{
	strength = -1;
	certified = false;
}

Player::Rank::Rank(const std::string & s)
{
	if (! build(s)) {
		strength = -1;
		certified = false;
		rank = " ";
	}
}

Player::Rank::~Rank()
{
}

bool Player::Rank::build(const std::string & s)
{
	std::istringstream ss(s);
	int r;
	ss >> r;
	if (ss.fail()) {
		ss.clear();
		std::string n;
		ss >> n;
		if (n != "NR") return false;
		strength = 0;
		certified = false;
		rank = "nr";
		return true;
	}
	char c = ss.get();
	if (c == 'p') strength = r + 40;
	else if (c == 'd') strength = r + 28;
	else if (c == 'k') strength = 29 - r;
	else return false;
	certified = ss.get() == '*';
	make_str();
	return true;
}

const std::string & Player::Rank::get_text() const
{
	return rank;
}

bool Player::Rank::operator < (const CustomText & c) const
{
	const Player::Rank * r = dynamic_cast<const Player::Rank *>(& c);
	if (! r) return false;
	return strength < r->strength;
}

Player::Rank::operator bool() const
{
	return strength >= 0;
}

const Glib::ustring Player::Rank::long_text() const
{
	Glib::ustring u;

	char buf[100];
	int r = int(strength + 0.5);
	if (r > 40) {
		snprintf(buf, 100, _("%d professional dan").c_str(), r - 40);
		u = buf;
	} else if (r > 28) {
		snprintf(buf, 100, _("%d dan").c_str(), r - 28);
		u = buf;
	} else if (r > 0) {
		snprintf(buf, 100, _("%d kyu").c_str(), 29 - r);
		u = buf;
	} else if (strength >= 0) {
		u = _("no rank");
	} else {
		u = _("unknown");
	}

	if (certified) {
		u += " " + _("[certified]");
	} else if (r > 0) {
		u += " " + _("[claimed]");
	}

	return u;
}

////////////////////////////////////////////////////////////////////////////////
Player::State::State()
{
	state = STATE_UNKNOWN;
	str = "?";
}

Player::State::State(const std::string & s)
{
	switch (s[1]) {
	case '!':
		state = STATE_LOOK;
		str = "!";
		break;
	case ' ':
		state = STATE_OPEN;
		str = "o";
		break;
	case 'X':
		state = STATE_CLOSE;
		str = "x";
		break;
	default:
		state = STATE_UNKNOWN;
	}
}

Player::State::~State()
{
}

const std::string & Player::State::get_text() const
{
	return str;
}

bool Player::State::operator < (const CustomText & c) const
{
	const Player::State * s = dynamic_cast<const Player::State *>(& c);
	if (! s) return false;
	return state < s->state;
}

Player::State::operator bool () const
{
	return state;
}

const std::string Player::State::long_text() const
{
	std::string r;
	switch (state) {
	case STATE_UNKNOWN:
		r = _("unknown");
		break;
	case STATE_CLOSE:
		r = _("not accepting match");
		break;
	case STATE_OPEN:
		r = _("accepting match");
		break;
	case STATE_LOOK:
		r = _("looking for match");
		break;
	}
	return r;
}
////////////////////////////////////////////////////////////////////////////////
Player::Idle::Idle()
{
	time = -2;
	str = "?";
}

Player::Idle::Idle(const std::string & st)
{
	std::istringstream s(st);
	s >> time;
	if (s.fail()) {
		time = -2;
		str = "?";
	}
	char c = s.get();
	std::ostringstream o;
	if (c == 's') {
		o << time;
		o << 's';
		time *= 1;
	} else if (c == 'm') {
		o << time;
		o << 'm';
		time *= 60;
	} else if (c == 'h') {
		o << time;
		o << 'm';
		time *= 3600;
	} else {
		time = -2;
		o << "?";
	}
	str = o.str();
}

Player::Idle::~Idle()
{
}

void Player::Idle::offline()
{
	time = -1;
	str = "N/A";
}

const std::string & Player::Idle::get_text() const
{
	return str;
}

bool Player::Idle::operator < (const CustomText & c) const
{
	const Player::Idle * i = dynamic_cast<const Player::Idle *>(& c);
	if (! i) return false;
	if (i->time < 0 && time >= 0) return false;
	if (time < 0 && i->time >= 0) return true;
	return time > i->time;
}

Player::Idle::operator bool () const
{
	return time != -2;
}

const Glib::ustring Player::Idle::long_text() const
{
	if (time == -1) return _("[offline]");
	else if (time == -2) return _("[unknown]");

	char buf[100];
	int t;

	if (time >= 3600) {
		t = time / 3600;
		if (t > 1) snprintf(buf, 100, _("%d hours").c_str(), t);
		else snprintf(buf, 100, _("%d hour").c_str(), t);
	} else if (time >= 60) {
		t = time / 60;
		if (t > 1) snprintf(buf, 100, _("%d minutes").c_str(), t);
		else snprintf(buf, 100, _("%d minute").c_str(), t);
	} else {
		t = time;
		if (t > 1) snprintf(buf, 100, _("%d seconds").c_str(), t);
		else snprintf(buf, 100, _("%d second").c_str(), t);
	}

	return buf;
}

bool Player::Idle::is_online() const
{
	return time >= 0;
}
////////////////////////////////////////////////////////////////////////////////
Player::Player()
{
	stamp = time(0);
	game = 0;
	rated_games = 0;
	wins = 0;
	losses = 0;
	observing = -1;
	playing = -1;
	def_time = -1;
	def_size = -1;
	def_byo_time = -1;
	def_byo_stones = -1;
}

void Player::update(const Player & p)
{
	stamp = p.stamp;
	name = p.name;
	rank = p.rank;
	idle = p.idle;
	if (p.state) state = p.state;
	if (p.game) game = p.game;
	if (p.language.length()) language = p.language;
	if (p.rated_games) rated_games = p.rated_games;
	if (p.self_rank) self_rank = p.self_rank;
	if (p.wins) wins = p.wins;
	if (p.losses) losses = p.losses;
	if (p.observing >= 0) observing = p.observing;
	if (p.playing >= 0) playing = p.playing;
	if (p.last) last = p.last;
	idle = p.idle;
	if (p.address.length()) address = p.address;
	if (p.reg_date) reg_date = p.reg_date;
	if (p.info.length()) info = p.info;
	if (p.def_time >= 0) def_time = p.def_time;
	if (p.def_size >= 0) def_size = p.def_size;
	if (p.def_byo_time >= 0) def_byo_time = p.def_byo_time;
	if (p.def_byo_stones >= 0) def_byo_stones = p.def_byo_stones;
}
