/* ccgo: igs/match_box.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "../config.hh"
#include "../gettext.h"
#include "match_box.hh"
#include <gtkmm/label.h>
#include <gtkmm/button.h>
#include <gtkmm/spinbutton.h>
#include <gtkmm/buttonbox.h>
#include <gtkmm/stock.h>
#include "../settings.hh"
#include <cstdio>
#define _(String) Glib::locale_to_utf8(gettext(String))

using namespace igs;

void MatchBox::send_match()
{
	match.my_turn = my_turn.get_history() == 0 ? go::TURN_BLACK : go::TURN_WHITE;
	match.board_size = int(board_size->get_value());
	match.time = int(time->get_value());
	match.byo_time = int (byo_time->get_value());
	match.set_option = set_option.get_active();
	match_request(match);
	head_line.set_text(_("Match request sent"));
	if (load_defs) {
		go::settings.set_int("match-board-size", int(board_size->get_value()));
		go::settings.set_int("match-main-time", int(time->get_value()));
		go::settings.set_int("match-byo-time", int(byo_time->get_value()));
	}
}

void MatchBox::set_option_toggle()
{
	if (set_option.get_active()) {
		match.set_option = true;
		option_box.set_sensitive(true);
		if (load_defs) {
			board_size->set_value(go::settings.get_int("match-board-size"));
			time->set_value(go::settings.get_int("match-main-time"));
			byo_time->set_value(go::settings.get_int("match-byo-time"));
		}
	} else {
		match.set_option = false;
		option_box.set_sensitive(false);
	}
}

void MatchBox::decline_match()
{
	match.decline = true;
	match_request(match);
	match_cancel(this); // close window after declining the match
}

MatchBox::MatchBox(const std::string & n) :
	Gtk::Table(2),
	set_option(_("Set details"), false),
	board_size(Gtk::manage(new Gtk::Adjustment(19, 5, 23))),
	time(Gtk::manage(new Gtk::Adjustment(75, 0, 530))),
	byo_time(Gtk::manage(new Gtk::Adjustment(0, 0, 900))),
	load_defs(true), // load defs when set_option is turned on
	decline_button(Gtk::Stock::NO)
{
	Gtk::VBox * vb;
	vb = Gtk::manage(new Gtk::VBox);
	attach(* vb, 0, 1, 0, 1);

	vb->pack_start(head_line, Gtk::PACK_SHRINK);
	char buf[100];
	snprintf(buf, 100, _("Match with %s").c_str(), n.c_str());
	head_line.set_text(buf);
	match.opponent = n;

	vb->pack_start(set_option, Gtk::PACK_SHRINK);
	set_option.signal_toggled().connect(mem_fun(* this, & MatchBox::set_option_toggle));

	vb->pack_start(option_box, Gtk::PACK_SHRINK);

	Gtk::HBox * hb;
	// my turn
	hb = Gtk::manage(new Gtk::HBox);
	option_box.pack_start(* hb, Gtk::PACK_SHRINK);
	hb->pack_start(* Gtk::manage(new Gtk::Label(_("My turn") + ": ")), Gtk::PACK_SHRINK);
	hb->pack_start(my_turn, Gtk::PACK_SHRINK);
	Gtk::Menu * m = Gtk::manage(new Gtk::Menu);
	m->append(* Gtk::manage(new Gtk::MenuItem(_("Black"))));
	m->append(* Gtk::manage(new Gtk::MenuItem(_("White"))));
	my_turn.set_menu(* m);
	my_turn.set_history(0);
	// board size
	hb = Gtk::manage(new Gtk::HBox);
	option_box.pack_start(* hb, Gtk::PACK_SHRINK);
	Gtk::SpinButton * sb;
	hb->pack_start(* Gtk::manage(new Gtk::Label(_("Board size") + ": ")), Gtk::PACK_SHRINK);
	sb = Gtk::manage(new Gtk::SpinButton(* board_size));
	hb->pack_start(* sb, Gtk::PACK_SHRINK);
	// main time
	hb = Gtk::manage(new Gtk::HBox);
	option_box.pack_start(* hb, Gtk::PACK_SHRINK);
	hb->pack_start(* Gtk::manage(new Gtk::Label(_("Main time") + ": ")), Gtk::PACK_SHRINK);
	sb = Gtk::manage(new Gtk::SpinButton(* time));
	hb->pack_start(* sb, Gtk::PACK_SHRINK);
	hb->pack_start(* Gtk::manage(new Gtk::Label(_("min."))), Gtk::PACK_SHRINK);
	// byo time
	hb = Gtk::manage(new Gtk::HBox);
	option_box.pack_start(* hb, Gtk::PACK_SHRINK);
	hb->pack_start(* Gtk::manage(new Gtk::Label(_("Byo-yomi time") + ": ")), Gtk::PACK_SHRINK);
	sb = Gtk::manage(new Gtk::SpinButton(* byo_time));
	hb->pack_start(* sb, Gtk::PACK_SHRINK);
	hb->pack_start(* Gtk::manage(new Gtk::Label(_("min."))), Gtk::PACK_SHRINK);

	attach(button_area, 0, 1, 1, 2, Gtk::FILL);
	Gtk::Button * bt;

	bt = Gtk::manage(new Gtk::Button(Gtk::Stock::CANCEL));
	button_area.pack_end(* bt);
	bt->signal_clicked().connect(bind(match_cancel.make_slot(), this));

	button_area.pack_end(decline_button);
	decline_button.signal_clicked().connect(mem_fun(* this, & MatchBox::decline_match));

	bt = Gtk::manage(new Gtk::Button(Gtk::Stock::OK));
	button_area.pack_end(* bt);
	bt->signal_clicked().connect(mem_fun(* this, & MatchBox::send_match));

	show_all_children();
	option_box.set_sensitive(false);
	decline_button.hide();
	show();
}

MatchBox::~MatchBox()
{
// 	std::cerr << "deleting MatchBox" << endl;
}

void MatchBox::update(const Match & m) // incoming match request
{
	char buf[100];
	if (m.get_automatch()) {
		snprintf(buf, 100, _("Automatch from %s (Byo. Stones:%d)").c_str(), m.opponent.c_str(), m.byo_stone);
		head_line.set_text(buf);
		match.set_automatch(true);
		set_option.set_active(false);
	} else {
		snprintf(buf, 100, _("Match request from %s").c_str(), m.opponent.c_str());
		head_line.set_text(buf);
		match.set_automatch(false);
	}
	if (m.set_option) {
		load_defs = false; // don't load defs when option is set by request
		if (m.my_turn == go::TURN_BLACK) my_turn.set_history(0);
		else my_turn.set_history(1);
		board_size->set_value(m.board_size);
		time->set_value(m.time);
		byo_time->set_value(m.byo_time);
		set_option.set_active(true);
	}
	decline_button.show(); // I can decline an incoming match
}

void MatchBox::decline()
{
	char buf[100];
	snprintf(buf, 100, _("%s declined the match").c_str(), match.opponent.c_str());
	head_line.set_text(buf);
}

void MatchBox::not_online()
{
	char buf[100];
	snprintf(buf, 100, _("%s is offline").c_str(), match.opponent.c_str());
	head_line.set_text(buf);
}

void MatchBox::not_open()
{
	char buf[100];
	snprintf(buf, 100, _("%s is not accepting match").c_str(), match.opponent.c_str());
	head_line.set_text(buf);
}

void MatchBox::in_match()
{
	char buf[100];
	snprintf(buf, 100, _("%s is already in a match").c_str(), match.opponent.c_str());
	head_line.set_text(buf);
}
