/* ccgo: igs/game_list.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "../config.hh"
#include "../gettext.h"
#include "game_list.hh"
#include "static.hh"
#include "../settings.hh"
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/treesortable.h>
#include <glibmm/listhandle.h>
#include <gtkmm/treeviewcolumn.h>
#include <gtkmm/stock.h>
#include <sstream>
#include <cstdio>

#define _(String) Glib::locale_to_utf8(gettext(String))

using namespace igs;

void GameList::update_n_games()
{
	Glib::ustring u;
	char buf[100];
	if (list.size() > 1) {
		snprintf(buf, 100, _("%d games").c_str(), list.size());
		u = buf;
	} else if (list.size() == 1) {
		u = _("one game");
	} else {
		u = _("no game");
	}
	n_games.set_text(u);
}

GameList::ModelCol::ModelCol()
{
	add(num);
	add(white);
	add(w_rank);
	add(black);
	add(b_rank);
}

void GameList::sort_by(Gtk::TreeView::Column * c)
{
	int s = c->get_sort_column_id();
	if (c == sort_col) {
		if (sort_type == Gtk::SORT_ASCENDING) sort_type = Gtk::SORT_DESCENDING;
		else sort_type = Gtk::SORT_ASCENDING;
		if (sort_toggle) {
			if (sort_col) sort_col->set_sort_indicator(false);
			store->set_sort_column_id(Gtk::TreeSortable::DEFAULT_SORT_COLUMN_ID, sort_type);
			sort_col = 0;
			return;
		}
		store->set_sort_column_id(s, sort_type);
		c->set_sort_order(sort_type);
		sort_toggle = true;
	} else {
		if (sort_col) sort_col->set_sort_indicator(false);
		sort_col = c;
		store->set_sort_column_id(s, sort_type);
		c->set_sort_indicator(true);
		c->set_sort_order(sort_type);
		sort_toggle = false;
	}
}

void GameList::select_game(const Gtk::TreeModel::Path & p, Gtk::TreeViewColumn * c)
{
	int g = (* store->get_iter(p))[m_col.num];
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		if (i->show) {
			i->show->present();
			return;
		}
		break;
	}
	game_selected(g);
}

void GameList::close_show(Show * s)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) {
		if (i->show == s) {
			i->show = 0;
			if (i->game.get_num() != my_match) {
				close_game(i->game.get_num());
				i->state = GS_CLOSE;
			}
			return;
		}
	}
}

void GameList::refresh_request()
{
	refresh_time = time(0);
	refresh_list();
}

void GameList::on_size_allocate(Gtk::Allocation & a)
{
	Gtk::Widget::on_size_allocate(a);
	if (a.get_width() != my_width || a.get_height() != my_height) {
		my_width = a.get_width();
		my_height = a.get_height();
		go::settings.set_int("igs-gamelist-width", my_width);
		go::settings.set_int("igs-gamelist-height", my_height);
	}
}

GameList::GameList()
{
	set_title(Glib::ustring("Igs: ") + _("Game List"));

	std::string ip = go::settings.get_image_path();
	if (Glib::file_test(ip + "/gamelist_icon.png", Glib::FILE_TEST_EXISTS)) {
		set_icon(Gdk::Pixbuf::create_from_file(ip + "/gamelist_icon.png"));
	}

	Gtk::VBox * vb = Gtk::manage(new Gtk::VBox);
	add(* vb);

	Gtk::HBox * hb = Gtk::manage(new Gtk::HBox);
	vb->pack_start(* hb, Gtk::PACK_SHRINK);

	Gtk::Image * im = Gtk::manage(new Gtk::Image(Gtk::Stock::REFRESH, Gtk::ICON_SIZE_SMALL_TOOLBAR));
	Gtk::Button * bt = Gtk::manage(new Gtk::Button);
	bt->add(* im);
	hb->pack_start(* bt, Gtk::PACK_SHRINK);
	bt->signal_clicked().connect(mem_fun(* this, & GameList::refresh_request));

	hb->pack_start(n_games);

	Gtk::ScrolledWindow * sw = Gtk::manage(new Gtk::ScrolledWindow);
	vb->pack_end(* sw);
	sw->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_ALWAYS);
	sw->add(view);
	store = Gtk::ListStore::create(m_col);
	view.set_model(store);

	Gtk::TreeView::Column * v;
	int c;

	v = view.get_column(c = view.append_column(_("num"), m_col.num) - 1);
	v->set_clickable();
	v->set_reorderable();
	v->signal_clicked().connect(bind(mem_fun(* this, & GameList::sort_by), v));
	v->set_sort_column_id(c);

	v = view.get_column(c = view.append_column(_("white"), m_col.white) - 1);
	v->set_clickable();
	v->set_reorderable();
	v->signal_clicked().connect(bind(mem_fun(* this, & GameList::sort_by), v));
	v->set_sort_column_id(c);

	v = view.get_column(c = view.append_column(_("w rank"), w_render) - 1);
	v->add_attribute(w_render.property_user_data(), m_col.w_rank);
	v->set_clickable();
	v->set_reorderable();
	v->signal_clicked().connect(bind(mem_fun(* this, & GameList::sort_by), v));
	store->set_sort_func(c, sigc::bind(sigc::ptr_fun(& CustomText::compare), m_col.w_rank));
	v->set_sort_column_id(c);

	v = view.get_column(c = view.append_column(_("black"), m_col.black) - 1);
	v->set_clickable();
	v->set_reorderable();
	v->signal_clicked().connect(bind(mem_fun(* this, & GameList::sort_by), v));
	v->set_sort_column_id(c);

	v = view.get_column(c = view.append_column(_("b rank"), b_render) - 1);
	v->add_attribute(b_render.property_user_data(), m_col.b_rank);
	v->set_clickable();
	v->set_reorderable();
	v->signal_clicked().connect(bind(mem_fun(* this, & GameList::sort_by), v));
	store->set_sort_func(c, sigc::bind(sigc::ptr_fun(& CustomText::compare), m_col.b_rank));
	v->set_sort_column_id(c);

	sort_col = 0;
	sort_type = Gtk::SORT_ASCENDING;
	store->set_default_sort_func(sigc::ptr_fun(& CustomText::null_compare));
	view.signal_row_activated().connect(mem_fun(* this, & GameList::select_game));
	my_match = 0;

	my_width = go::settings.get_int("igs-gamelist-width");
	if (my_width <= 0) my_width = 200;
	my_height = go::settings.get_int("igs-gamelist-height");
	if (my_height <= 0) my_height = 300;
	resize(my_width, my_height);

	update_n_games();
}

GameList::~GameList()
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) {
		if (i->show) {
			i->show->hang_up(_("disconnected"));
		}
		void * v;
		if ((v = (* i->iter)[m_col.w_rank])) delete (Player::Rank *) v;
		if ((v = (* i->iter)[m_col.b_rank])) delete (Player::Rank*) v;
	}
}

void GameList::set_name(const std::string & n)
{
	name = n;
	set_title(Glib::ustring("Igs(") + n + ") " + _("Game List"));
}

void GameList::update(const Game & g)
{
	char buf[100];
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g.get_num()) {
		i->game = g;
		if (i->state == GS_NEED_WIN) { // open a window for this game
			if (i->show) i->show->hang_up(_("Mutated"));
			i->show = new Show(i->game.get_board_size());
			i->show->set_komi(i->game.get_komi());
			i->show->show_close.connect(mem_fun(* this, & GameList::close_show));
			i->show->start_time(600, i->game.get_byo_time() * 60, 25);

// 			std::ostringstream s;
// 			s << "Game " << i->game.get_num() << ": ";
// 			s << i->game.get_white() << '[' << i->game.get_w_rank().get_text() << "] vs ";
// 			s << i->game.get_black() << '[' << i->game.get_b_rank().get_text() << ']';
// 			i->show->set_title(s.str());

			snprintf(buf, 100, _("Game %d: %s[%s] vs %s[%s]").c_str(), i->game.get_num(), i->game.get_white().c_str(), i->game.get_w_rank().get_text().c_str(), i->game.get_black().c_str(), i->game.get_b_rank().get_text().c_str());
			i->show->set_title(buf);
			i->show->set_header(buf);

			skip_move(i->game.get_num());
			i->state = GS_NORMAL;
			i->show->input_line.connect(bind(game_input_kibitz.make_slot(), i->game.get_num()));
			i->show->set_play(i->game.get_white() == name, i->game.get_black() == name);
			if (i->game.get_white() == name || i->game.get_black() == name) { // am i playing
				my_match = i->game.get_num();
				i->show->input_handicap.connect(bind(game_input_handicap.make_slot(), my_match));
				i->show->input_put.connect(bind(game_input_put.make_slot(), my_match));
				i->show->input_pass.connect(bind(game_input_pass.make_slot(), my_match));
				i->show->input_undo.connect(bind(game_input_undo.make_slot(), my_match));
				i->show->input_terri.connect(bind(game_input_terri.make_slot(), my_match));
				i->show->input_reset.connect(bind(game_input_reset.make_slot(), my_match));
				i->show->input_done.connect(bind(game_input_done.make_slot(), my_match));

				i->show->input_adjourn.connect(bind(game_input_adjourn.make_slot(), my_match));
				i->show->input_resign.connect(bind(game_input_resign.make_slot(), my_match));
				// i->show->set_no_close();
			}
		}
		(* i->iter)[m_col.white] = i->game.get_white();
		(* i->iter)[m_col.black] = i->game.get_black();
		void * v;
		if ((v = (* i->iter)[m_col.w_rank])) delete (Player::Rank *) v;
		if ((v = (* i->iter)[m_col.b_rank])) delete (Player::Rank*) v;
		(* i->iter)[m_col.w_rank] = new Player::Rank(i->game.get_w_rank());
		(* i->iter)[m_col.b_rank] = new Player::Rank(i->game.get_b_rank());
		i->stamp = time(0);
		return;
	}

	Record r;

	r.game = g;
	r.state = GS_NORMAL;
	r.show = 0;
	r.scoring = false;

	r.iter = store->append();
	(* r.iter)[m_col.num] = r.game.get_num();
	(* r.iter)[m_col.white] = r.game.get_white();
	(* r.iter)[m_col.w_rank] = new Player::Rank(r.game.get_w_rank()); // need to clean up
	(* r.iter)[m_col.black] = r.game.get_black();
	(* r.iter)[m_col.b_rank] = new Player::Rank(r.game.get_b_rank()); // need to clean up

	r.stamp = time(0);
	list.push_back(r);

	update_n_games();
}

void GameList::end_update()
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->stamp < refresh_time) {
		if (i->show) continue;
		void * v;
		if ((v = (* i->iter)[m_col.w_rank])) delete (Player::Rank *) v;
		if ((v = (* i->iter)[m_col.b_rank])) delete (Player::Rank*) v;
		store->erase(i->iter);
		std::vector<Record>::iterator ii = i;
		i --; // TODO: make sure ((list.begin() - 1) + 1) == list.begin()!
		list.erase(ii);
	}

	update_n_games();
}

void GameList::add_move(int g, go::Move * m)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		if (i->state) return;  // NEED_WIN or CLOSE
		if (! i->show) { // no window
			i->state = GS_NEED_WIN;
			miss_game(g); // query the game info
			return;
		}
		if (int(m->get_num()) > i->show->move_num() + 1) {
			skip_move(g);
			return;
		}
		i->show->add_move(m);
		return;
	}
	Record r;
	r.game.set_num(g);
	r.state = GS_NEED_WIN;
	r.show = 0;
	r.scoring = false;

	r.iter = store->append();
	(* r.iter)[m_col.num] = g;
	(* r.iter)[m_col.w_rank] = 0;
	(* r.iter)[m_col.b_rank] = 0;

	r.stamp = time(0);
	list.push_back(r);
	miss_game(g);

	update_n_games();
}

void GameList::undo_move(int g)
{
	if (! g) {
		g = my_match;
	}
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		if (i->state || ! i->show) return;  // NEED_WIN or CLOSE or no show
		i->show->undo_move();
		return;
	}
}

void GameList::sync_time(int g, int wt, int wm, int bt, int bm)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		if (i->state) return;  // still waiting for window to open
		if (! i->show) { // no window
			i->state = GS_NEED_WIN;
			miss_game(g); // query the game info again
			return;
		}
		i->show->sync_time(wt, wm, bt, bm);
		return;
	}
	Record r;
	r.game.set_num(g);
	r.state = GS_NEED_WIN;
	r.show = 0;
	r.scoring = false;

	r.iter = store->append();
	(* r.iter)[m_col.num] = g;
	(* r.iter)[m_col.w_rank] = 0;
	(* r.iter)[m_col.b_rank] = 0;

	r.stamp = time(0);
	list.push_back(r);
	miss_game(g);

	update_n_games();
}

void GameList::add_title(int g, const std::string & t)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		if (i->state || ! i->show) return;
		i->show->set_header(t);
		return;
	}
}

void GameList::unob_game(int g)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		i->state = GS_NORMAL;
		if (i->show) {
			i->show->hang_up("delisted");
			i->show = 0;
		}
		break;
	}
}

void GameList::end_game(int g, const std::string & s)
{
	if (g == 0) g = my_match;
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		if (i->show) {
			i->show->hang_up(s);
		}
		void * v;
		if ((v = (* i->iter)[m_col.w_rank])) delete (Player::Rank *) v;
		if ((v = (* i->iter)[m_col.b_rank])) delete (Player::Rank*) v;
		store->erase(i->iter);
		list.erase(i);
		if (g == my_match) my_match = 0;
		break;
	}
}

void GameList::msg_game(int g, const std::string & s)
{
	if (g == 0) g = my_match;
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		if (i->show) {
			i->show->add_message(s);
		} else {
			std::cerr << "kibitz with no game: " << s << std::endl;
		}
		break;
	}
}

void GameList::game_score(int g)
{
	if (! g) g = my_match;
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		if (i->state || ! i->show) return;  // NEED_WIN or CLOSE or no show
		i->show->show_score();
		i->scoring = true;
		return;
	}
}

void GameList::game_restore(int g)
{
	if (! g) g = my_match;
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		if (i->state || ! i->show) return;  // NEED_WIN or CLOSE or no show
		i->show->show_restore();
		i->scoring = true;
		return;
	}
}

void GameList::game_remove(int g, const go::Loc & l)
{
	if (! g) g = my_match;
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		if (i->state || ! i->show) return;  // NEED_WIN or CLOSE or no show
		i->show->show_terri(l);
		i->scoring = true;
		return;
	}
}

void GameList::game_adjourn(int g)
{
	if (! g) g = my_match;
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		if (i->show) {
			i->show->hang_up("<game adjourned>");
		}
		void * v;
		if ((v = (* i->iter)[m_col.w_rank])) delete (Player::Rank *) v;
		if ((v = (* i->iter)[m_col.b_rank])) delete (Player::Rank*) v;
		store->erase(i->iter);
		list.erase(i);
		if (g == my_match) my_match = 0;
		break;

// 		if (i->state || ! i->show) return;  // NEED_WIN or CLOSE or no show
// 		i->show->show_adjourn();
// 		i->scoring = false;
// 		return;
	}
}

void GameList::game_board(const Board & b)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_white() == b. white && i->game.get_black() == b.black) {
		if (i->show) i->show->show_score_board(b);
		return;
	}
	// no on going games for the board, must be a stored game...
	Static * s = new Static(b.board_size);
	s->set_board(b);
}

void GameList::match_resign(const std::string & s)
{
	// std::cerr << "resign match" << std::endl;
	bool is_w;
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if ((is_w = i->game.get_white() == s) || i->game.get_black() == s) {
		// TODO: multiple matches
		if (i->state || ! i->show) return;
		i->show->hang_up(std::string(is_w ? "white" : "black") + " resigned");
		i->show = 0;
		i->state = GS_NORMAL;
		break;
	}
}

void GameList::match_timeup(const std::string & s)
{
	bool is_w;
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if ((is_w = i->game.get_white() == s) || i->game.get_black() == s) {
		// TODO: multiple matches
		if (i->state || ! i->show) return;
		i->show->hang_up(std::string(is_w ? "white" : "black") + " lost by time");
		i->show = 0;
		i->state = GS_NORMAL;
		break;
	}
}

void GameList::remove_match(const std::string & w, const std::string & b)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_white() == w || i->game.get_black() == b) {
		if (i->show) i->show->hang_up("game removed");
		void * v;
		if ((v = (* i->iter)[m_col.w_rank])) delete (Player::Rank *) v;
		if ((v = (* i->iter)[m_col.b_rank])) delete (Player::Rank*) v;
		store->erase(i->iter);
		list.erase(i);
		if (i->game.get_num() == my_match) my_match = 0;
		break;
	}
}

void GameList::wish_select_game(int g)
{
	for (std::vector<Record>::iterator i = list.begin(); i != list.end(); i ++) if (i->game.get_num() == g) {
		if (i->show) {
			i->show->present();
			return;
		}
		break;
	}
	game_selected(g);
}
