/* ccgo: igs/fork.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "fork.hh"
#include "../settings.hh"
#include <iostream>
#include <csignal>
extern "C" {
#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/wait.h>
#include <unistd.h>
}

using namespace igs;
using namespace std;

bool Fork::grab_line(Glib::IOCondition)
{
	/*
	  should grab the text line by line...
	 */
	if (! child_pid) return false;
	int s;
	// Check if child is dead
	if (waitpid(child_pid, & s, WNOHANG)) {
		grab_line_connection.disconnect();
		time_up_connection.disconnect();
		child_pid = 0;
		connect_ok = false;
		conn_lost();
		return false;
	}
	// connect to read_buffer
	grab_line_connection = Glib::signal_idle().connect(mem_fun(* this, & Fork::read_buffer));
	return false;
}

bool Fork::read_buffer()
{
	char b[256];
	int s = 0;
	s = read(from_child, b, 128); // only process 128 chars at a time, this slows the processing but makes the UI more responsive.
	if (s > 0) {
		idle_time = 0;
		char * sb = b;
		int p = 0;
		int i;
		for (i = 0; p < s; i ++) {
			if (iscntrl(sb[i])) { // line break at control characters
				line_buffer.append(sb, i);
				if (sb[i] == '\n' || sb[i] == '\r') {
					if (line_buffer.length()) { // drop empty lines
						recv_line(line_buffer);
						line_buffer = "";
					}
				} else if (sb[i] == 1) { // control sequence for telnet?!
					line_buffer = "";
				} else { // append others
					line_buffer.append(1, sb[i]);
				}
				sb += i + 1;
				i = - 1;
			}
			p ++;
		}
		line_buffer.append(sb, i); // put the rest in the line buffer
		return true;
	}
	if (check_prompt(line_buffer)) {
		recv_line(line_buffer);
		line_buffer = "";
	}
	// reconnect to grab_line
	grab_line_connection = Glib::signal_io().connect(mem_fun(* this, & Fork::grab_line), from_child, Glib::IO_IN | Glib::IO_HUP);
	return false;
}

bool Fork::time_up()
{
	if (idle_time < 30000) {
		idle_time += 100;
		if (idle_time == 3000 && line_buffer.length()) {
			std::cerr << "timeout empty buffer" << std::endl;
			recv_line(line_buffer);
			line_buffer = "";
		}
	}
	return true;
}

Fork::Fork(const std::string & n) :
	Connect()
{
	open_conn(n);
}

Fork::~Fork()
{
	if (child_pid != 0) {
		grab_line_connection.disconnect();
		time_up_connection.disconnect();
		kill(child_pid, SIGHUP);
		int s;
		waitpid(child_pid, & s, 0);
		child_pid = 0;
	}
}

void Fork::send_line(const std::string & s)
{
	if (go::settings.get_option(go::Settings::OPT_DUMP_IGS)) std::cout << s << std::endl;
	write(to_child, s.c_str(), s.length());
	write(to_child, "\n", 1);
}

bool Fork::open_conn(const std::string & n)
{
	connect_ok = false;
	int c2p[2];
	int p2c[2];
	pipe(c2p);
	pipe(p2c);
	child_pid = fork();
	if (child_pid == -1) return false;
	if (child_pid == 0) {
		close(0); // close cin
		dup(p2c[0]); // from parent
		close(1); // close cout
		dup(c2p[1]); // to parent
		close(p2c[0]);
		close(p2c[1]);
		close(c2p[0]);
		close(c2p[1]);
		close(2); // close cerr
		const char * shell = "/bin/bash";
		std::string c = std::string ("exec ") + n;
		execl(shell, shell, "-c", c.c_str(), (void *) 0);
		std::cerr << "Fail to fork: " << n << std::endl;
		exit(EXIT_FAILURE);
	}
	close(p2c[0]);
	to_child = p2c[1];
	from_child = c2p[0];
	close(c2p[1]);
	// set nonblock on reading child
	fcntl(from_child, F_SETFL, O_NONBLOCK);
	grab_line_connection = Glib::signal_io().connect(mem_fun(* this, & Fork::grab_line), from_child, Glib::IO_IN | Glib::IO_HUP);
	time_up_connection = Glib::signal_timeout().connect(mem_fun(* this, & Fork::time_up), 100);
	connect_ok = true;
	return true;
}

void Fork::close_conn()
{
	// just kill the child
	if (child_pid) {
		kill(child_pid, SIGHUP);
	}
}


