/* ccgo: go/gtk/gtk.cc
 * 
 * Copyright (C) 2004 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include <gtk.hh>
#include <go/sgf0.hh>
#include <debug.hh>

#include <config.hh>
extern "C" {
#include <gettext.h>
}
#include <cstdio>

#include <gtkmm/box.h>
#include <gtkmm/frame.h>
#include <gtkmm/stock.h>
#include <gtkmm/menubar.h>
#include <gtkmm/alignment.h>
#include <gtkmm/separator.h>
#include <gtkmm/spinbutton.h>
#include <gtkmm/scrolledwindow.h>
#include <gtkmm/radioaction.h>
#include <gconfmm/client.h>
#include <gtkmm/action.h>
#include <sstream>
#include <fstream>

#define _(String) Glib::locale_to_utf8(gettext(String))

using namespace gtk;

// members of class MarkWin

MarkWin::MarkWin() :
	Gtk::Window(Gtk::WINDOW_TOPLEVEL)
{
	Gtk::Frame * f = Gtk::manage(new Gtk::Frame);
	f->set_shadow_type(Gtk::SHADOW_OUT);
	add(* f);
	Gtk::VBox * vb = Gtk::manage(new Gtk::VBox);
	f->add(* vb);
	vb->pack_start(* Gtk::manage(new Gtk::Label(_("Use Mark:"))));
	Gtk::RadioButton::Group gp;
	rbs.push_back(Gtk::manage(new Gtk::RadioButton(gp, _("None"))));
	rbs.back()->signal_clicked().connect(bind(mem_fun(* this, & MarkWin::select_mark), ccgo::MARK_NONE));
	vb->pack_start(* rbs.back());
	rbs.push_back(Gtk::manage(new Gtk::RadioButton(gp, _("Circle"))));
	rbs.back()->signal_clicked().connect(bind(mem_fun(* this, & MarkWin::select_mark), ccgo::MARK_CIRCLE));
	vb->pack_start(* rbs.back());
	rbs.push_back(Gtk::manage(new Gtk::RadioButton(gp, _("Triangle"))));
	rbs.back()->signal_clicked().connect(bind(mem_fun(* this, & MarkWin::select_mark), ccgo::MARK_TRIANGLE));
	vb->pack_start(* rbs.back());
	rbs.push_back(Gtk::manage(new Gtk::RadioButton(gp, _("Square"))));
	rbs.back()->signal_clicked().connect(bind(mem_fun(* this, & MarkWin::select_mark), ccgo::MARK_SQUARE));
	vb->pack_start(* rbs.back());
	rbs.push_back(Gtk::manage(new Gtk::RadioButton(gp, _("Cross Mark"))));
	rbs.back()->signal_clicked().connect(bind(mem_fun(* this, & MarkWin::select_mark), ccgo::MARK_CROSS));
	vb->pack_start(* rbs.back());
	rbs.push_back(Gtk::manage(new Gtk::RadioButton(gp, _("Select"))));
	rbs.back()->signal_clicked().connect(bind(mem_fun(* this, & MarkWin::select_mark), ccgo::MARK_SELECT));
	vb->pack_start(* rbs.back());
	rbs.push_back(Gtk::manage(new Gtk::RadioButton(gp, _("White Territory"))));
	rbs.back()->signal_clicked().connect(bind(mem_fun(* this, & MarkWin::select_mark), ccgo::MARK_W_TERRITORY));
	vb->pack_start(* rbs.back());
	rbs.push_back(Gtk::manage(new Gtk::RadioButton(gp, _("Black Territory"))));
	rbs.back()->signal_clicked().connect(bind(mem_fun(* this, & MarkWin::select_mark), ccgo::MARK_B_TERRITORY));
	vb->pack_start(* rbs.back());

	vb->pack_start(* Gtk::manage(new Gtk::HSeparator));

	// entry
	vb->pack_start(* Gtk::manage(new Gtk::Label(_("Use Label:"))));
	vb->pack_start(et);
	et.set_width_chars(3);
	et.signal_activate().connect(mem_fun(* this, & MarkWin::enter_label));

	vb->pack_start(* Gtk::manage(new Gtk::HSeparator));
	Gtk::Button * bt = Gtk::manage(new Gtk::Button(Gtk::Stock::CLOSE));
	bt->signal_clicked().connect(mem_fun(* this, & Gtk::Window::hide));
	vb->pack_end(* bt);
}

void MarkWin::show_win(int x, int y, std::string loc)
{
	if (loc == "") set_title(_("Marking"));
	else {
		char buf[100];
		snprintf(buf, 100, _("Marking at %s").c_str(), loc.c_str());
		set_title(buf);
	}
	show_all();
	int xx;
	int yy;
	get_size(xx, yy);
	move(x - xx / 2, y - yy / 2);
}

void MarkWin::set_mark(ccgo::Mark marking)
{
	m = marking;
	rbs[m]->set_active();
	l = "";
	et.set_text(l);
}

void MarkWin::set_label(std::string label)
{
	m = ccgo::MARK_NONE;
	rbs[m]->set_active();
	l = label;
	et.set_text(l);
}

ccgo::Mark MarkWin::get_mark()
{
	return m;
}

const std::string & MarkWin::get_label()
{
	return l;
}

void MarkWin::select_mark(ccgo::Mark marking)
{
	if (! rbs[marking]->get_active()) return;
	m = marking;
	l = "";
	hide();
}

void MarkWin::enter_label()
{
	msg(DBG_DEBUG) << "label = " << et.get_text() << '\n';
	l = et.get_text();
	m = ccgo::MARK_NONE;
	hide();
}

// members of class View
namespace {
	const char * act_name[] = {
		"/MenuBar/GameMenu/GameMode/ModeView",
		"/MenuBar/GameMenu/GameMode/ModeInit",
		"/MenuBar/GameMenu/GameMode/ModePlay",
		"/MenuBar/GameMenu/GameMode/ModeMark",
		"/MenuBar/GameMenu/GameMode/ModeSetup",
		"/MenuBar/GameMenu/GameMode/ModeScore"};
}

View::View() :
	b(0),
	mark_last(true),
	show_ko(true),
	dp("---"),
	sv_w(310),
	sv_h(350),
	sv_p(250),
	mode(MODE_VIEW),
	persist_mode(mode),
	mk_loc(ccgo::MOVE_NONE),
	terri(0),
	my_game(0),
	list(0)
{
	Glib::RefPtr<Gnome::Conf::Client> cf = Gnome::Conf::Client::get_default_client();
	sv_w = cf->get_int("/apps/ccgo/gtk-win-width");
	sv_h = cf->get_int("/apps/ccgo/gtk-win-height");
	sv_p = cf->get_int("/apps/ccgo/gtk-win-paned");

	set_title("gtk");
	set_default_size(sv_w, sv_h);
	Gtk::VBox * vb = Gtk::manage(new Gtk::VBox);
	add(* vb);

	// board
	b = new gtk::Board;
	b->set_board(this);
	b->clicked.connect(mem_fun(* this, & View::board_clicked));



	m_act = Gtk::ActionGroup::create();
	m_init = Gtk::ActionGroup::create();
	m_act->add(Gtk::Action::create("FileMenu", _("_File")));
	m_act->add(Gtk::Action::create("FileOpen", Gtk::Stock::OPEN),  mem_fun(* this, & View::file_open));
	m_act->add(Gtk::Action::create("FileSave", Gtk::Stock::SAVE), mem_fun(* this, & View::file_save));
	m_act->add(Gtk::Action::create("FileClose", Gtk::Stock::CLOSE), mem_fun(* this, & View::hide));

	m_act->add(Gtk::Action::create("GameMenu", _("_Game")));
	m_init->add(Gtk::Action::create("GameInit", _("_Initialize")));
	for (int i = 2; i < 10; i ++) {
		char bn[100];
		char buf[100];
		snprintf(bn, 100, "Handi%d", i);
		snprintf(buf, 100, _("_%d Handicap Stones").c_str(), i);
		m_init->add(Gtk::Action::create(bn, buf), bind(mem_fun(* this, & View::game_handicap), i));
	}
	m_init->add(Gtk::Action::create("GameInitClear", _("_Clear Handicaps")), bind(mem_fun(* this, & View::game_handicap), - 1));
	m_act->add(Gtk::Action::create("GameMode", _("Game _Mode")));
	Gtk::RadioAction::Group g_mode;
	m_act->add(Gtk::RadioAction::create(g_mode, "ModeView", _("_View Only")), bind(mem_fun(* this, & View::game_mode), MODE_VIEW));
	m_init->add(Gtk::RadioAction::create(g_mode, "ModeInit", _("Handicap _Initialization")), bind(mem_fun(* this, & View::game_mode), MODE_INIT));
	m_act->add(Gtk::RadioAction::create(g_mode, "ModePlay", _("_Play Moves")), bind(mem_fun(* this, & View::game_mode), MODE_PLAY));
	m_act->add(Gtk::RadioAction::create(g_mode, "ModeMark", _("Place _Marks")), bind(mem_fun(* this, & View::game_mode), MODE_MARK));
	m_act->add(Gtk::RadioAction::create(g_mode, "ModeSetup", _("_Setup Board")), bind(mem_fun(* this, & View::game_mode), MODE_SETUP));
	m_act->add(Gtk::RadioAction::create(g_mode, "ModeScore", _("Sco_re Board")), bind(mem_fun(* this, & View::game_mode), MODE_SCORE));
	m_act->add(Gtk::Action::create("GamePass", "_Pass"), mem_fun(* this, & View::game_pass));

	m_act->add(Gtk::Action::create("GameNewNode", _("Create _Non-Move Node")), Gtk::AccelKey("<control>n"), mem_fun(* this, & View::create_node));
	m_act->add(Gtk::Action::create("GameDelNode", _("_Delete Node Branch")), Gtk::AccelKey("<control>d"), mem_fun(* this, & View::delete_moves));
	m_act->add(Gtk::Action::create("GameClearMark", _("_Clear Marks")), mem_fun(* this, & View::clear_marks));

	m_act->add(Gtk::Action::create("ViewMenu", _("_View")));
	m_act->add(Gtk::ToggleAction::create("ViewLast", _("_Mark Last Move")), Gtk::AccelKey("<control>m"), mem_fun(* this, & View::toggle_mark_last));
	m_act->add(Gtk::ToggleAction::create("ViewKo", _("Show _Ko")), Gtk::AccelKey("<control>k"), mem_fun(* this, & View::toggle_show_ko));
	m_act->add(Gtk::Action::create("ViewBoard", _("Board")));

	Glib::RefPtr<Gtk::Action> rec = Gtk::Action::create("RecordMenu", _("_Record"));
	// rec->set_sensitive(false);
	// rec->property_hide_if_empty().set_value(false);
	rec->signal_activate().connect(mem_fun(* this, & View::check_list));
	m_act->add(rec);

	m_init->set_sensitive(false);
	m_uim = Gtk::UIManager::create();
	m_uim->insert_action_group(m_act);
	m_uim->insert_action_group(m_init);
	m_uim->insert_action_group(b->get_configs());

	try {
		Glib::ustring ui_info =
			"<ui>"
			"  <menubar name='MenuBar'>"
			"    <menu action='FileMenu'>"
			"      <menuitem action='FileOpen'/>"
			"      <menuitem action='FileSave'/>"
			"      <menuitem action='FileClose'/>"
			"    </menu>"
			"    <menu action='GameMenu'>"
			"      <menu action='GameInit'>"
			"        <menuitem action='Handi2'/>"
			"        <menuitem action='Handi3'/>"
			"        <menuitem action='Handi4'/>"
			"        <menuitem action='Handi5'/>"
			"        <menuitem action='Handi6'/>"
			"        <menuitem action='Handi7'/>"
			"        <menuitem action='Handi8'/>"
			"        <menuitem action='Handi9'/>"
			"        <separator/>"
			"        <menuitem action='GameInitClear'/>"
			"      </menu>"
			"      <menu action='GameMode'>"
			"        <menuitem action='ModeView'/>"
			"        <menuitem action='ModeInit'/>"
			"        <menuitem action='ModePlay'/>"
			"        <menuitem action='ModeMark'/>"
			"        <menuitem action='ModeSetup'/>"
			"        <menuitem action='ModeScore'/>"
			"      </menu>"
			"      <separator/>"
			"      <menuitem action='GamePass'/>"
			"      <separator/>"
			"      <menuitem action='GameNewNode'/>"
			"      <menuitem action='GameDelNode'/>"
			"      <menuitem action='GameClearMark'/>"
			"    </menu>"
			"    <menu action='ViewMenu'>"
			"      <menuitem action='ViewLast'/>"
			"      <menuitem action='ViewKo'/>"
			"      <menu action='ViewBoard'>"
			"      </menu>"
			"    </menu>"
			"    <menu action='RecordMenu'>"
			"    </menu>"
			"  </menubar>"
			"</ui>";
		m_uim->add_ui_from_string(ui_info);
		const Glib::ListHandle<Glib::RefPtr<Gtk::Action> > & act = b->get_configs()->get_actions();
		Gtk::UIManager::ui_merge_id mid = m_uim->new_merge_id();
		for (Glib::ListHandle<Glib::RefPtr<Gtk::Action> >::const_iterator i = act.begin(); i != act.end(); i ++) {
			Glib::ustring n = (* i)->get_name();
			m_uim->add_ui(mid, "/MenuBar/ViewMenu/ViewBoard", n, n, Gtk::UI_MANAGER_AUTO, false);
			msg(DBG_DEBUG) << "add ui " << n << '\n';
		}
	}  catch (const Glib::Error & ex) {
		std::cerr << "building menus failed: " << ex.what();
	}

	Gtk::Widget * pm = m_uim->get_widget("/MenuBar");
	if (pm) {
		vb->pack_start(* pm, Gtk::PACK_SHRINK);
		add_accel_group(m_uim->get_accel_group());
	}

	vb->pack_end(vp);

	Gtk::HBox * hb = Gtk::manage(new Gtk::HBox);
	Gtk::Frame * fr = Gtk::manage(new Gtk::Frame);
	fr->set_shadow_type(Gtk::SHADOW_IN);
	vp.add1(* fr);
	fr->add(* hb);

	hb->pack_start(* b);

	// control box
	Gtk::VBox * vb2 = Gtk::manage(new Gtk::VBox);
	hb->pack_end(* vb2, Gtk::PACK_SHRINK);

	Gtk::Alignment * al = Gtk::manage(new Gtk::Alignment(0.5, 0.5, 0, 0));
	// map button
	al->add(btM);
	vb2->pack_start(* al, Gtk::PACK_SHRINK);

	// move up and down
	Gtk::SpinButton * sb = Gtk::manage(new Gtk::SpinButton());
	sb->set_increments(- 1, - 10);
	sb->set_numeric();
	prog = sb->get_adjustment();
	prog->set_lower(0);
	prog->signal_value_changed().connect(mem_fun(* this, & View::depth_change));
	vb2->pack_start(* sb, Gtk::PACK_SHRINK);

	// move number
	vb2->pack_start(dp, Gtk::PACK_SHRINK);

	// left and right
	al = Gtk::manage(new Gtk::Alignment(0.5, 0.5, 0, 0));
	vb2->pack_start(* al, Gtk::PACK_SHRINK);
	hb = Gtk::manage(new Gtk::HBox);
	al->add(* hb);
	hb->pack_start(btL, Gtk::PACK_SHRINK);
	// hb->pack_start(btM, Gtk::PACK_SHRINK);
	hb->pack_end(btR, Gtk::PACK_SHRINK);
	btL.add(* Gtk::manage(new Gtk::Image(Gtk::Stock::GO_BACK, Gtk::ICON_SIZE_SMALL_TOOLBAR)));
	// btM.add(* Gtk::manage(new Gtk::Label("M")));
	btM.add(daM);
	btR.add(* Gtk::manage(new Gtk::Image(Gtk::Stock::GO_FORWARD, Gtk::ICON_SIZE_SMALL_TOOLBAR)));
	btL.signal_clicked().connect(mem_fun(* this, & View::alternative_left));
	btR.signal_clicked().connect(mem_fun(* this, & View::alternative_right));
	btM.signal_toggled().connect(mem_fun(* this, & View::map_toggle));
	daM.signal_expose_event().connect(mem_fun(* this, & View::map_handle_draw));

	// information
	vb2->pack_start(* Gtk::manage(new Gtk::Label(_("score"))), Gtk::PACK_SHRINK);
	al = Gtk::manage(new Gtk::Alignment(0.5, 0.5, 0, 0));
	vb2->pack_start(* al, Gtk::PACK_SHRINK);
	hb = Gtk::manage(new Gtk::HBox);
	al->add(* hb);
	hb->pack_start(* Gtk::manage(new Gtk::Label(_("B:"))), Gtk::PACK_SHRINK);
	hb->pack_start(b_cap, Gtk::PACK_SHRINK);
	al = Gtk::manage(new Gtk::Alignment(0.5, 0.5, 0, 0));
	vb2->pack_start(* al, Gtk::PACK_SHRINK);
	hb = Gtk::manage(new Gtk::HBox);
	al->add(* hb);
	hb->pack_start(* Gtk::manage(new Gtk::Label(_("W:"))), Gtk::PACK_SHRINK);
	hb->pack_start(w_cap, Gtk::PACK_SHRINK);

	// mode options

	vb2->pack_start(o_mode, Gtk::PACK_SHRINK);
	Gtk::Menu * om = Gtk::manage(new Gtk::Menu);
	o_mode.set_menu(* om);
	om->items().push_back(Gtk::Menu_Helpers::MenuElem(_("View")));
	om->items().push_back(Gtk::Menu_Helpers::MenuElem(_("Init")));
	om->items().push_back(Gtk::Menu_Helpers::MenuElem(_("Move")));
	om->items().push_back(Gtk::Menu_Helpers::MenuElem(_("Mark")));
	om->items().push_back(Gtk::Menu_Helpers::MenuElem(_("Set")));
	om->items().push_back(Gtk::Menu_Helpers::MenuElem(_("Score")));
	o_mode.set_history(mode);
	o_mode.signal_changed().connect(bind(mem_fun(* this, & View::game_mode), MODE_UNKNOWN));

	// turn options
	al = Gtk::manage(new Gtk::Alignment(0.5, 0.5, 0, 0));
	al->add(o_turn);
	vb2->pack_start(* al, Gtk::PACK_SHRINK);
	om = Gtk::manage(new Gtk::Menu);
	o_turn.set_menu(* om);
	om->items().push_back(Gtk::Menu_Helpers::MenuElem(_("B")));
	om->items().push_back(Gtk::Menu_Helpers::MenuElem(_("W")));
	o_turn.set_history(get_turn() - 1);
	o_turn.signal_changed().connect(mem_fun(* this, & View::game_turn_change));
	o_turn.set_sensitive(mode == MODE_SETUP);

	// text box
	Gtk::ScrolledWindow * sw = Gtk::manage(new Gtk::ScrolledWindow);
	sw->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_ALWAYS);
	sw->add(tv);
	tv.set_wrap_mode(Gtk::WRAP_WORD);
	fr = Gtk::manage(new Gtk::Frame);
	fr->set_shadow_type(Gtk::SHADOW_IN);
	vp.pack2(* fr, false, true);
	fr->add(* sw);
	fr->signal_size_allocate().connect(mem_fun(* this, & View::paned_moved));

	// game map
	mw.signal_hide().connect(mem_fun(* this, & View::map_hide));
	mw.set_transient_for(* this);
	mw.set_modal(true);
	mw.gm.node_select.connect(mem_fun(* this, & View::select_node));

	// markwin
	mkw.set_transient_for(* this);
	mkw.set_modal(true);
	mkw.signal_hide().connect(mem_fun(* this, & View::mark_set));

	// file dialog
	file_win.set_transient_for(* this);
	file_win.set_modal(true);
	// file_win.signal_response().connect(mem_fun(* this, & View::file_selected));

	vp.set_position(sv_p);
}

View::~View()
{
	if (where()) PathWalk::set_node(0);
	if (b) delete b;
	if (terri) delete [] terri;
	if (my_game) {
		delete my_game;
	}
}

void View::set_node(ccgo::Node * node)
{
	msg(DBG_DEBUG) << "setting node\n";
	if (where()) do_old_node();
	PathWalk::set_node(node);
		b->set_board(this); // size might be changed
	if (where()) {
		std::string t = "ccgo: ";
		set_title(t + where()->get_game()->get_name());
		do_new_node();
		b->update();
	} else {
		clear();
	}
}

void View::up()
{
	do_old_node();
	PathWalk::up();
	do_new_node();
}

void View::down(unsigned index)
{
	do_old_node();
	PathWalk::down(index);
	do_new_node();
}

void View::down(ccgo::Node * node)
{
	do_old_node();
	PathWalk::down(node);
	do_new_node();
}

void View::do_geom()
{
	set_max_depth();
	set_alternative();
	switch (mode) {
	case MODE_INIT:
		if (! game_can_init()) {
			Glib::RefPtr<Gtk::Action> a = m_uim->get_action(act_name[persist_mode]);
			assert(a);
			a->activate();
		}
		break;
	case MODE_SETUP:
		if (! node_can_setup()) {
			Glib::RefPtr<Gtk::Action> a = m_uim->get_action(act_name[persist_mode]);
			assert(a);
			a->activate();
		}
		break;
	default:
		break;
	}

	if (game_can_init()) {
		m_init->set_sensitive(true);
		o_mode.get_menu()->items()[MODE_INIT].set_sensitive(true);
	} else {
		assert(mode != MODE_INIT);
		m_init->set_sensitive(false);
		o_mode.get_menu()->items()[MODE_INIT].set_sensitive(false);
	}
}

void View::set_list(ccgo::List * l)
{
	msg(DBG_DEBUG) << "setting record list\n";
	list = l;
	if (is_realized()) check_list();
}

void View::on_realize()
{
	Gtk::Window::on_realize();

	Glib::RefPtr<Gtk::ToggleAction> ta;
	ta = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(m_act->get_action("ViewLast"));
	ta->set_active(mark_last);
	ta = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(m_act->get_action("ViewKo"));
	ta->set_active(show_ko);

	if (list) check_list();
	msg(DBG_DEBUG) << "View realized\n";

	// sv_w = get_width();
	// sv_h = get_height();
	// sv_p = vp.get_position();
}

bool View::on_configure_event(GdkEventConfigure * event)
{
	if (! Gtk::Window::on_configure_event(event)) return false;
	int w;
	int h;
	create_pango_layout("999")->get_pixel_size(w, h);
	daM.set_size_request(w, h);
	// btM.resize_children();
	// btM.check_resize();
	return true;
}

void View::on_size_allocate(Gtk::Allocation & allocation)
{
	Gtk::Window::on_size_allocate(allocation);
	if (allocation.get_width() != sv_w) {
		sv_w = allocation.get_width();
		Gnome::Conf::Client::get_default_client()->set("/apps/ccgo/gtk-win-width", sv_w);
	}
	if (allocation.get_height() != sv_h) {
		sv_h = allocation.get_height();
		Gnome::Conf::Client::get_default_client()->set("/apps/ccgo/gtk-win-height", sv_h);
	}
}

void View::clear()
{
	msg(DBG_DEBUG) << "clearing view\n";
	if (Board::is_valid()) PathWalk::clear();
	b->update();
}

void View::set_state(int loc, State state)
{
	PathWalk::set_state(loc, state);
	b->update(loc);
}

// void View::set_ko(const std::set<int> & ko)
// {
// 	PathWalk::set_ko(ko);
// 	for (std::set<int>::const_iterator i = ko.begin(); i != ko.end(); i ++) b->update(* i);
// }

void View::toggle_mark_last()
{
	Glib::RefPtr<Gtk::ToggleAction> ta = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(m_act->get_action("ViewLast"));
	mark_last = ta->get_active();
	if (where()->get_move() == ccgo::MOVE_NONE) return;
	if (mark_last) b->add_mark(where()->get_move(), ccgo::MARK_CIRCLE);
	else b->add_mark(where()->get_move(), ccgo::MARK_NONE);
}

void View::toggle_show_ko()
{
	Glib::RefPtr<Gtk::ToggleAction> ta = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(m_act->get_action("ViewKo"));
	show_ko = ta->get_active();
	const std::set<int> & k = where()->get_ko();
	ccgo::Mark m = show_ko ? ccgo::MARK_CROSS :  ccgo::MARK_NONE;
	for (std::set<int>::const_iterator i = k.begin(); i != k.end(); i ++) b->add_mark(* i, m);
}

void View::set_max_depth()
{
	msg(DBG_DEBUG) << "setting max depth\n";
        int r = path_depth();
	prog->set_upper(r);
	std::ostringstream s;
	s << r;
	dp.set_text(s.str());
}

void View::depth_change()
{
	unsigned d = unsigned(prog->get_value());
	if (where()->get_level() == d) return;

	msg(DBG_DEBUG) << "changing level to " << d << "\n";
	do_old_node();

	while (where()->get_level() > d) PathWalk::up();
	while (where()->get_level() < d) PathWalk::down();

	do_new_node();
}

void View::set_alternative()
{
	// if (! is_realized()) return;
	if (where()->get_level()) {
		unsigned i = where()->get_index();
		btL.set_sensitive(i > 0);
		btR.set_sensitive(i + 1 < where()->up()->down_num());
	} else {
		btL.set_sensitive(false);
		btR.set_sensitive(false);
	}
}

void View::alternative_left()
{
	unsigned i = where()->get_index();
	if (i < 1) return;

	do_old_node();

	PathWalk::up();
	PathWalk::down(i - 1);

	do_new_node();
}

void View::alternative_right()
{
	unsigned i = where()->get_index();
	if (i + 1 >= where()->up()->down_num()) return;

	do_old_node();

	PathWalk::up();
	PathWalk::down(i + 1);

	do_new_node();
}

void View::board_clicked(int loc)
{
	int i;
	msg(DBG_DEBUG) << "board clicked at " << loc << '\n';

	switch (mode) {
	case MODE_VIEW:
		break;
	case MODE_INIT:
		game_add_handi(loc);
		break;
	case MODE_PLAY:
		i = game_add_move(loc);
		if (i < 0) {
			msg(DBG_INFO) << "move not allowed\n";
			break;
		}
		down(i);
		break;
	case MODE_MARK:
		{
			int x;
			int y;
			get_pointer(x, y);
			int x0;
			int y0;
			get_window()->get_origin(x0, y0);
			mk_loc = loc;


			const ccgo::Marking * mk = dynamic_cast<const ccgo::Marking *>(where()->get_prop(ccgo::Marking::the_id()));
			if (mk) mkw.set_mark(mk->get_mark(loc));
			else mkw.set_mark(ccgo::MARK_NONE);

			const ccgo::Labels * lb = dynamic_cast<const ccgo::Labels *>(where()->get_prop(ccgo::Labels::the_id()));
			if (lb) {
				const std::string & s = lb->get_label(loc);
				if (s != "") mkw.set_label(s);
			}
			b->add_mark(loc, ccgo::MARK_SELECT);
			mkw.show_win(x0 + x, y0 + y, loc_to_str(loc));
		}
		break;
	case MODE_SETUP:
		node_set_site(loc, ccgo::Board::State((Board::get_state(loc) + 1) % 3));
		draw_map_handle();
		break;
	case MODE_SCORE:
		{
			assert(Board::get_state(loc));
			std::vector<int> g;
			dead_group(loc, g);
			Terri t;
			ccgo::Mark m;
			if (terri[loc] == TERRI_NEUTRAL) {
				t = Board::get_state(loc) == STATE_BLACK ? TERRI_WHITE : TERRI_BLACK;
				m = t == TERRI_BLACK ? ccgo::MARK_B_TERRITORY : ccgo::MARK_W_TERRITORY;
			} else {
				t = TERRI_NEUTRAL;
				m = ccgo::MARK_NONE;
			}
			for (std::vector<int>::iterator i = g.begin(); i != g.end(); i ++) {
				terri[* i] = t;
			}
			msg(DBG_INFO) << (m ? "mark dead " : "mark alive ") << g.size() << " stones\n";
			do_score();
		}
		break;
	default:
		break;
	}
}

void View::delete_moves()
{
	if (! where()->up()) return;
	do_old_node();
	game_del_node();
	do_new_node();
}

void View::create_node()
{
	int i;
	msg(DBG_DEBUG) << "before create_node() mode = " << mode << '\n';
	i = game_add_setup();
	down(i);
	msg(DBG_DEBUG) << "after create_node() mode = " << mode << '\n';
}

void View::do_old_node()
{
	switch (mode) { // ending some modes
	case MODE_SCORE:
		{
			Glib::RefPtr<Gtk::Action> a = m_uim->get_action(act_name[persist_mode]);
			assert(a);
			a->activate();
		}
		break;
	default:
		break;
	}
	b->clear_marks();
	b->clear_labels();
	b->clear_blocks();
	std::string s = tv.get_buffer()->get_text();
	if (s == "") where()->set_prop(ccgo::Comment::the_id(), 0);
	else {
		ccgo::Comment * c = new ccgo::Comment;
		c->set_comment(s);
		where()->set_prop(ccgo::Comment::the_id(), c);
	}
	// remove empty Marking prop
	const ccgo::Marking * mk = dynamic_cast<const ccgo::Marking *>(where()->get_prop(ccgo::Marking::the_id()));
	if (mk && ! mk->get_marks().size()) {
		where()->set_prop(ccgo::Marking::the_id(), 0);
	}
	// remove empty Labels prop
	const ccgo::Labels * lb = dynamic_cast<const ccgo::Labels *>(where()->get_prop(ccgo::Labels::the_id()));
	if (lb && ! lb->get_labels().size()) {
		where()->set_prop(ccgo::Labels::the_id(), 0);
	}
}

void View::do_new_node()
{
	assert(where());
	set_max_depth();
	prog->set_value(where()->get_level());
	set_alternative();
	b->set_track(gtk::Board::TrackMode(get_turn()));

	switch (mode) {
	case MODE_VIEW:
		b->set_track(gtk::Board::TRACK_NONE);
		break;
	case MODE_INIT:
		b->set_track(gtk::Board::TRACK_BLACK);
		break;
	case MODE_PLAY:
		b->set_track(gtk::Board::TrackMode(get_turn()));
		break;
	case MODE_MARK:
		b->set_track(gtk::Board::TRACK_ALL);
		break;
	case MODE_SETUP:
		if (! node_can_setup()) { // can not setup
			msg(DBG_DEBUG) << "node can not setup, changing mode to " << persist_mode << '\n';
			Glib::RefPtr<Gtk::Action> a = m_uim->get_action(act_name[persist_mode]);
			assert(a);
			a->activate();
		} else b->set_track(gtk::Board::TRACK_ALL);
		break;
	case MODE_SCORE:
		// impossible
		break;
	default:
		break;
	}

	// last move
	if (mark_last && where()->get_move() >= 0) b->add_mark(where()->get_move(), ccgo::MARK_CIRCLE);
	// ko
	if (show_ko) {
		const std::set<int> & k = where()->get_ko();
		for (std::set<int>::const_iterator i = k.begin(); i != k.end(); i ++) {
			b->add_mark(* i, ccgo::MARK_CROSS);
		}
	}
	ccgo::Comment * c = (ccgo::Comment *)(where()->get_prop(ccgo::Comment::the_id()));
	if (c) {
		tv.get_buffer()->set_text(c->get_comment());
	} else {
		tv.get_buffer()->set_text("");
	}

	ccgo::Labels * l = (ccgo::Labels *)(where()->get_prop(ccgo::Labels::the_id()));
	if (l) {
		const std::map<int, std::string> & lb  = l->get_labels();
		for (std::map<int, std::string>::const_iterator i = lb.begin(); i != lb.end(); i ++) {
			msg(DBG_DEBUG) << i->first << ':' << i->second << "\n";
			b->add_label(i->first, i->second);
		}
	}
	double bs = 0;
	double ws = 0;
	ccgo::Marking * m = (ccgo::Marking *)(where()->get_prop(ccgo::Marking::the_id()));
	if (m) {
		const std::map<int, ccgo::Mark> & mk = m->get_marks();
		for (std::map<int, ccgo::Mark>::const_iterator i = mk.begin(); i != mk.end(); i ++) {
			b->add_mark(i->first, i->second);
			// counting territory
			switch (i->second) {
			case ccgo::MARK_B_TERRITORY:
				switch (Board::get_state(i->first)) {
				case STATE_WHITE:
					bs ++;
					break;
				case STATE_BLACK:
					ws ++;
					break;
				case STATE_EMPTY:
					break;
				}
				bs ++;
				break;
			case ccgo::MARK_W_TERRITORY:
				switch (Board::get_state(i->first)) {
				case STATE_WHITE:
					bs ++;
					break;
				case STATE_BLACK:
					ws ++;
					break;
				case STATE_EMPTY:
					break;
				}
				ws ++;
				break;
			default:
				break;
			}
		}
	}
	const std::set<int> & ko = where()->get_ko();
	for (std::set<int>::const_iterator i = ko.begin(); i != ko.end(); i ++) {
		b->set_block(* i, true);
	}

	draw_map_handle();

	if (game_can_init()) {
		m_init->set_sensitive(true);
		o_mode.get_menu()->items()[MODE_INIT].set_sensitive(true);
	} else {
		if (mode == MODE_INIT) {
			Glib::RefPtr<Gtk::Action> a = m_uim->get_action(act_name[persist_mode]);
			assert(a);
			a->activate();
		}
		m_init->set_sensitive(false);
		o_mode.get_menu()->items()[MODE_INIT].set_sensitive(false);
	}
	o_turn.set_history(get_turn() - 1);

	// information
	std::ostringstream s;
	bs += get_b_cap();
	ws += get_w_cap();
	double km = where()->get_game()->get_komi();
	if (km > 0) ws += km;
	else bs -= km;
	s << bs;
	b_cap.set_text(s.str());
	s.str("");
	s << ws;
	w_cap.set_text(s.str());
	if (list) list->set_node(where());
}

void View::paned_moved(Gtk::Allocation & allocation)
{
	if (vp.get_position() == sv_p) return;
	sv_p = vp.get_position();
	Gnome::Conf::Client::get_default_client()->set("/apps/ccgo/gtk-win-paned", sv_p);
	msg(DBG_DEBUG) << "paned moved: height = " << allocation.get_height()
		       << "; position = " << sv_p << "\n";
}

void View::map_toggle()
{
	if (btM.get_active()) {
		int x;
		int y;
		int x1;
		int y1;
		mw.show_map(where(), where()->get_level(), & x1, & y1);
		// reposition the window
		daM.get_window()->get_origin(x, y);
		Gtk::Allocation r = daM.get_allocation();
		x += r.get_width() / 2;
		y += r.get_height() / 2;
		mw.move_map(x - x1, y - y1);
	} else mw.hide();
}

void View::map_hide()
{
	if (btM.get_active()) btM.set_active(false);
}

bool View::map_handle_draw(GdkEventExpose * event)
{
	draw_map_handle();
	return true;
}

void View::draw_map_handle()
{
	if (! daM.is_realized()) return;
	Gtk::Allocation r = daM.get_allocation();
	int cx = r.get_width() / 2;
	int cy = r.get_height() / 2;
	daM.get_window()->clear_area(0, 0, r.get_width(), r.get_height());
	if (where()->get_move() != ccgo::MOVE_NONE) { // played node
		std::string s;
		if (where()->get_move() >= 0) s = loc_to_str(where()->get_move());
		else s = "pa.";
		Glib::RefPtr<Pango::Layout> t = daM.create_pango_layout(s);
		int w;
		int h;
		t->get_pixel_size(w, h);
		if (get_turn() == ccgo::Position::TURN_BLACK) {
			daM.get_window()->draw_rectangle(daM.get_style()->get_white_gc(), true, cx - w / 2 - 1, cy - h / 2 - 1, w + 1, h + 1);
			daM.get_window()->draw_layout(daM.get_style()->get_black_gc(), cx - w / 2, cy - h / 2, t);
		} else {
			daM.get_window()->draw_rectangle(daM.get_style()->get_black_gc(), true, cx - w / 2 - 1, cy - h / 2 - 1, w + 1, h + 1);
			daM.get_window()->draw_layout(daM.get_style()->get_white_gc(), cx - w / 2, cy - h / 2, t);
		}
	} else if (where()->get_delta()) { // setup
		Glib::RefPtr<Pango::Layout> t = daM.create_pango_layout("-/+");
		int w;
		int h;
		t->get_pixel_size(w, h);
		daM.get_window()->draw_layout(daM.get_style()->get_fg_gc(daM.get_state()), cx - w / 2, cy - h / 2, t);
	} else if (where()->up() == 0) { // start
		std::string s = "[S]";
		ccgo::Game * g = dynamic_cast<ccgo::Game *>(where());
		assert(g);
		if (g->get_init() && g->get_init()->get_turn() == ccgo::Position::TURN_WHITE) {
			s = "[H]";
		}
		Glib::RefPtr<Pango::Layout> t = daM.create_pango_layout(s);
		int w;
		int h;
		t->get_pixel_size(w, h);
		daM.get_window()->draw_layout(daM.get_style()->get_fg_gc(daM.get_state()), cx - w / 2, cy - h / 2, t);		
	}
}

void View::select_node(ccgo::Node * node)
{
	mw.hide();

	assert(node->get_game() == where()->get_game());
	if (node == where()) return;
	// move to node
	// find path to node
	// do_old_node();
	set_node(node);
	// do_new_node();
}

void View::file_open()
{
	file_win.set_title("Opening SGF file");
	file_win.show_all();
	file_win.complete("*.sgf");
	int result = file_win.run();
	file_win.hide();
	if (result == Gtk::RESPONSE_OK) {
		std::string s = file_win.get_filename();
		msg(DBG_INFO) << "opening SGF file " << s << '\n';
		if (list) {
			ccgo::Game * g = list->load_game(s);
			if (g) {
				set_node(g);
				if (my_game) {
					delete my_game;
					my_game = 0;
				}
			} else {
				msg(DBG_ERROR) << "fail to load SGF file " << s << '\n';
			}
		} else { // no list
			std::ifstream f(s.c_str()); // , std::ios::in);
			ccgo::Game * g = 0;
			if (f && (g = ccgo::load_sgf(f))) {
				set_node(g);
				if (my_game) delete my_game;
				my_game = g;
			} else {
				msg(DBG_ERROR) << "fail to load SGF file " << s << '\n';
			}
		}
	}
}

void View::file_save()
{
	file_win.set_title("Saving SGF file");
	file_win.show_all();
	file_win.complete("*.sgf");
	int result = file_win.run();
	file_win.hide();
	if (result == Gtk::RESPONSE_OK) {
		std::string s = file_win.get_filename();
		msg(DBG_INFO) << "saving SGF file " << s << '\n';
		std::ofstream f(s.c_str()); // , std::ios::out);
		if (f) {
			ccgo::save_sgf(where()->get_game(), f);
		} else {
			msg(DBG_ERROR) << "can open file " << s << " for SGF save \n";
		}
	}
}

void View::game_pass()
{
	int i = game_add_move(ccgo::MOVE_PASS);
	if (i < 0) {
		msg(DBG_INFO) << "pass not allowed\n";
		return;
	}
	if (mode != MODE_PLAY) {
		Glib::RefPtr<Gtk::Action> a = m_uim->get_action(act_name[persist_mode]);
		assert(a);
		a->activate();
	}
	down(i);
}

void View::game_handicap(int num_stones)
{
	assert(game_can_init());
	game_set_handi(num_stones);
	do_new_node();
}

void View::game_mode(Mode new_mode)
{
	Mode old_mode = mode;
	if (new_mode == MODE_UNKNOWN) {
		new_mode = (Mode) o_mode.get_history();
		if (new_mode == mode) return;
		msg(DBG_DEBUG) << "mode change triggered by option menu new_mode = " << new_mode << '\n';
		Glib::RefPtr<Gtk::RadioAction> a = Glib::RefPtr<Gtk::RadioAction>::cast_dynamic(m_uim->get_action(act_name[new_mode]));
		assert(a);
		mode = new_mode;
		if (! a->get_active()) a->activate();
	} else { // from menu
		Glib::RefPtr<Gtk::RadioAction> a = Glib::RefPtr<Gtk::RadioAction>::cast_dynamic(m_uim->get_action(act_name[new_mode]));
		assert(a);
		if (new_mode == mode) return;
		if (! a->get_active()) return;
		msg(DBG_DEBUG) << "mode change triggered by menu new_mode = " << new_mode << '\n';
		mode = new_mode;
		if (o_mode.get_history() != new_mode) o_mode.set_history(new_mode);
	}

	msg(DBG_INFO) << "mode change to " << act_name[mode] << '\n';

	if (old_mode == MODE_SCORE) { // save scoring when mode changes
		int bt = 0;
		int wt = 0;
		int bc = 0;
		int wc = 0;
		ccgo::Marking * mk = new ccgo::Marking;
		for (int i = loc_range() - 1; i >= 0; i --) switch (terri[i]) {
		case TERRI_BLACK:
			mk->add_mark(i, ccgo::MARK_B_TERRITORY);
			bt ++;
			if (Board::get_state(i) == STATE_WHITE) bc ++;
			break;
		case TERRI_WHITE:
			mk->add_mark(i, ccgo::MARK_W_TERRITORY);
			wt ++;
			if (Board::get_state(i) == STATE_BLACK) wc ++;
			break;
		case TERRI_NEUTRAL:
			b->add_mark(i, ccgo::MARK_NONE);
			break;
		}
		where()->set_prop(ccgo::Marking::the_id(), mk);

		double komi = where()->get_game()->get_komi();
		double ts = bt + bc + get_b_cap() - wt - wc - get_w_cap() - komi;
		std::ostringstream s;
		s << _("Score:") << ' ';
		char buf[100];
		if (ts > 0) {
			snprintf(buf, 100, _("B+%g").c_str(), ts);
			s << buf;
		} else if (ts < 0) {
			snprintf(buf, 100, _("W+%g").c_str(), - ts);
			s << buf;
		} else s << _("tie!");
		s << '\n';
		// s << " (B:" << bt << '+' << bc + get_b_cap() << " W:" << wt << '+' << wc + get_w_cap();
		// if (komi >= 0) s << '+';
		// s << komi << ")\n";
		tv.get_buffer()->insert(tv.get_buffer()->begin(), s.str());
	}

	assert(new_mode != MODE_INIT || game_can_init());
	assert(where());
	o_turn.set_sensitive(mode == MODE_SETUP);
	switch (mode) {
	case MODE_VIEW:
		b->set_track(gtk::Board::TRACK_NONE);
		persist_mode = MODE_VIEW;
		break;
	case MODE_INIT:
		b->set_track(gtk::Board::TRACK_BLACK);
		game_set_handi(0);
		do_new_node();
		draw_map_handle();
		break;
	case MODE_PLAY:
		b->set_track(gtk::Board::TrackMode(get_turn()));
		persist_mode = MODE_PLAY;
		break;
	case MODE_MARK:
		b->set_track(gtk::Board::TRACK_ALL);
		persist_mode = MODE_MARK;
		break;
	case MODE_SETUP:
		if (! (where()->get_level() && node_can_setup())) {
			// create a setup node
			create_node();
		}
		b->set_track(gtk::Board::TRACK_ALL);
		break;
	case MODE_SCORE:
		{
			// create terri from Marking
			if (terri) delete [] terri;
			terri = new Terri[loc_range()];
			for (int i = loc_range() - 1; i >= 0; i --) terri[i] = ccgo::Board::TERRI_NEUTRAL;
			const ccgo::Marking * mk = dynamic_cast<const ccgo::Marking *>(where()->get_prop(ccgo::Marking::the_id()));
			if (mk) {
				const std::map<int, ccgo::Mark> m = mk->get_marks();
				for (std::map<int, ccgo::Mark>::const_iterator i = m.begin(); i != m.end(); i ++) {
					if (i->second == ccgo::MARK_W_TERRITORY) {
						terri[i->first] = TERRI_WHITE;
					} else if (i->second == ccgo::MARK_B_TERRITORY) {
						terri[i->first] = TERRI_BLACK;
					}
				}
			}
			do_score();
		}
		b->set_track(gtk::Board::TRACK_STONE);
		break;
	default:
		break;
	}
}

void View::game_turn_change()
{
	if (ccgo::Position::Turn(o_turn.get_history() + 1) == get_turn()) return;
	assert(mode == MODE_SETUP);
	ccgo::Position::Turn t = ccgo::Position::Turn(o_turn.get_history() + 1);

	node_set_turn(t);
	draw_map_handle();
}

void View::mark_set()
{
	if (mk_loc < 0) return;
	assert(where());

	const ccgo::Marking * mk = dynamic_cast<const ccgo::Marking *>(where()->get_prop(ccgo::Marking::the_id()));
	ccgo::Marking * nm = mk ? dynamic_cast<ccgo::Marking *>(mk->dup()) : new ccgo::Marking; // new marking
	assert(nm);

	nm->add_mark(mk_loc, mkw.get_mark());
	b->add_mark(mk_loc, mkw.get_mark());

	if (! nm->get_marks().size()) {
		delete nm;
		nm = 0;
	}
	where()->set_prop(ccgo::Marking::the_id(), nm);

	const ccgo::Labels * lb = dynamic_cast<const ccgo::Labels *>(where()->get_prop(ccgo::Labels::the_id()));
	ccgo::Labels * nl = lb ? dynamic_cast<ccgo::Labels *>(lb->dup()) : new ccgo::Labels; // new Labels
	assert(nl);

	nl->add_label(mk_loc, mkw.get_label());
	b->add_label(mk_loc, mkw.get_label());

	if (! nl->get_labels().size()) {
		delete nl;
		nl = 0;
	}
	where()->set_prop(ccgo::Labels::the_id(), nl);
}

void View::clear_marks()
{
	if (mode == MODE_SCORE) {
		Glib::RefPtr<Gtk::Action> a = m_uim->get_action(act_name[persist_mode]);
		assert(a);
		a->activate();
	}
	where()->set_prop(ccgo::Marking::the_id(), 0);
	for (int i = loc_range() - 1; i >= 0; i --) b->add_mark(i, ccgo::MARK_NONE);
}

void View::do_score()
{
	assert(terri);
	score_terri(terri);
	int bt = 0;
	int wt = 0;
	int bc = 0;
	int wc = 0;
	for (int i = loc_range() - 1; i >= 0; i --) switch (terri[i]) {
	case TERRI_BLACK:
		b->add_mark(i, ccgo::MARK_B_TERRITORY);
		bt ++;
		if (Board::get_state(i) == STATE_WHITE) bc ++;
		break;
	case TERRI_WHITE:
		b->add_mark(i, ccgo::MARK_W_TERRITORY);
		wt ++;
		if (Board::get_state(i) == STATE_BLACK) wc ++;
		break;
	case TERRI_NEUTRAL:
		b->add_mark(i, ccgo::MARK_NONE);
		break;
	}
	msg(DBG_INFO) << "B: territory " << bt << ", capture " << bc << "; W: territory " << wt << ", capture " << wc << '\n';

	// update score
	std::ostringstream s;
	double bs = bt + bc + get_b_cap();
	double ws = wt + wc + get_w_cap();
	double km = where()->get_game()->get_komi();
	if (km > 0) ws += km;
	else bs -= km;
	s << bs;
	b_cap.set_text(s.str());
	s.str("");
	s << ws;
	w_cap.set_text(s.str());
}

class MItem :
	public Gtk::MenuItem
{
public:
	std::string str;
	unsigned index;
};

void View::check_list() // update record list
{
	assert(list);

	std::vector<Act *>::iterator j = rec_act.begin();
	for (unsigned n = 0; n < list->size(); n ++) {
		std::ostringstream s;
		s << "Rec" << n;
		Glib::ustring nm = s.str();
		Glib::ustring lb = list->get_name(n);
		if (lb == "") lb = "no name";
		if (j == rec_act.end()) {
			msg(DBG_DEBUG) << "adding " << nm << ": " << lb << '\n';
			Act * ac = new Act;
			ac->act = Gtk::Action::create(nm, lb);
			ac->act->signal_activate().connect(bind(mem_fun(* this, & View::list_select), n));
			m_act->add(ac->act);
			ac->id = m_uim->new_merge_id();
			m_uim->add_ui(ac->id, "/MenuBar/RecordMenu", nm, nm, Gtk::UI_MANAGER_AUTO, false);
			rec_act.push_back(ac);
			j = rec_act.end();
			j --;
		} else {
			(* j)->act->property_label().set_value(lb);
		}
		(* j)->act->set_sensitive(list->get_game(n) != where()->get_game());
		j ++;
	}
	if (j != rec_act.end()) {
		msg(DBG_DEBUG) << "erasing extra\n";
		for (std::vector<Act *>::iterator k = j; k != rec_act.end(); k ++) {
			Act * ac = * k;
			m_uim->remove_ui(ac->id);
			m_act->remove(ac->act);
			delete ac;
		}
		rec_act.erase(j, rec_act.end());
	}
}

void View::list_select(unsigned index)
{
	// msg(DBG_ERROR) << "list: " << index << " selected\n";
	set_node(list->get_node(index));
}
