/* ccgo: go/gtk/gmap.cc
 * 
 * Copyright (C) 2004 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include <gmap.hh>
#include <debug.hh>

#include <gtkmm/scrolledwindow.h>
#include <sstream>

using namespace gtk;

// members of class GMap
GMap::GMap() :
	tl(- 1)
{
	add_events(Gdk::POINTER_MOTION_MASK |
		   Gdk::ENTER_NOTIFY_MASK |
		   Gdk::LEAVE_NOTIFY_MASK |
		   Gdk::BUTTON_PRESS_MASK |
		   Gdk::BUTTON_RELEASE_MASK);
}

GMap::~GMap()
{
}

void GMap::draw_map(ccgo::Node * node)
{
	cn = node;
	h.clear();
	mp.clear();
	mh = 0;
	dp = 0;
	assert(cn->get_game());
	walk.set_node(cn->get_game());
	lay_nodes(0);
	set_size_request((mh + 1) * cell_w, (dp + 1) * cell_h);
	msg(DBG_DEBUG) << "draw map (mh, dp) = (" << mh << ", " << dp << ")\n";
	tl = - 1;
	cl = - 1;
	// walk.set_node(0);
}

void GMap::find_node(ccgo::Node * node, unsigned level, int & x, int & y)
{
	assert(level < mp.size());
	Node * n = 0;
	for (unsigned i = 0; i < mp[level].size(); i ++) {
		n = & mp[level][i];
		if (n->n == node) break;
	}
	if (n->n != node) return;
	y = level * cell_h + cell_h / 2;
	x = n->h * cell_w + cell_w / 2;
}

void GMap::retrack()
{
	int x;
	int y;
	get_pointer(x, y);
	cl = int(y) / cell_h;
	cs = int(x) / cell_w;
	do_track();
}

void GMap::on_realize()
{
	Gtk::DrawingArea::on_realize();
	win = get_window();
	style = get_style();
}

bool GMap::on_configure_event(GdkEventConfigure * event)
{
	int w;
	int h;
	create_pango_layout("999")->get_pixel_size(w, h);
	cell_w = w + 10;
	cell_h = h + 10;
	return true;
}

bool GMap::on_expose_event(GdkEventExpose * event)
{
	if (event->count > 0) return true;
	int x1 = event->area.x / cell_w;
	int x2 = (event->area.x + event->area.width + cell_w - 1) / cell_w;
	int y1 = event->area.y / cell_h;
	int y2 = (event->area.y + event->area.height + cell_h - 1) / cell_h;
	for (int y = y1; y < y2; y ++) for (int x = x1; x < x2; x ++) draw_cell(y, x);
	return true;
}

bool GMap::on_enter_notify_event(GdkEventCrossing * event)
{
	cl = int(event->y) / cell_h;
	cs = int(event->x) / cell_w;
	do_track();
	return true;
}

bool GMap::on_leave_notify_event(GdkEventCrossing * event)
{
	cl = - 1;
	cs = - 1;
	do_track();
	return true;
}

bool GMap::on_motion_notify_event(GdkEventMotion * event)
{
	int l = int(event->y) / cell_h;
	int s = int(event->x) / cell_w;
	if (l == cl && s == cs) return true;
	cl = l;
	cs = s;
	do_track();
	return true;
}

bool GMap::on_button_press_event(GdkEventButton * event)
{
	if (event->button == 1 && tl >= 0) mark_track();
	else off_select();
	return true;
}

bool GMap::on_button_release_event(GdkEventButton * event)
{
	if (md) {
		draw_cell(tl, ts); // erase track

		// find corresponding node
		for (unsigned i = 0; i < mp[tl].size(); i ++) if (mp[tl][i].h == ts) {
			node_select(mp[tl][i].n);
			break;
		}
	}
	do_track();
	return true;
}

int GMap::lay_nodes(int buoyant)
{
	unsigned l = walk.where()->get_level();
	if (h.size() <= l) h.resize(l + 1);
	if (mp.size() <= l) mp.resize(l + 1);
	if (buoyant < h[l]) buoyant = h[l];
	Node n;
	n.n = walk.where();
	if (l > 0) n.pa = mp[l - 1].size();
	else n.pa = 0;
	if (n.n->down_num()) {
		walk.down(0u);
		int b = lay_nodes(buoyant);
		walk.up();
		n.kd.push_back(mp[l + 1].size() - 1);
		if (b > buoyant) buoyant = b - 1;
		for (unsigned i = 1; i < n.n->down_num(); i ++) {
			b ++;
			walk.down(i);
			b = lay_nodes(b);
			walk.up();
			n.kd.push_back(mp[l + 1].size() - 1);
		}
	}
	n.h = buoyant;
	if (walk.where()->get_move() >= 0 || walk.where()->get_move() == ccgo::MOVE_PASS) { // played node
		n.t = Node::Type(ccgo::Position::flip_turn(walk.get_turn()));
	} else if (walk.where()->get_delta()) {
		n.t = Node::TYPE_SETUP;
	} else if (walk.where()->up() == 0) {
		ccgo::Game * g = dynamic_cast<ccgo::Game *>(walk.where());
		assert(g);
		if (g->get_init() && g->get_init()->get_turn() == ccgo::Position::TURN_WHITE) n.t = Node::TYPE_HANDI;
		else n.t = Node::TYPE_START;
	} else {
		n.t = Node::TYPE_NONE;
	}
	h[l] = buoyant + 1;
	mp[l].push_back(n);
	if (buoyant > mh) mh = buoyant;
	if ((int) l > dp) dp = l;
	return buoyant;
}

void GMap::draw_cell(int level, int span)
{
	if (level >= (int) mp.size()) return;
	int j = - 1;
	do {
		j ++;
		if (j >= (int) mp[level].size()) return;
	} while (mp[level][j].h < span);
	// center
	int x = cell_w * span + cell_w / 2;
	int y = cell_h * level + cell_h / 2;
	int dh;
	Node & n = mp[level][j];
	Glib::RefPtr<const Gdk::GC> gc = style->get_fg_gc(get_state());
	win->clear_area(x - cell_w / 2, y - cell_h / 2, cell_w, cell_h);
	if (mp[level][j].h > span) { // bridging
		if (mp[level - 1][n.pa].h < span) {
			win->draw_line(gc, x - cell_w / 2, y - cell_h / 2, x + cell_w - cell_w / 2 - 1, y - cell_h / 2);
		}
		return;
	}
	// lines to parent
	if (level > 0) {
		dh = n.h - mp[level - 1][n.pa].h;
		int k = mp[level - 1][n.pa].kd.back(); // last sibling
		if (dh) {
			win->draw_line(gc, x, y, x - cell_w / 2, y - cell_h / 2);
			if (j != k) win->draw_line(gc, x - cell_w / 2, y - cell_h / 2, x + cell_w - cell_w / 2 - 1, y - cell_h / 2);
		} else {
			win->draw_line(gc, x, y, x, y - cell_h / 2);
			if (j != k && mp[level][k].h - n.h > 1) win->draw_line(gc, x, y - cell_h / 2, x + cell_w - cell_w / 2 - 1, y - cell_h / 2);
		}
	}
	// lines to kids
	if (n.kd.size()) {
		dh = mp[level + 1][n.kd[0]].h - n.h;
		if (dh == 0) win->draw_line(gc, x, y, x, y + cell_h / 2);
		dh = mp[level + 1][n.kd.back()].h - n.h;
		if (dh == 1) win->draw_line(gc, x, y, x + cell_w - cell_w / 2 - 1, y + cell_h / 2);
	}

	if (n.n == cn) { // current node
		win->draw_rectangle(gc, false, x - cell_w / 2 + 2, y - cell_h / 2 + 2, cell_w - 5, cell_h - 5);
	}

	std::ostringstream s;
	Glib::RefPtr<Pango::Layout> mn;
	int w;
	int h;

	int m = n.n->get_move();
	switch (n.t) {
	case Node::TYPE_NONE:
		win->draw_rectangle(gc, false, x - 2, y - 2, 4, 4);
		break;
	case Node::TYPE_BLACK:
		if (m >= 0) s << this->walk.loc_to_str(m);
		else s << "pa.";
		mn = create_pango_layout(s.str());
		mn->get_pixel_size(w, h);
		win->draw_rectangle(style->get_black_gc(), true, x - w / 2 - 1, y - h / 2 - 1, w + 2, h + 2);
		win->draw_layout(style->get_white_gc(), x - w / 2, y - h / 2, mn);
		break;
	case Node::TYPE_WHITE:
		// s << n.n->get_num();
		if (m >= 0) s << this->walk.loc_to_str(m);
		else s << "pa.";
		mn = create_pango_layout(s.str());
		mn->get_pixel_size(w, h);
		win->draw_rectangle(style->get_white_gc(), true, x - w / 2 - 1, y - h / 2 - 1, w + 2, h + 2);
		win->draw_layout(style->get_black_gc(), x - w / 2, y - h / 2, mn);
		break;
	case Node::TYPE_SETUP:
		s << "-/+";
		mn = create_pango_layout(s.str());
		mn->get_pixel_size(w, h);
		win->clear_area(x - w / 2 - 1, y - h / 2 - 1, w + 2, h + 2);
		win->draw_layout(gc, x - w / 2, y - h / 2, mn);
		break;
	case Node::TYPE_START:
		s << "[S]";
		mn = create_pango_layout(s.str());
		mn->get_pixel_size(w, h);
		win->clear_area(x - w / 2 - 1, y - h / 2 - 1, w + 2, h + 2);
		win->draw_layout(gc, x - w / 2, y - h / 2, mn);
		break;
	case Node::TYPE_HANDI:
		s << "[H]";
		mn = create_pango_layout(s.str());
		mn->get_pixel_size(w, h);
		win->clear_area(x - w / 2 - 1, y - h / 2 - 1, w + 2, h + 2);
		win->draw_layout(gc, x - w / 2, y - h / 2, mn);
		break;		
	}
}

void GMap::do_track()
{
	md = false;
	if (tl >= 0) {
		draw_cell(tl, ts);
		tl = - 1;
	}
	if (cl < 0 || cl >= (int) mp.size()) return;
	int i = 0;
	while (mp[cl][i].h < cs) {
		i ++;
		if (i >= (int) mp[cl].size()) return;
	}
	if (mp[cl][i].h > cs) return;

	tl = cl;
	ts = cs;
	Glib::RefPtr<const Gdk::GC> gc = style->get_fg_gc(get_state());
	int y = tl * cell_h + cell_h / 2;
	int x = ts * cell_w + cell_w / 2;
	win->draw_rectangle(gc, false, x - cell_w / 2, y - cell_h / 2, cell_w - 1, cell_h - 1);
}

void GMap::mark_track()
{
	if (tl < 0) return;
	Glib::RefPtr<const Gdk::GC> gc = style->get_fg_gc(get_state());
	int y = tl * cell_h + cell_h / 2;
	int x = ts * cell_w + cell_w / 2;
	win->draw_rectangle(gc, false, x - cell_w / 2 + 1, y - cell_h / 2 + 1, cell_w - 3, cell_h - 3);
	md = true;
}

// members of class MapWin
MapWin::MapWin() :
	Gtk::Window(Gtk::WINDOW_POPUP)
	// Gtk::Window(Gtk::WINDOW_TOPLEVEL)
{
	Gtk::ScrolledWindow * sw = Gtk::manage(new Gtk::ScrolledWindow);
	sw->set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	sw->add(gm);
	add(* sw);
	gm.off_select.connect(mem_fun(* this, & Gtk::Window::hide));
	vadj = sw->get_vadjustment();
	hadj = sw->get_hadjustment();
	set_default_size(200, 300);
}

void MapWin::show_map(ccgo::Node * node, unsigned level, int * x, int * y)
{
	show_all();
	gm.draw_map(node);
	resize_children();
	int xx;
	int yy;
	gm.find_node(node, level, xx, yy);
	double v = yy - vadj->get_page_size() / 2;
	if (v >  vadj->get_upper() - vadj->get_page_size()) v = vadj->get_upper() - vadj->get_page_size();
	if (v < vadj->get_lower()) v = vadj->get_lower();
	vadj->set_value(v);
	yy -= int(v);
	v = xx - hadj->get_page_size() / 2;
	if (v > hadj->get_upper() - hadj->get_page_size()) v = hadj->get_upper() - hadj->get_page_size();
	if (v < hadj->get_lower()) v = hadj->get_lower();
	hadj->set_value(v);
	xx -= int(v);
	if (x) * x = xx;
	if (y) * y = yy;
}

void MapWin::move_map(int x, int y)
{
	move(x, y);
	gm.hide();
	gm.show();
	gm.retrack();
}
