/* ccgo: go/gtk/board.cc
 * 
 * Copyright (C) 2004 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include <config.hh>
extern "C" {
#include <gettext.h>
}
#define _(String) Glib::locale_to_utf8(gettext(String))

#include <board.hh>
#include <debug.hh>
#include <gtkmm/frame.h>
#include <gtkmm/toggleaction.h>
#include <gconfmm/client.h>
#include <sstream>

using namespace gtk;

Board::Board() :
	b(0),
	lc(ccgo::MOVE_NONE),
	tm(TRACK_NONE),
	tl(ccgo::MOVE_PASS),
	mk(0),
	lb(0),
	tp(Gtk::WINDOW_POPUP),
	block(0)
{
	// configuration defaults
	Glib::RefPtr<Gnome::Conf::Client> cf = Gnome::Conf::Client::get_default_client();
	show_coordinates = cf->get_bool("/apps/ccgo/show-coordinate");
	pic_board = cf->get_bool("/apps/ccgo/picture-board");
	pic_stone = cf->get_bool("/apps/ccgo/picture-stone");
	square_board = true;
	std::string ip = cf->get_string("/apps/ccgo/image-path");

	if (pic_board && Glib::file_test(ip + "/board.png", Glib::FILE_TEST_EXISTS)) {
		Glib::RefPtr<Gdk::Bitmap> bm;
		Gdk::Pixbuf::create_from_file(ip + "/board.png")->render_pixmap_and_mask(board_pixmap, bm, 0);
		bg_pixmap_ok = true;
		msg(DBG_DEBUG) << "using bg pixmap\n";
	} else bg_pixmap_ok = false;

	if (pic_stone && Glib::file_test(ip + "/white_stone.png", Glib::FILE_TEST_EXISTS) && Glib::file_test(ip + "/black_stone.png", Glib::FILE_TEST_EXISTS)) {
		w_stone_image = Gdk::Pixbuf::create_from_file(ip + "/white_stone.png");
		b_stone_image = Gdk::Pixbuf::create_from_file(ip + "/black_stone.png");
		stone_image_ok = true;
	} else stone_image_ok = false;
	scaled_ok = false;

	add_events(Gdk::POINTER_MOTION_MASK |
		   Gdk::ENTER_NOTIFY_MASK |
		   Gdk::LEAVE_NOTIFY_MASK |
		   Gdk::BUTTON_PRESS_MASK |
		   Gdk::BUTTON_RELEASE_MASK);
	Gtk::Frame * f = Gtk::manage(new Gtk::Frame);
	tp.add(* f);
	f->add(tt);

	// setup configuration actions
	configs = Gtk::ActionGroup::create();
	configs->add(Gtk::ToggleAction::create("BoardCoord", _("Show _Coordinates")), Gtk::AccelKey("<control>c"), mem_fun(* this, & Board::toggle_coordinate));
	configs->add(Gtk::ToggleAction::create("BoardPicBoard", _("Use Picture _Board")), mem_fun(* this, & Board::toggle_pic_board));
	configs->add(Gtk::ToggleAction::create("BoardPicStone", _("Use Picture _Stones")), mem_fun(* this, & Board::toggle_pic_stone));
	configs->add(Gtk::ToggleAction::create("BoardSquare", _("Keep Board S_quare")), mem_fun(* this, & Board::toggle_square_board));

	msg(DBG_DEBUG) << "finishing init board\n";
}

Board::~Board()
{
	if (mk) delete[] mk;
	if (lb) {
		int r = b->loc_range();
		for (int i = 0; i < r; i ++) if (lb[i]) delete lb[i];
		delete[] lb;
	}
	if (block) delete[] block;
}

void Board::set_board(const ccgo::Board * board)
{
	b = board;
	if (mk) delete[] mk;
	if (lb) delete[] lb;
	if (block) delete[] block;
	if (b->get_size()) {
		int r = b->loc_range();
		mk = new ccgo::Mark [r];
		lb = new std::string * [r];
		block = new bool[r];
		for (int i = 0; i < r; i ++) {
			mk[i] = ccgo::MARK_NONE;
			lb[i] = 0;
			block[i] = false;
		}
	} else {
		mk = 0;
		lb = 0;
		block = 0;
	}
}

void Board::update()
{
	if (! is_realized()) return;
	do_resize();
	if (pic_board && bg_pixmap_ok) win->set_back_pixmap(board_pixmap, false);
	else style->set_background(win, get_state());
	win->clear_area(0, 0, win_w, win_h);
	if (! b || ! b->get_size()) return;
	if (show_coordinates) draw_coordinates();
	int r = b->loc_range();
	for (int i = 0; i < r; i ++) update(i);
}

void Board::update(int loc)
{
	if (! is_realized()) return;
	int z = b->get_size();
	int x = loc % z;
	int y = loc / z;

	int cx = pad_w + cell_w * x;
	int cy = pad_h + cell_h * (z - 1 - y);

	win->clear_area(cx - cell_w / 2, cy - cell_h / 2, cell_w, cell_h);
	// grids on the board
	if (cell_w > 0)	win->draw_line(style->get_fg_gc(get_state()),
				       (x == 0 ? cx : cx - cell_w / 2), cy,
				       (x == z - 1 ? cx : cx + cell_w - cell_w / 2 - 1), cy);
	if (cell_h > 0) win->draw_line(style->get_fg_gc(get_state()),
				       cx, (y == z - 1 ? cy : cy - cell_h / 2),
				       cx, (y == 0 ? cy : cy + cell_h - cell_h / 2 - 1));
	if (b->loc_is_star(loc)) win->draw_arc(style->get_fg_gc(get_state()), true,
					       cx - cell_w / 5, cy - cell_h / 5,
					       2 * (cell_w / 5), 2 * (cell_h / 5),
					       0, 64 * 360); // star positions
	if (cell_w > 0 && cell_h > 0) switch (b->get_state(loc)) {
	case ccgo::Board::STATE_WHITE:
		if (pic_stone && scaled_ok) {
			scaled_w_stone->render_to_drawable_alpha(win, 0, 0, cx - cell_w / 2, cy - cell_h / 2, cell_w, cell_h, Gdk::PIXBUF_ALPHA_FULL, 0, Gdk::RGB_DITHER_NORMAL, 0, 0);
		} else {
			win->draw_arc(style->get_white_gc(), true,
				      cx - cell_w / 2, cy - cell_h / 2,
				      cell_w - 1, cell_h - 1, 0, 64 * 360);
			win->draw_arc(style->get_black_gc(), false,
				      cx - cell_w / 2, cy - cell_h / 2,
				      cell_w - 1, cell_h - 1, 0, 64 * 360);
		}
		break;
	case ccgo::Board::STATE_BLACK:
		if (pic_stone && scaled_ok) {
			scaled_b_stone->render_to_drawable_alpha(win, 0, 0, cx - cell_w / 2, cy - cell_h / 2, cell_w, cell_h, Gdk::PIXBUF_ALPHA_FULL, 0, Gdk::RGB_DITHER_NORMAL, 0, 0);
		} else {
			win->draw_arc(style->get_black_gc(), true,
				      cx - cell_w / 2, cy - cell_h / 2,
				      cell_w - 1, cell_h - 1, 0, 64 * 360);
			win->draw_arc(style->get_white_gc(), false,
				      cx - cell_w / 2, cy - cell_h / 2,
				      cell_w - 1, cell_h - 1, 0, 64 * 360);
		}
		break;
	case ccgo::Board::STATE_EMPTY:
		break;
	}
	if (cell_w > 0 && cell_h > 0) if (mk && mk[loc]) {
		Glib::RefPtr<const Gdk::GC> gc;
		switch (b->get_state(loc)) {
		case ccgo::Board::STATE_WHITE:
			gc = style->get_black_gc();
			break;
		case ccgo::Board::STATE_BLACK:
			gc = style->get_white_gc();
			break;
		case ccgo::Board::STATE_EMPTY:
			gc = style->get_fg_gc(get_state());
			break;
		}
		switch (mk[loc]) {
		case ccgo::MARK_CIRCLE:
			if (! b->get_state(loc)) win->clear_area(cx - cell_w * 3 / 10, cy - cell_h * 3 / 10,
								 cell_w * 3 / 5 - 1, cell_h * 3 / 5 - 1);
			win->draw_arc(gc, false, cx - cell_w * 3 / 10, cy - cell_h * 3 / 10,
				      cell_w * 3 / 5 - 1, cell_h * 3 / 5 - 1, 0, 64 * 360);
			break;
		case ccgo::MARK_TRIANGLE:
			if (! b->get_state(loc)) win->clear_area(cx - cell_w * 29 / 100, cy - cell_h * 49 / 100,
							      cell_w * 58 / 100, cell_h * 73 / 100 - 1);
			win->draw_line(gc, cx - cell_w * 43 / 100, cy + cell_h * 6 / 25 - 1,
				       cx, cy - cell_h * 49 / 100);
			win->draw_line(gc, cx + cell_w * 43 / 100, cy + cell_h * 6 / 25 - 1,
				       cx, cy - cell_h * 49 / 100);
			win->draw_line(gc, cx - cell_w * 43 / 100, cy + cell_h * 6 / 25 - 1,
				       cx + cell_w * 43 / 100 - 1, cy + cell_h * 6 / 25 - 1);
			break;
		case ccgo::MARK_SQUARE:
			if (! b->get_state(loc)) win->clear_area(cx - cell_w * 34 / 100, cy - cell_h * 34 / 100,
								 cell_w * 68 / 100 - 1, cell_h * 68 / 100 - 1);
			win->draw_rectangle(gc, false, cx - cell_w * 34 / 100, cy - cell_h * 34 / 100,
					    cell_w * 68 / 100 - 1, cell_h * 68 / 100 - 1);
			break;
		case ccgo::MARK_CROSS:
			win->draw_line(gc, cx - cell_w * 34 / 100, cy - cell_h * 34 / 100,
				       cx + cell_w * 34 / 100 - 1, cy + cell_h * 34 / 100 - 1);
			win->draw_line(gc, cx + cell_w * 34 / 100 - 1, cy - cell_h * 34 / 100,
				       cx - cell_w * 34 / 100, cy + cell_h * 34 / 100 - 1);
			break;
		case ccgo::MARK_SELECT:
			win->draw_rectangle(gc, true, cx - cell_w * 2 / 10, cy - cell_h * 2 / 10,
					    cell_w * 4 / 10 - 1, cell_h * 4 / 10 - 1);
			break;
		case ccgo::MARK_W_TERRITORY:
			win->draw_arc(style->get_white_gc(), true, cx - cell_w / 4, cy - cell_h / 4,
				      cell_w / 2 - 1, cell_h / 2 - 1, 0, 64 * 360);
			win->draw_arc(style->get_black_gc(), false, cx - cell_w / 4, cy - cell_h / 4,
				      cell_w / 2 - 1, cell_h / 2 - 1, 0, 64 * 360);
			break;
		case ccgo::MARK_B_TERRITORY:
			win->draw_arc(style->get_black_gc(), true, cx - cell_w / 4, cy - cell_h / 4,
				      cell_w / 2 - 1, cell_h / 2 - 1, 0, 64 * 360);
			win->draw_arc(style->get_white_gc(), false, cx - cell_w / 4, cy - cell_h / 4,
				      cell_w / 2 - 1, cell_h / 2 - 1, 0, 64 * 360);
			break;
		case ccgo::MARK_NONE:
			break;
		}
	}
	if (cell_w > 0 && cell_h > 0 && lb && lb[loc]) {
		Glib::RefPtr<Pango::Layout> tx = create_pango_layout(* lb[loc]);
		int w;
		int h;
		tx->get_pixel_size(w, h);
		Glib::RefPtr<const Gdk::GC> gc;
		switch (b->get_state(loc)) {
		case ccgo::Board::STATE_WHITE:
			gc = style->get_black_gc();
			break;
		case ccgo::Board::STATE_BLACK:
			gc = style->get_white_gc();
			break;
		case ccgo::Board::STATE_EMPTY:
			gc = style->get_fg_gc(get_state());
			win->clear_area(cx - cell_w / 2, cy - cell_h / 2, cell_w - 1, cell_h - 1);
			break;
		}
		if (w <= cell_w && h <= cell_h) win->draw_layout(gc, cx - w / 2, cy - h / 2, tx);
		else if (cell_w / 2 > 0 && cell_h / 2 > 0) {
			win->draw_rectangle(gc, false, cx - cell_w / 4, cy - cell_h / 4, cell_w / 2 - 1, cell_h / 2 - 1);
			win->draw_line(gc, cx, cy - cell_h / 4, cx - cell_w / 4, cy + cell_h / 4 - 1);
			win->draw_line(gc, cx + cell_w / 4 - 1, cy, cx - cell_w / 4, cy + cell_h / 4 - 1);
		}
	}
}

void Board::set_track(TrackMode trackmode)
{
	tm = trackmode;
	do_track();
}

Board::TrackMode Board::get_track()
{
	return tm;
}

void Board::add_mark(int loc, ccgo::Mark mark)
{
	assert(mk);
	msg(DBG_DEBUG) << "marking " << loc << " with " << mark << '\n';
	if (mk[loc] == mark) return;
	mk[loc] = mark;
	update(loc);
}

void Board::clear_marks()
{
	assert(mk);
	int r = b->loc_range();
	msg(DBG_DEBUG) << "clearing all marks loc_range() = " << r << "\n";
	for (int i = 0; i < r; i ++) if (mk[i]) {
		msg(DBG_DEBUG) << "clearing mark at " << i << '\n';
		mk[i] = ccgo::MARK_NONE;
		update(i);
	}
}

void Board::add_label(int loc, const std::string & label)
{
	assert(lb);
	if (lb[loc]) delete lb[loc];
	if (label != "") lb[loc] = new std::string(label);
	else lb[loc] = 0;
	update(loc);
}

void Board::clear_labels()
{
	assert(lb);
	int r = b->loc_range();
	for (int i = 0; i < r; i ++) if (lb[i]) {
		delete lb[i];
		lb[i] = 0;
		update(i);
	}
}

void Board::set_block(int loc, bool is_blocked)
{
	assert(block);
	block[loc] = is_blocked;
}

void Board::clear_blocks()
{
	assert(block);
	int r = b->loc_range();
	for (int i = 0; i < r; i ++) block[i] = false;
}

const Glib::RefPtr<Gtk::ActionGroup> & Board::get_configs()
{
	return configs;
}

void Board::on_realize()
{
	Gtk::DrawingArea::on_realize();
	win = get_window();
	style = get_style();
	Glib::RefPtr<Gtk::ToggleAction> ta;
	ta = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(configs->get_action("BoardCoord"));
	ta->set_active(show_coordinates);
	ta = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(configs->get_action("BoardPicBoard"));
	ta->set_active(pic_board);
	ta = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(configs->get_action("BoardPicStone"));
	ta->set_active(pic_stone);
	ta = Glib::RefPtr<Gtk::ToggleAction>::cast_dynamic(configs->get_action("BoardSquare"));
	ta->set_active(square_board);
}

void Board::on_style_changed(const Glib::RefPtr<Gtk::Style> & prev_style)
{
	DrawingArea::on_style_changed(prev_style);
	style = get_style();
	msg(DBG_DEBUG) << "changing style\n";
}

bool Board::on_configure_event(GdkEventConfigure * event)
{
	win_w = event->width;
	win_h = event->height;
	do_resize();
	return true;
}

bool Board::on_expose_event(GdkEventExpose * event)
{
	if (event->count > 0) return true;
	// if (show_coordinates) draw_coordinates();
	// for (int l = b->loc_range() - 1; l >=0; l --) update(l);
	update();
	return true;
}

bool Board::on_enter_notify_event(GdkEventCrossing * event)
{
	msg(DBG_DEBUG) << "entered\n";
	if (! b || ! b->get_size()) {
		lc = ccgo::MOVE_PASS;
		return true;
	}
	lc = find_loc(int(event->x), int(event->y));
	do_track();
	return true;
}

bool Board::on_leave_notify_event(GdkEventCrossing * event)
{
	if (! b || ! b->get_size()) {
		lc = ccgo::MOVE_NONE;
		return true;
	}
	lc = ccgo::MOVE_NONE;
	do_track();
	return true;
}

bool Board::on_motion_notify_event(GdkEventMotion * event)
{
	if (! b || ! b->get_size()) return true;
	int l = find_loc(int(event->x), int(event->y));
	if (l == lc) return true;
	lc = l;
	do_track();
	return true;
}

bool Board::on_button_press_event(GdkEventButton * event)
{
	if (event->button == 1) mark_track();
	else {
		md = false;
		update(tl);
		tl = - 1;
		lc = - 1;
	}
	return true;
}

bool Board::on_button_release_event(GdkEventButton * event)
{
	if (md) {
		update(tl); // erase track
		clicked(tl);
		tl = ccgo::MOVE_PASS;
	}
	do_track();
	return true;
}

void Board::do_resize()
{
	if (! b || ! b->get_size()) return;

	int z = b->get_size();
	int w;
	int h;
	if (show_coordinates) {
		// space for coordinates
		create_pango_layout("A")->get_pixel_size(w, h);
		cell_h = (win_h - 2 * h) / z;
		create_pango_layout("29")->get_pixel_size(w, h);
		cell_w = (win_w - 2 * w) / z;
	} else {
		cell_h = win_h / z;
		cell_w = win_w / z;
	}

	if (square_board) {
		if (cell_h > cell_w) cell_h = cell_w;
		else cell_w = cell_h;
	}

	// padding space
	pad_w = (win_w - cell_w * (z - 1)) / 2;
	pad_h = (win_h - cell_h * (z - 1)) / 2;

	if (stone_image_ok && cell_w > 2 && cell_h > 2) { // INTERP_HYPER gets stuck when the size is too tiny...
		scaled_w_stone = w_stone_image->scale_simple(cell_w, cell_h, Gdk::INTERP_HYPER);
		scaled_b_stone = b_stone_image->scale_simple(cell_w, cell_h, Gdk::INTERP_HYPER);
		scaled_ok = true;
	} else scaled_ok = false;
}

void Board::draw_coordinates()
{
	// draw coordinates
	int z = b->get_size();
	int xm = 0;
	for (int x = 0; x < z; ++ x) {
		std::string s = "";
		s += (x < 8 ? char('A' + x) : char ('B' + x));
		Glib::RefPtr<Pango::Layout> tx = create_pango_layout(s);
		int w;
		int h;
		tx->get_pixel_size(w, h);
		int xx = pad_w + cell_w * x - w / 2;
		if (xx < xm) continue;
		if (pad_h - cell_h / 2 < h) continue;
		win->draw_layout(style->get_fg_gc(get_state()),
				 xx, pad_h - cell_h / 2 - h, tx);
		win->draw_layout(style->get_fg_gc(get_state()),
				 xx, win_h - (pad_h - cell_h / 2), tx);
		xm = xx + w - w / 4;
	}
	int ym = win_h - 1;
	for (int y = 0; y < z; ++ y) {
		std::ostringstream s;
		s << y + 1;
		Glib::RefPtr<Pango::Layout> tx = create_pango_layout(s.str());
		int w;
		int h;
		tx->get_pixel_size(w, h);
		int yy = pad_h + cell_h * (z - 1 - y) - h / 2;
		if (yy + h > ym) continue;
		if (pad_w - cell_w / 2 < w) continue;
		win->draw_layout(style->get_fg_gc(get_state()),
				 pad_w - cell_w / 2 - w, yy, tx);
		win->draw_layout(style->get_fg_gc(get_state()),
				 win_w - (pad_w - cell_w / 2), yy, tx);
		ym = yy + h / 2;
	}
}

int Board::find_loc(int x, int y)
{
	if (cell_w == 0 || cell_h == 0) return ccgo::MOVE_PASS;
	int z = b->get_size();
	if (! z) return ccgo::MOVE_PASS;
	int xx = (x - pad_w + cell_w / 2) / cell_w;
	int yy = z - 1 - (y - pad_h + cell_h / 2) / cell_h;
	if (xx < 0 || xx >= z || yy < 0 || yy >= z) return ccgo::MOVE_PASS;
	return xx + yy * z;
}

void Board::do_track()
{
	md = false;
	if (tl >= 0) {
		update(tl);
		tl = ccgo::MOVE_PASS;
	}
	tp.hide();
	if (lc < 0) return;
	if (cell_h == 0 || cell_w == 0) return;
	int z = b->get_size();
	int cx = pad_w + cell_w * (lc % z);
	int cy = pad_h + cell_h * (z - 1 - lc / z);

	// pop up label
	if (lb && lb[lc]) {
		tt.set_text(* lb[lc]);
		tp.show_all();
		tp.resize(1, 1);
		int x;
		int y;
		win->get_origin(x, y);
		int x1;
		int y1;
		tp.get_size(x1, y1);
		tp.move(x + cx, y + cy - y1 - cell_h - 2);
	}

	Glib::RefPtr<const Gdk::GC> gc;
	switch (tm) {
	case TRACK_NONE:
		return;
	case TRACK_WHITE:
		if (b->get_state(lc) != ccgo::Board::STATE_EMPTY) return;
		// if (p && p->loc_is_ko(lc)) return;
		if (block && block[lc]) return;
		gc = style->get_white_gc();
		win->draw_arc(gc, true,
			      cx - cell_w / 2, cy - cell_h / 2,
			      cell_w - 1, cell_h - 1, 0, 64 * 360);
		break;
	case TRACK_BLACK:
		if (b->get_state(lc) != ccgo::Board::STATE_EMPTY) return;
		// if (p && p->loc_is_ko(lc)) return;
		if (block && block[lc]) return;
		gc = style->get_black_gc();
		win->draw_arc(gc, true,
			      cx - cell_w / 2, cy - cell_h / 2,
			      cell_w - 1, cell_h - 1, 0, 64 * 360);
		break;
	case TRACK_STONE:
		if (b->get_state(lc) == ccgo::Board::STATE_EMPTY) return;
		gc = b->get_state(lc) == ccgo::Board::STATE_WHITE ?
			style->get_black_gc() :
			style->get_white_gc();
		win->draw_arc(gc, false,
			      cx - cell_w / 4, cy - cell_h / 4,
			      cell_w / 2, cell_h / 2, 0, 64 * 360);
		break;
	case TRACK_ALL:
		gc = style->get_fg_gc(get_state());
		win->draw_rectangle(gc, false, cx - cell_w / 2, cy - cell_h / 2, cell_w - 1, cell_h - 1);
		break;
	}
	tl = lc;
}

void Board::mark_track()
{
	if (tl < 0) return; // no track
	int z = b->get_size();
	int cx = pad_w + cell_w * (lc % z);
	int cy = pad_h + cell_h * (z - 1 - lc / z);
	Glib::RefPtr<const Gdk::GC> gc;
	switch (tm) {
	case TRACK_BLACK:
		gc = style->get_white_gc();
		win->draw_arc(gc, false, cx - cell_w / 2, cy - cell_h / 2, cell_w - 1, cell_h - 1, 0, 64 * 360);
		break;
	case TRACK_WHITE:
		gc = style->get_black_gc();
		win->draw_arc(gc, false, cx - cell_w / 2, cy - cell_h / 2, cell_w - 1, cell_h - 1, 0, 64 * 360);
		break;
	case TRACK_STONE:
		gc = (b->get_state(tl) == ccgo::Board::STATE_BLACK ?
			style->get_white_gc() :
		      style->get_black_gc());
		win->draw_arc(gc, true,
			      cx - cell_w / 4, cy - cell_h / 4,
			      cell_w / 2, cell_h / 2, 0, 64 * 360);
		break;
	case TRACK_ALL:
		if (cell_h < 3 || cell_w < 3) break;
		gc = style->get_fg_gc(get_state());
		win->draw_rectangle(gc, false, cx - cell_w / 2 + 1, cy - cell_h / 2 + 1, cell_w - 3, cell_h - 3);
		break;
	case TRACK_NONE:
		break;
	}
	md = true;
}

void Board::toggle_coordinate()
{
	Glib::RefPtr<const Gtk::ToggleAction> ta = Glib::RefPtr<const Gtk::ToggleAction>::cast_dynamic(configs->get_action("BoardCoord"));
	bool shown = ta->get_active();
	if (show_coordinates != shown) {
		show_coordinates = shown;
		update();
	}
}

void Board::toggle_pic_board()
{
	Glib::RefPtr<const Gtk::ToggleAction> ta = Glib::RefPtr<const Gtk::ToggleAction>::cast_dynamic(configs->get_action("BoardPicBoard"));
	bool r = ta->get_active();
	if (r != pic_board) {
		pic_board = r;
		if (pic_board) {
			std::string ip = Gnome::Conf::Client::get_default_client()->get_string("/apps/ccgo/image-path");
			if (Glib::file_test(ip + "/board.png", Glib::FILE_TEST_EXISTS)) {
				Glib::RefPtr<Gdk::Bitmap> bm;
				Gdk::Pixbuf::create_from_file(ip + "/board.png")->render_pixmap_and_mask(board_pixmap, bm, 0);
				bg_pixmap_ok = true;
			} else bg_pixmap_ok = false;
		}
		update();
	}
}

void Board::toggle_pic_stone()
{
	Glib::RefPtr<const Gtk::ToggleAction> ta = Glib::RefPtr<const Gtk::ToggleAction>::cast_dynamic(configs->get_action("BoardPicStone"));
	bool r = ta->get_active();
	if (r != pic_stone) {
		pic_stone = r;
		if (pic_stone) {
			std::string ip = Gnome::Conf::Client::get_default_client()->get_string("/apps/ccgo/image-path");
			if (Glib::file_test(ip + "/white_stone.png", Glib::FILE_TEST_EXISTS) && Glib::file_test(ip + "/black_stone.png", Glib::FILE_TEST_EXISTS)) {
				w_stone_image = Gdk::Pixbuf::create_from_file(ip + "/white_stone.png");
				b_stone_image = Gdk::Pixbuf::create_from_file(ip + "/black_stone.png");
				stone_image_ok = true;
				if (cell_w > 0 && cell_h > 0) {
					scaled_w_stone = w_stone_image->scale_simple(cell_w, cell_h, Gdk::INTERP_HYPER);
					scaled_b_stone = b_stone_image->scale_simple(cell_w, cell_h, Gdk::INTERP_HYPER);
					scaled_ok = true;
				} else scaled_ok = false;
			} else {
				stone_image_ok = false;
				scaled_ok = false;
			}
		}
		update();
	}
}

void Board::toggle_square_board()
{
	Glib::RefPtr<const Gtk::ToggleAction> ta = Glib::RefPtr<const Gtk::ToggleAction>::cast_dynamic(configs->get_action("BoardSquare"));
	bool r = ta->get_active();
	if (r != square_board) {
		square_board = r;
		do_resize();
		update();
	}
}
