/* ccgo: go/sgf.cc
 *
 * Copyright (C) 2004 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include <debug.hh>
#include <go/sgf0.hh>
#include <sstream>
using namespace std;

namespace {
	void skip_white(istream & f)
	{
		while (! f.eof() && isspace(f.peek())) f.get();
	}

	struct Label
	{
		int l;
		string s;
	};

	class Prop
	{
		string id;
		vector<string> raw_values;
	public:
		bool load(istream &);
		void save(ostream &) const;
		const string & get_id() const;
		int as_int() const;
		double as_real() const;
		int as_loc(const ccgo::Board & board) const;
		vector<int> ar_loc(const ccgo::Board & board) const;
		char as_char() const;
		const string & as_string() const;
		Label as_label(const ccgo::Board & board) const;
		vector<Label> ar_label(const ccgo::Board & board) const;
	};

	class Node
	{
		vector<Prop> prop_list;
	public:
		bool load(istream &);
		void save(ostream &) const;
		const Prop * find(const string &) const;
	};


	bool Prop::load(istream & f)
	{
		id = "";
		raw_values.clear();
		skip_white(f);
		if (f.eof() || ! isalpha(f.peek())) return false;
		// read id string
		do {
			id += f.get();
		} while ((! f.eof()) && isalpha(f.peek())); 
		do {
			skip_white(f);
			if (f.eof() || f.peek() != '[') break;
			f.get();
			string s = "";
			bool escape = false;
			int line_break = 0;
			while ((! f.eof()) && (f.peek() != ']' || escape)) {
				char c = f.get();
				if (c == '\r') {
					if (line_break == 1) {
						line_break = 0;
						continue;
					}
					c = '\n';
					line_break = 2;
				} else if (c == '\n') {
					if (line_break ==2) {
						line_break = 0;
						continue;
					}
					line_break = 1;
				} else line_break = 0;
				if (escape) {
					escape = false;
					if (c == '\n') continue;
				} else escape = c == '\\';
				if (! escape) s += c;
			}
			if (f.eof()) {
				msg(DBG_ERROR) << "incomplete value field!\n";
			} else f.get();
			raw_values.push_back(s);
		} while (true);
		return true;
	}

	void Prop::save(ostream & f) const
	{
		f << id;
		for (vector<string>::const_iterator i = raw_values.begin(); i != raw_values.end(); i ++) {
			f << '[';
			for (unsigned j = 0; j < i->length(); j ++) {
				if ((* i)[j] == ']' || (* i)[j] == '\\') f << '\\';
				f << (* i)[j];
			}
			f << ']';
		}
	}

	const string & Prop::get_id() const
	{
		return id;
	}

	int Prop::as_int() const
	{
		if (raw_values.size() != 1) return 0;
		istringstream s(raw_values[0]);
		int i;
		s >> i;
		return i;
	}

	double Prop::as_real() const
	{
		if (raw_values.size() != 1) return 0;
		istringstream s(raw_values[0]);
		double d;
		s >> d;
		return d;
	}

	int Prop::as_loc(const ccgo::Board & board) const
	{
		string s = raw_values[0];
		if (s.length() < 2) return ccgo::MOVE_PASS;
		int x = s[0] - 'a';
		int y = s[1] - 'a';
		int z = board.get_size();
		if (x < 0 || x >= z) return ccgo::MOVE_PASS;
		if (y < 0 || y >= z) return ccgo::MOVE_PASS;
		// compressed points can not be handle here!
		return board.xy_to_loc(x, board.flip_axis(y));
	}

	vector<int> Prop::ar_loc(const ccgo::Board & board) const
	{
		vector<int> r;

		for (vector<string>::const_iterator i = raw_values.begin();
		     i != raw_values.end(); i ++) {
			string s = * i;
			if (s.length() < 2) continue;
			int x = s[0] - 'a';
			int y = s[1] - 'a';
			if (s.length() == 2) {
				r.push_back(board.xy_to_loc(x, board.flip_axis(y)));
				continue;
			}
			if (s[2] == ':') { //compressed point
				msg(DBG_DEBUG) << "compressed points\n";
				if (s.length() < 5) continue;
				int x2 = s[3] - 'a';
				int y2 = s[4] - 'a';
				if (x > x2) {
					int t = x2;
					x2 = x;
					x = t;
				}
				if (y > y2) {
					int t = y2;
					y2 = y;
					y = t;
				}
				for (int xx = x; xx <= x2; xx++) for (int yy = y; yy <= y2; yy ++) r.push_back(board.xy_to_loc(xx, board.flip_axis(yy)));
				continue;
			}
			msg(DBG_ERROR) << "sgf: error in compressed format\n";
		}
		return r;
	}

	char Prop::as_char() const
	{
		return raw_values[0][0];
	}

	const string & Prop::as_string() const
	{
		return raw_values[0];
	}

	Label Prop::as_label(const ccgo::Board & board) const
	{
		Label l;
		string s = raw_values[0];
		l.l = board.xy_to_loc(s[0] - 'a', board.flip_axis(s[1] - 'a'));
		// assert(s[2] == ':');
		l.s = s.substr(3);
		return l;
	}

	vector<Label> Prop::ar_label(const ccgo::Board & board) const
	{
		vector<Label> la;
		for (unsigned i = 0; i < raw_values.size(); i ++) {
			string s = raw_values[i];
			Label l;
			l.l = board.xy_to_loc(s[0] - 'a', board.flip_axis(s[1] - 'a'));
			// assert(s[2] == ':');
			l.s = s.substr(3);
			la.push_back(l);
		}
		return la;
	}


	bool Node::load(istream & f)
	{
		prop_list.clear();
		skip_white(f);
		if (f.eof() || f.peek() != ';') return false;
		f.get();
		Prop p;
		while (p.load(f)) prop_list.push_back(p);
		return true;
	}

	void Node::save(ostream & f) const
	{
		f << ';';
		for (vector<Prop>::const_iterator i = prop_list.begin(); i != prop_list.end(); i ++) {
			i->save(f);
		}
		// f << endl;
	}

	const Prop * Node::find(const string & s) const
	{
		for (vector<Prop>::const_iterator i = prop_list.begin(); i != prop_list.end(); i ++) {
			if (s == i->get_id()) return & (* i);
		}
		return NULL;
	}

	// non-game-related node properties
	void set_marks(ccgo::Node * n, Node & m)
	{
		const Prop * p;
		int z = n->get_game()->get_init()->get_size();

		if ((p = m.find("N"))) { // node name
			// p->as_string();
		}

		if ((p = m.find("C"))) { // comment
			ccgo::Comment * c = new ccgo::Comment;
			c->set_comment(p->as_string());
			n->set_prop(ccgo::Comment::the_id(), c);
		}

		if ((p = m.find("LB"))) {	// labels
			vector<Label> l = p->ar_label(z);
			ccgo::Labels * lb = new ccgo::Labels;
			for (vector<Label>::iterator i = l.begin(); i != l.end(); i ++) {
				lb->add_label(i->l, i->s);
			}
			n->set_prop(ccgo::Labels::the_id(), lb);
		}

		ccgo::Marking * mk = new ccgo::Marking;
		if ((p = m.find("CR"))) { // circle
			vector<int> l = p->ar_loc(z);
			for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) {
				mk->add_mark(* i, ccgo::MARK_CIRCLE);
			}
		}

		if ((p = m.find("TR"))) { // triangle
			vector<int> l = p->ar_loc(z);
			for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) {
				mk->add_mark(* i, ccgo::MARK_TRIANGLE);
			}
		}

		if ((p = m.find("SQ"))) { // square
			vector<int> l = p->ar_loc(z);
			for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) {
				mk->add_mark(* i, ccgo::MARK_SQUARE);
			}
		}

		if ((p = m.find("MA"))) { // mark
			vector<int> l = p->ar_loc(z);
			for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) {
				mk->add_mark(* i, ccgo::MARK_CROSS);
			}
		}

		if ((p = m.find("SL"))) { // select
			vector<int> l = p->ar_loc(z);
			for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) {
				mk->add_mark(* i, ccgo::MARK_SELECT);
			}
		}

		if ((p = m.find("TW"))) { // territory of white
			vector<int> l = p->ar_loc(z);
			for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) {
				mk->add_mark(* i, ccgo::MARK_W_TERRITORY);
			}
		}

		if ((p = m.find("TB"))) { // territory of black
			vector<int> l = p->ar_loc(z);
			for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) {
				mk->add_mark(* i, ccgo::MARK_B_TERRITORY);
			}
		}
		if (mk->get_marks().size()) n->set_prop(ccgo::Marking::the_id(), mk);
		else delete mk;
	}

	bool load_node(istream & f, ccgo::Walk & w)
	{
		Node m;
		if (! m.load(f)) return false;

		const Prop * p;

		// check if there is any "setup" operations and put them in a Delta
		bool any_setup = false;
		ccgo::Delta d;
		if ((p = m.find("AB"))) { // add black stones...
			vector<int> l = p->ar_loc(w);
			for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) {
				if (d.is_changed(* i)) {
					msg(DBG_ERROR) << "duplicate AB at " << w.loc_to_str(* i) << '\n';
					continue;
				}
				d.add_change(* i, w.get_state(* i), ccgo::Board::STATE_BLACK);
			}
			any_setup = true;
		}
		if ((p = m.find("AW"))) { // add white stones
			vector<int> l = p->ar_loc(w);
			for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) {
				if (d.is_changed(* i)) {
					msg(DBG_ERROR) << "duplicate AW at " << w.loc_to_str(* i) << '\n';
					continue;
				}
				d.add_change(* i, w.get_state(* i), ccgo::Board::STATE_WHITE);
			}
			any_setup = true;
		}
		if ((p = m.find("AE"))) { // add empty sites
			vector<int> l = p->ar_loc(w);
			for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) {
				if (d.is_changed(* i)) {
					msg(DBG_ERROR) << "duplicate AE at " << w.loc_to_str(* i) << '\n';
					continue;
				}
				d.add_change(* i, w.get_state(* i), ccgo::Board::STATE_EMPTY);
			}
			any_setup = true;
		}
		if ((p = m.find("PL"))) { // set turn
			ccgo::Position::Turn tn;
			if (p->as_char() == 'B') tn = ccgo::Position::TURN_BLACK;
			else if (p->as_char() == 'W') tn = ccgo::Position::TURN_WHITE;
			else tn = w.get_turn();
			d.set_turn_flip(tn != w.get_turn());
			any_setup = true;
		}

		ccgo::Node * n = 0;
		if (any_setup) { // a setup node
			if ((p = m.find("B"))) { // illegal in FF[4]
				int i = p->as_loc(w);
				if (i != ccgo::MOVE_PASS) {
					d.del_change(i);
					d.add_change(i, w.get_state(i), ccgo::Board::STATE_BLACK);
				}
				d.set_turn_flip(w.get_turn() == ccgo::Position::TURN_BLACK);
			}
			if ((p = m.find("W"))) { // illegal in FF[4]
				int i = p->as_loc(w);
				if (i != ccgo::MOVE_PASS) {
					d.del_change(i);
					d.add_change(i, w.get_state(i), ccgo::Board::STATE_WHITE);
				}
				d.set_turn_flip(w.get_turn() == ccgo::Position::TURN_WHITE);
			}
			n = w.make_setup(d);
		} else {
			// a move node
			int move = ccgo::MOVE_NONE;
			bool foul_turn = false;
			if ((p = m.find("B"))) {  // only one of B or W should happen
				if (w.get_turn() != ccgo::Position::TURN_BLACK) {
					msg(DBG_ERROR) << "implicited turn change!\n";
					foul_turn = true;
				}
				move = p->as_loc(w);
				// if (move == ccgo::MOVE_NONE) move = ccgo::MOVE_PASS;
			}

			if ((p = m.find("W"))) {
				if (move != ccgo::MOVE_NONE) { // B existed!!
					msg(DBG_WARNING) << "Both B & W properties exist in the same node!\n";
				}
				if (w.get_turn() != ccgo::Position::TURN_WHITE) {
					msg(DBG_ERROR) << "implicited turn change!\n";
					foul_turn = true;
				}
				move = p->as_loc(w);
				// if (move == ccgo::MOVE_NONE) move = ccgo::MOVE_PASS;
			}

			if (foul_turn) msg(DBG_WARNING) << "foul turn!\n";
			if (move == ccgo::MOVE_NONE) { // no move, no setup ??
				msg(DBG_ERROR) << "No move nor setup in node.\n";
			} else {
				n = foul_turn ? w.make_extra_move(move) : w.make_move(move);
				if (! n) {
					msg(DBG_WARNING) << "Illegal move in SGF.";
				}
			}
		}
		if (! n) {
			assert(d.is_null());
			n = w.make_setup(d); // empty setup node
		}
		// substitutive properties
		if ((p = m.find("MN"))) n->set_num(p->as_int());
		set_marks(n, m);

		w.where()->add_child(n);
		w.down(n->get_index());

		return true;
	}

	bool load_tree(istream & f, ccgo::Walk & w)
	{
		ccgo::Node * sn = w.where(); // keep starting point
		skip_white(f);
		if (f.eof() || f.peek() != '(') return false;
		f.get();
		int n = 0;
		while (load_node(f, w)) n ++;
		if (! n) return false; // there should be one node at least
		msg(DBG_DEBUG) << "loaded tree n = " << n << "\n";
		n = 0;
		while (load_tree(f, w)) n ++;
		msg(DBG_DEBUG) << "loaded " << n << " subtrees\n";
		n = 0;
		while (w.where() != sn) { // return to starting point
			if (w.where()->up()) w.up();
			else {
				msg(DBG_ERROR) << "Walk lost during SGF tree loading!\n";
				// try to fix
				w.set_node(sn);
			}
			n ++;
		}
		skip_white(f);
		if (f.eof() || f.peek() != ')') {
			msg(DBG_ERROR) << "No closing ')' for tree!\n";
			return false;
		}
		f.get();
		return true;
	}

	string sgf_quote(string s)
	{
		string r;
		for (unsigned j = 0; j < s.length(); j ++) {
			if (s[j] == ']' || s[j] == '\\') r += '\\';
			r += s[j];
		}
		return r;
	}

	string loc_to_sgf(int loc, ccgo::Board & board)
	{
		string r;
		int x;
		int y;
		board.loc_to_xy(loc, x, y);
		y = board.flip_axis(y);
		r += ('a' + x);
		r += ('a' + y);
		return r;
	}

	void save_marks(ccgo::Walk & w, ostream & f)
	{
		ccgo::Node * n = w.where();
		ccgo::Marking * m = (ccgo::Marking *) n->get_prop(ccgo::Marking::the_id());
		if (m) {
			vector<int> cr;
			vector<int> tr;
			vector<int> sq;
			vector<int> ma;
			vector<int> sl;
			vector<int> tw;
			vector<int> tb;

			for (int i = 0; i < w.loc_range(); i ++) switch (m->get_mark(i)) {
			case ccgo::MARK_NONE:
				break;
			case ccgo::MARK_CIRCLE:
				cr.push_back(i);
				break;
			case ccgo::MARK_TRIANGLE:
				tr.push_back(i);
				break;
			case ccgo::MARK_SQUARE:
				sq.push_back(i);
				break;
			case ccgo::MARK_CROSS:
				ma.push_back(i);
				break;
			case ccgo::MARK_SELECT:
				sl.push_back(i);
				break;
			case ccgo::MARK_W_TERRITORY:
				tw.push_back(i);
				break;
			case ccgo::MARK_B_TERRITORY:
				tb.push_back(i);
				break;
			}

			if (cr.size()) { // circle
				f << "CR";
				for (vector<int>::iterator i = cr.begin(); i != cr.end(); i ++) {
					f << '[' << loc_to_sgf(* i, w) << ']';
				}
			}

			if (tr.size()) { // triangle
				f << "TR";
				for (vector<int>::iterator i = tr.begin(); i != tr.end(); i ++) {
					f << '[' << loc_to_sgf(* i, w) << ']';
				}
			}

			if (sq.size()) { // square
				f << "SQ";
				for (vector<int>::iterator i = sq.begin(); i != sq.end(); i ++) {
					f << '[' << loc_to_sgf(* i, w) << ']';
				}
			}

			if (ma.size()) { // mark
				f << "MA";
				for (vector<int>::iterator i = ma.begin(); i != ma.end(); i ++) {
					f << '[' << loc_to_sgf(* i, w) << ']';
				}
			}

			if (sl.size()) { // select
				f << "SL";
				for (vector<int>::iterator i = sl.begin(); i != sl.end(); i ++) {
					f << '[' << loc_to_sgf(* i, w) << ']';
				}
			}

			if (tw.size()) { // territory of white
				f << "TW";
				for (vector<int>::iterator i = tw.begin(); i != tw.end(); i ++) {
					f << '[' << loc_to_sgf(* i, w) << ']';
				}
			}

			if (tb.size()) { // territory of white
				f << "TB";
				for (vector<int>::iterator i = tb.begin(); i != tb.end(); i ++) {
					f << '[' << loc_to_sgf(* i, w) << ']';
				}
			}
		}
		ccgo::Labels * l = (ccgo::Labels *) n->get_prop(ccgo::Labels::the_id());
		if (l && l->get_labels().size()) {
			f << "LB";
			const map<int, string> & lb = l->get_labels();
			for (map<int, string>::const_iterator i = lb.begin(); i != lb.end(); i ++) {
				f << '[' << loc_to_sgf(i->first, w) << ':' << sgf_quote(i->second) << ']';
			}
		}
		ccgo::Comment * c = (ccgo::Comment *) n->get_prop(ccgo::Comment::the_id());
		if (c) {
			f << "C[" << sgf_quote(c->get_comment()) << ']';
		}
	}

	void save_branch(ccgo::Walk & w, ostream & f)
	{
		f << ";";
		save_marks(w, f);

		ccgo::Node * n = w.where();
		if (n->get_move() != ccgo::MOVE_NONE) { // move node
			int u = n->up() ? n->up()->get_num() : 0;
			if (n->get_num() != u + 1) {
				f << "MN[" << n->get_num() << "]";
			}
			f << (w.get_turn() == ccgo::Position::TURN_BLACK ? 'W' : 'B');
			if (n->get_move() < 0) f << "[]";
			else f << '[' << loc_to_sgf(n->get_move(), w) << ']';
		} else if (n->get_delta()) { // setup node
			ccgo::Delta * d = n->get_delta();
			vector<int> ae;
			vector<int> ab;
			vector<int> aw;
			for (int i = 0; i < w.loc_range(); i ++) if (d->is_changed(i)) switch (d->get_new_state(i)) {
			case ccgo::Board::STATE_EMPTY:
				ae.push_back(i);
				break;
			case ccgo::Board::STATE_BLACK:
				ab.push_back(i);
				break;
			case ccgo::Board::STATE_WHITE:
				aw.push_back(i);
				break;
			}
			if (ae.size()) {
				f << "AE";
				for (vector<int>::iterator i = ae.begin(); i != ae.end(); i ++) {
					f << '[' << loc_to_sgf(* i, w) << ']';
				}
			}
			if (ab.size()) {
				f << "AB";
				for (vector<int>::iterator i = ab.begin(); i != ab.end(); i ++) {
					f << '[' << loc_to_sgf(* i, w) << ']';
				}
			}
			if (aw.size()) {
				f << "AW";
				for (vector<int>::iterator i = aw.begin(); i != aw.end(); i ++) {
					f << '[' << loc_to_sgf(* i, w) << ']';
				}
			}
			if (d->get_turn_flip()) {
				f << "PL[" << (w.get_turn() == ccgo::Position::TURN_BLACK ? 'B' : 'W') << ']';
			}
		}

		if (n->down_num() > 1) for (unsigned i = 0; i < n->down_num(); i ++) {
			f << "\n(";
			w.down(i);
			save_branch(w, f);
			w.up();
			f << ")";
		} else if (n->down_num()) {
			w.down(0u);
			save_branch(w, f);
			w.up();
		}
	}
}

ccgo::Game * ccgo::load_sgf(istream & f)
{
	skip_white(f);
	if (f.eof() || f.peek() != '(') return 0;  // no tree!
	ccgo::Game * g = new ccgo::Game;
	ccgo::Rule r;
	r.opt_overwrite_stone = true;
	r.opt_ignore_ko = true;
	r.opt_allow_suicide = true;
	g->set_rule(r);
	f.get(); // '('
	::Node m;
	// read the root node
	if (! m.load(f)) {
		delete g;
		return 0;
	}
	const ::Prop * p;
	int bz = 19; // default board size
	if ((p = m.find("SZ"))) { // board size
		bz = p->as_int();
		// TODO: check range
	}
	ccgo::Position b(bz);
	g->set_init(b);
	ccgo::Walk w;
	w.set_node(g);
	if ((p = m.find("KM"))) { // komi
		g->set_komi(p->as_real());
	}
	if ((p = m.find("GN"))) { // game name
		g->set_name(p->as_string());
	}
	if ((p = m.find("HA"))) { // handicap
		int h = p->as_int();
		if ((p = m.find("AB"))) { // specified handicap positions
			vector<int> l = p->ar_loc(b);
			w.game_set_handi(0); // FIXME
			for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) w.game_add_handi(* i); // FIXME
		} else { // no handicap position, try using default positions
			msg(DBG_WARNING) << "no handicap stone specified for handicap game, assuming Japanese\n";
			w.game_set_handi(h); // FIXME
		}
	} else if ((p = m.find("AB"))) {
		vector<int> l = p->ar_loc(b);
		for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) g->get_init()->set_state(* i, ccgo::Board::STATE_BLACK);
	}
	if ((p = m.find("AW"))) { // adding white stones
		vector<int> l = p->ar_loc(b);
		for (vector<int>::iterator i = l.begin(); i != l.end(); i ++) g->get_init()->set_state(* i, ccgo::Board::STATE_WHITE);
	}
	if ((p = m.find("PL"))) { // set turn
		ccgo::Position::Turn tn;
		if (p->as_char() == 'B') tn = ccgo::Position::TURN_BLACK;
		else if (p->as_char() == 'W') tn = ccgo::Position::TURN_WHITE;
		else tn = w.get_turn();
		w.node_set_turn(tn); // FIXME
	}
	set_marks(g, m);
	while (load_node(f, w));
	while (load_tree(f, w));
	if (f.eof() || f.peek() != ')') {
		msg(DBG_ERROR) << "Failed to load SGF file\n";
		delete g;
		return 0;
	}
	f.get();
	return g;
}

void ccgo::save_sgf(ccgo::Game * g, ostream & f)
{
	assert(g->get_init());
	ccgo::Walk w;
	w.set_node(g);

	f << "(;";
	f << "SZ[" << w.get_size() << "]";
	if (g->get_name() != "") f << "GN[" << sgf_quote(g->get_name()) << "]";
	if (g->get_komi() != 0) f << "KM[" << g->get_komi() << "]";
	vector<int> ab;
	vector<int> aw;
	for (int i = 0; i < w.loc_range(); i ++) switch (w.get_state(i)) {
	case ccgo::Board::STATE_BLACK:
		ab.push_back(i);
		break;
	case ccgo::Board::STATE_WHITE:
		aw.push_back(i);
		break;
	case ccgo::Board::STATE_EMPTY:
		break;
	}
	bool init_set = false;;
	if (w.get_turn() == ccgo::Position::TURN_WHITE && ! aw.size()) { // assuming handicap
		f << "HA[" << ab.size() << "]";
		init_set = true;
	}
	if (ab.size()) {
		f << "AB";
		for (vector<int>::iterator i = ab.begin(); i != ab.end(); i ++) {
			f << '[' << loc_to_sgf(* i, w) << ']';
		}
		init_set = true;
	}
	if (aw.size()) {
		f << "AW";
		for (vector<int>::iterator i = aw.begin(); i != aw.end(); i ++) {
			f << '[' << loc_to_sgf(* i, w) << ']';
		}
		init_set = true;
	}
	if (init_set || w.get_turn() == ccgo::Position::TURN_WHITE) {
		f << "PL[" << (w.get_turn() == ccgo::Position::TURN_WHITE ? 'W' : 'B') << ']';
	}

	save_marks(w, f);

	if (g->down_num() > 1) for (unsigned i = 0; i < g->down_num(); i ++) {
		f << "\n(";
		w.down(i);
		save_branch(w, f);
		w.up();
		f << ")";
	} else if (g->down_num()) {
		w.down(0u);
		save_branch(w, f);
		w.up();
	}
	f << ")\n";
}
