/* ccgo: go/sgf.cc
 *
 * Copyright (C) 2006 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
#include <go/sgf.hh>
#include <go/walk.hh>
#include <debug.hh>
#include <sstream>
using namespace std;
using namespace go;

namespace { // avoid namespace contamination
	void skip_white(istream & f)
	{
		while (! f.eof() && isspace(f.peek())) f.get();
	}

	struct Label
	{
		Loc l;
		string s;
	};

	class Prop
	{
		string id;
		vector<string> raw_values;
	public:
		bool load(istream &);
		void save(ostream &) const;
		const string & get_id() const;
		int as_int() const;
		unsigned as_uint() const;
		double as_real() const;
		Loc as_loc(unsigned board_size) const;
		vector<Loc> ar_loc(unsigned board_size) const;
		char as_char() const;
		const string & as_string() const;
		Label as_label(unsigned board_size) const;
		vector<Label> ar_label(unsigned board_size) const;
	};

	class sNode // internal representation of an SGF node
	{
		vector<Prop> prop_list;
	public:
		bool load(istream &);
		void save(ostream &) const;
		const Prop * find(const string &) const;
	};

	class SGFWalk :
		private Walk
	{
		void set_marks(sNode * m);
		bool load_node(istream & f);
		bool load_tree(istream & f);
		void save_marks(ostream & f);
		void save_branch(ostream & f);
	public:
		go::Game * load_sgf(istream & f); // load a game tree from 'f'
		void save_sgf(ostream & f, Game * g); // save the game tree 'g' to 'f'
	};

	bool Prop::load(istream & f)
	{
		id = "";
		raw_values.clear();
		skip_white(f);
		if (f.eof() || ! isalpha(f.peek())) return false;
		// read id string
		do {
			id += f.get();
		} while ((! f.eof()) && isalpha(f.peek())); 
		do {
			skip_white(f);
			if (f.eof() || f.peek() != '[') break;
			f.get();
			string s = "";
			bool escape = false;
			int line_break = 0;
			while ((! f.eof()) && (f.peek() != ']' || escape)) {
				char c = f.get();
				if (c == '\r') {
					if (line_break == 1) {
						line_break = 0;
						continue;
					}
					c = '\n';
					line_break = 2;
				}
				else if (c == '\n') {
					if (line_break ==2) {
						line_break = 0;
						continue;
					}
					line_break = 1;
				}
				else line_break = 0;
				if (escape) {
					escape = false;
					if (c == '\n') continue;
				}
				else escape = c == '\\';
				if (! escape) s += c;
			}
			if (f.eof()) {
				msg(DBG_ERROR) << "incomplete value field!\n";
			}
			else f.get();
			raw_values.push_back(s);
		} while (true);
		return true;
	}

	void Prop::save(ostream & f) const
	{
		f << id;
		for (vector<string>::const_iterator i = raw_values.begin(); i != raw_values.end(); i ++) {
			f << '[';
			for (unsigned j = 0; j < i->length(); j ++) {
				if ((* i)[j] == ']' || (* i)[j] == '\\') f << '\\';
				f << (* i)[j];
			}
			f << ']';
		}
	}

	const string & Prop::get_id() const
	{
		return id;
	}

	int Prop::as_int() const
	{
		if (raw_values.size() != 1) return 0;
		istringstream s(raw_values[0]);
		int i;
		s >> i;
		return i;
	}

	unsigned Prop::as_uint() const
	{
		if (raw_values.size() != 1) return 0;
		istringstream s(raw_values[0]);
		unsigned i;
		s >> i;
		return i;
	}

	double Prop::as_real() const
	{
		if (raw_values.size() != 1) return 0;
		istringstream s(raw_values[0]);
		double d;
		s >> d;
		return d;
	}

	Loc Prop::as_loc(unsigned board_size) const
	{
		string s = raw_values[0];
		if (s.length() < 2) return board_size * board_size;
		int x = s[0] - 'a';
		int y = s[1] - 'a';
		if (x < 0 || x >= int(board_size)) return board_size * board_size;
		if (y < 0 || y >= int(board_size)) return board_size * board_size;
		// compressed points can not be handle here!
		return Loc(x + board_size * (board_size - y - 1));
	}

	vector<Loc> Prop::ar_loc(unsigned board_size) const
	{
		vector<Loc> r;

		for (vector<string>::const_iterator i = raw_values.begin();
		     i != raw_values.end(); i ++) {
			string s = * i;
			if (s.length() < 2) continue;
			int x = s[0] - 'a';
			int y = s[1] - 'a';
			if (s.length() == 2) {
				r.push_back(Loc(x + board_size * (board_size - y - 1)));
				continue;
			}
			if (s[2] == ':') { //compressed point
				msg(DBG_DEBUG) << "compressed points\n";
				if (s.length() < 5) continue;
				int x2 = s[3] - 'a';
				int y2 = s[4] - 'a';
				if (x > x2) {
					int t = x2;
					x2 = x;
					x = t;
				}
				if (y > y2) {
					int t = y2;
					y2 = y;
					y = t;
				}
				for (int xx = x; xx <= x2; xx++) for (int yy = y; yy <= y2; yy ++) r.push_back(Loc(xx + board_size * (board_size - yy - 1)));
				continue;
			}
			msg(DBG_ERROR) << "sgf: error in compressed format\n";
		}
		return r;
	}

	char Prop::as_char() const
	{
		return raw_values[0][0];
	}

	const string & Prop::as_string() const
	{
		return raw_values[0];
	}

	Label Prop::as_label(unsigned board_size) const
	{
		Label l;
		string s = raw_values[0];
		l.l = Loc(s[0] - 'a' + board_size * (board_size - s[1] + 'a' - 1));
		// assert(s[2] == ':');
		l.s = s.substr(3);
		return l;
	}

	vector<Label> Prop::ar_label(unsigned board_size) const
	{
		vector<Label> la;
		for (unsigned i = 0; i < raw_values.size(); i ++) {
			string s = raw_values[i];
			Label l;
			l.l = Loc(s[0] - 'a' + board_size * (board_size - s[1] + 'a' - 1));
			// assert(s[2] == ':');
			l.s = s.substr(3);
			la.push_back(l);
		}
		return la;
	}


	bool sNode::load(istream & f)
	{
		prop_list.clear();
		skip_white(f);
		if (f.eof() || f.peek() != ';') return false;
		f.get();
		Prop p;
		while (p.load(f)) prop_list.push_back(p);
		return true;
	}

	void sNode::save(ostream & f) const
	{
		f << ';';
		for (vector<Prop>::const_iterator i = prop_list.begin(); i != prop_list.end(); i ++) {
			i->save(f);
		}
		// f << endl;
	}

	const Prop * sNode::find(const string & s) const
	{
		for (vector<Prop>::const_iterator i = prop_list.begin(); i != prop_list.end(); i ++) {
			if (s == i->get_id()) return & (* i);
		}
		return NULL;
	}


	string sgf_quote(string s)
	{
		string r;
		for (unsigned j = 0; j < s.length(); j ++) {
			if (s[j] == ']' || s[j] == '\\') r += '\\';
			r += s[j];
		}
		return r;
	}

	string loc_to_sgf(Loc loc, unsigned board_size)
	{
		string r;
		unsigned x = loc % board_size;
		unsigned y = board_size - loc / board_size - 1;
		r += ('a' + x);
		r += ('a' + y);
		return r;
	}

	using namespace go;


	// set non-game-related node properties
	void SGFWalk::set_marks(sNode * m)
	{
		const Prop * p;

		if ((p = m->find("N"))) { // node name
			// p->as_string();
		}
		if ((p = m->find("C"))) { // comment
			where()->set_text(p->as_string());
		}
		if ((p = m->find("LB"))) {	// labels
			map<Loc, string> lbs;
			vector<Label> l = p->ar_label(get_size());
			for (vector<Label>::iterator i = l.begin(); i != l.end(); i ++) lbs[i->l] = i->s;
			where()->set_labels(lbs);
		}
		map<Loc, Mark> mk;
		if ((p = m->find("CR"))) { // circle
			vector<Loc> l = p->ar_loc(get_size());
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) mk[* i] = MARK_CIRCLE;
		}
		if ((p = m->find("TR"))) { // triangle
			vector<Loc> l = p->ar_loc(get_size());
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) mk[* i] = MARK_TRIANGLE;
		}
		if ((p = m->find("SQ"))) { // square
			vector<Loc> l = p->ar_loc(get_size());
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) mk[* i] = MARK_SQUARE;
		}
		if ((p = m->find("MA"))) { // mark
			vector<Loc> l = p->ar_loc(get_size());
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) mk[* i] = MARK_CROSS;
		}
		if ((p = m->find("SL"))) { // select
			vector<Loc> l = p->ar_loc(get_size());
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) mk[* i] = MARK_SELECT;
		}
		if ((p = m->find("TW"))) { // territory of white
			vector<Loc> l = p->ar_loc(get_size());
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) mk[* i] = MARK_W_TERRITORY;
		}
		if ((p = m->find("TB"))) { // territory of black
			vector<Loc> l = p->ar_loc(get_size());
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) mk[* i] = MARK_B_TERRITORY;
		}
		if (mk.size()) where()->set_marking(mk);
	}

	bool SGFWalk::load_node(istream & f)
	{
		sNode m;
		if (! m.load(f)) return false;
		
		const Prop * p;

		// check if there is any "setup" operations
		bool any_setup = false;

		Node * nd = where();

		Board b = * this;
		Turn tn = nd->get_turn();
		unsigned cb = nd->get_capb();
		unsigned cw = nd->get_capw();
		set<Loc> k = nd->get_ko();

		if ((p = m.find("AB"))) { // add black stones...
			vector<Loc> l = p->ar_loc(get_size());
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) b.set_state(* i, Black);
			any_setup = true;
		}

		if ((p = m.find("AW"))) { // add white stones
			vector<Loc> l = p->ar_loc(get_size());
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) b.set_state(* i, White);
			any_setup = true;
		}

		if ((p = m.find("AE"))) { // add empty sites
			vector<Loc> l = p->ar_loc(get_size());
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) b.set_state(* i, Empty);
			any_setup = true;
		}

		if ((p = m.find("PL"))) { // set turn
			if (p->as_char() == 'B') tn = Black;
			else if (p->as_char() == 'W') tn = White;
			else msg(DBG_ERROR) << "neither 'B' nor 'W' in PL!\n";
			any_setup = true;
		}

		Node * n = 0;
		if (any_setup) { // a setup node
			if ((p = m.find("B"))) { // illegal in FF[4]
				msg(DBG_ERROR) << "illegal 'B' in a setup node\n";
				Loc i = p->as_loc(get_size());
				if (i != get_size() * get_size()) b.set_state(i, Black);
				tn = White;
			}
			if ((p = m.find("W"))) { // illegal in FF[4]
				msg(DBG_ERROR) << "illegal 'W' in a setup node\n";
				Loc i = p->as_loc(get_size());
				if (i != get_size() * get_size()) b.set_state(i, White);
				tn = Black;
			}
			n = add_setup(b, tn, cb, cw, k);
		}
		else {
			// a move node
			Move move = MoveNone;
			if ((p = m.find("B"))) {  // only one of B or W should happen
				tn = Black;
				move = p->as_loc(get_size());
			}
			if ((p = m.find("W"))) {
				if (move != MoveNone) { // B existed!!
					msg(DBG_ERROR) << "Both B & W properties exist in the same node!\n";
				}
				tn = White;
				move = p->as_loc(get_size());
			}
			if (move == MoveNone) { // no move, no setup ??
				msg(DBG_ERROR) << "No move nor setup in node.\n";
			}
			else {
				if (move == int (get_size() * get_size())) move = MovePass;
				if (where()->get_turn() != tn) where()->flip_turn(); // correct turn
				n = add_move(move);
			}
		}
		if (! n) n = add_node(); // empty setup node
		// substitutive properties
		if ((p = m.find("MN"))) {
			n->set_move_num(p->as_int());
		}

		n->link();
		move_to(n);
		set_marks(& m);
		return true;
	}

	bool SGFWalk::load_tree(istream & f)
	{
		Node * sn = where(); // keep starting point
		skip_white(f);
		if (f.eof() || f.peek() != '(') return false;
		f.get();
		unsigned n = 0;
		while (load_node(f)) n ++;
		if (! n) return false; // there should be one node at least
		msg(DBG_DEBUG) << "loaded tree n = " << n << "\n";
		n = 0;
		while (load_tree(f)) n ++;
		msg(DBG_DEBUG) << "loaded " << n << " subtrees\n";
		n = 0;
		move_to(sn); // return to starting point
		skip_white(f);
		if (f.eof() || f.peek() != ')') {
			msg(DBG_ERROR) << "No closing ')' for tree!\n";
			return false;
		}
		f.get();
		return true;
	}

	void SGFWalk::save_marks(ostream & f)
	{
		vector<Loc> cr;
		vector<Loc> tr;
		vector<Loc> sq;
		vector<Loc> ma;
		vector<Loc> sl;
		vector<Loc> tw;
		vector<Loc> tb;
		const map<Loc, Mark> & mk = where()->get_marking();
		for (map<Loc, Mark>::const_iterator i = mk.begin(); i != mk.end(); i ++) switch (i->second) {
			case MARK_NONE:
				break;
			case MARK_CIRCLE:
				cr.push_back(i->first);
				break;
			case MARK_TRIANGLE:
				tr.push_back(i->first);
				break;
			case MARK_SQUARE:
				sq.push_back(i->first);
				break;
			case MARK_CROSS:
				ma.push_back(i->first);
				break;
			case MARK_SELECT:
				sl.push_back(i->first);
				break;
			case MARK_W_TERRITORY:
				tw.push_back(i->first);
				break;
			case MARK_B_TERRITORY:
				tb.push_back(i->first);
				break;
			default:
				;
			}
		if (cr.size()) { // circle
			f << "CR";
			for (vector<Loc>::iterator i = cr.begin(); i != cr.end(); i ++) f << '[' << loc_to_sgf(* i, get_size()) << ']';
		}
		if (tr.size()) { // triangle
			f << "TR";
			for (vector<Loc>::iterator i = tr.begin(); i != tr.end(); i ++) f << '[' << loc_to_sgf(* i, get_size()) << ']';
		}
		if (sq.size()) { // square
			f << "SQ";
			for (vector<Loc>::iterator i = sq.begin(); i != sq.end(); i ++) f << '[' << loc_to_sgf(* i, get_size()) << ']';
		}
		if (ma.size()) { // mark
			f << "MA";
			for (vector<Loc>::iterator i = ma.begin(); i != ma.end(); i ++) f << '[' << loc_to_sgf(* i, get_size()) << ']';
		}
		if (sl.size()) { // select
			f << "SL";
			for (vector<Loc>::iterator i = sl.begin(); i != sl.end(); i ++) f << '[' << loc_to_sgf(* i, get_size()) << ']';
		}
		if (tw.size()) { // territory of white
			f << "TW";
			for (vector<Loc>::iterator i = tw.begin(); i != tw.end(); i ++) f << '[' << loc_to_sgf(* i, get_size()) << ']';
		}
		if (tb.size()) { // territory of white
			f << "TB";
			for (vector<Loc>::iterator i = tb.begin(); i != tb.end(); i ++) f << '[' << loc_to_sgf(* i, get_size()) << ']';
		}
		if (where()->get_labels().size()) {
			const map<Loc, string> & lb = where()->get_labels();
			f << "LB";
			for (map<Loc, string>::const_iterator i = lb.begin(); i != lb.end(); i ++)
				f << '[' << loc_to_sgf(i->first, get_size()) << ':' << sgf_quote(i->second) << ']';
		}
		if (where()->get_text().size()) f << "C[" << sgf_quote(where()->get_text()) << ']';
	}

	void SGFWalk::save_branch(ostream & f)
	{
		f << ";";
		save_marks(f);
		Node * n = where();
		if (n->get_move() != MoveNone) { // move node
			// int u = n->up() ? n->up()->get_num() : 0;
			// if (n->get_num() != u + 1) {
			//       f << "MN[" << n->get_num() << "]";
			// }
			f << (n->get_turn() == Black ? 'W' : 'B');
			if (n->get_move() == MovePass) f << "[]";
			else f << '[' << loc_to_sgf(n->get_move(), get_size()) << ']';
		}
		else { // setup node
			vector<Loc> ae;
			vector<Loc> ab;
			vector<Loc> aw;
			Board b = * this;
			b += n->get_delta();
			for (Loc i = 0; i < range(); i ++) if (b.state(i) != state(i)) switch (b.state(i)) {
					case Empty:
						ae.push_back(i);
						break;
					case Black:
						ab.push_back(i);
						break;
					case White:
						aw.push_back(i);
						break;
					}
			if (ae.size()) {
				f << "AE";
				for (vector<Loc>::iterator i = ae.begin(); i != ae.end(); i ++) f << '[' << loc_to_sgf(* i, get_size()) << ']';
			}
			if (ab.size()) {
				f << "AB";
				for (vector<Loc>::iterator i = ab.begin(); i != ab.end(); i ++) f << '[' << loc_to_sgf(* i, get_size()) << ']';
			}
			if (aw.size()) {
				f << "AW";
				for (vector<Loc>::iterator i = aw.begin(); i != aw.end(); i ++) f << '[' << loc_to_sgf(* i, get_size()) << ']';
			}
			if (n->up()) {
				if (n->up()->get_turn() != n->get_turn()) f << "PL[" << (n->get_turn() == Black ? 'B' : 'W') << ']';
			}
			else if (n->get_turn() != Black) f << "PL[W]";
		}
		if (n->down().size() > 1) for (vector<Node *>::const_iterator i = n->down().begin(); i != n->down().end(); i ++) {
				f << "\n(";
				move_to(* i);
				save_branch(f);
				move_to(n);
				f << ")";
			}
		else if (n->down().size()) {
			move_to(n->down()[0]);
			save_branch(f);
			move_to(n);
		}
	}
	
	Game * SGFWalk::load_sgf(istream & f)
	{
		skip_white(f);
		if (f.eof() || f.peek() != '(') return 0;  // no tree!
		f.get(); // '('
		sNode m;
		// read the root node
		if (! m.load(f)) return 0;
		
		// root node read fine, creating the game
		const Prop * p;
		unsigned bz = 19; // default board size
		if ((p = m.find("SZ"))) { // board size
			bz = p->as_uint();
		}
		Board b;
		b.resize(bz);
		Turn tn = Black;

		if ((p = m.find("HA"))) { // handicap
			unsigned h = p->as_uint();
			vector<Loc> l;
			if ((p = m.find("AB"))) { // specified handicap positions
				l = p->ar_loc(bz);
			}
			else { // no handicap position, try using default positions
				msg(DBG_WARNING) << "no handicap stone specified for handicap game, assuming Japanese\n";
				l = b.handi_sites(h);
			}
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) b.set_state(* i, Black);
			tn = White;
		}
		else if ((p = m.find("AB"))) {
			vector<Loc> l = p->ar_loc(bz);
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) b.set_state(* i, Black);
		}
		if ((p = m.find("AW"))) { // adding white stones
			vector<Loc> l = p->ar_loc(bz);
			for (vector<Loc>::iterator i = l.begin(); i != l.end(); i ++) b.set_state(* i, White);
		}
		if ((p = m.find("PL"))) { // set turn
			if (p->as_char() == 'B') tn = Black;
			else if (p->as_char() == 'W') tn = White;
			else msg(DBG_ERROR) << "neither 'B' nor 'W' for PL[]\n";
		}

		Game * g = new Game(b, MoveNone, tn, 0, 0, set<Loc>());

		if ((p = m.find("KM"))) { // komi
			g->set_komi(p->as_real());
		}
		if ((p = m.find("GN"))) { // game name
			// g->set_name(p->as_string()); FIXME
		}
		move_to(g);
		set_marks(& m);
		while (load_node(f));
		while (load_tree(f));
		if (f.eof() || f.peek() != ')') {
			msg(DBG_ERROR) << "Failed to load SGF file\n";
			delete g;
			return 0;
		}
		f.get();
		return g;
	}

	void SGFWalk::save_sgf(ostream & f, Game * g)
	{
		move_to(g);
		
		f << "(;";
		f << "SZ[" << get_size() << "]";
		// if (g->get_name() != "") f << "GN[" << sgf_quote(g->get_name()) << "]";
		// if (g->get_komi() != 0) f << "KM[" << g->get_komi() << "]";
		vector<Loc> ab;
		vector<Loc> aw;
		for (Loc i = 0; i < range(); i ++) switch (state(i)) {
			case Black:
				ab.push_back(i);
				break;
			case White:
				aw.push_back(i);
				break;
			case Empty:
				break;
			}
		bool init_set = false;
		if (g->get_turn() == White && ! aw.size()) { // assuming handicap
			f << "HA[" << ab.size() << "]";
			init_set = true;
		}
		if (ab.size()) {
			f << "AB";
			for (vector<Loc>::iterator i = ab.begin(); i != ab.end(); i ++) {
				f << '[' << loc_to_sgf(* i, get_size()) << ']';
			}
			init_set = true;
		}
		if (aw.size()) {
			f << "AW";
			for (vector<Loc>::iterator i = aw.begin(); i != aw.end(); i ++) {
				f << '[' << loc_to_sgf(* i, get_size()) << ']';
			}
			init_set = true;
		}
		if (g->get_turn() == White) f << "PL[W]";
		else if (init_set) f << "PL[B]";

		save_marks(f);
		
		if (g->down().size() > 1) for (vector<Node *>::const_iterator i = g->down().begin(); i != g->down().end(); i ++) {
				f << "\n(";
				move_to(* i);
				save_branch(f);
				move_to(g);
				f << ")";
			}
		else if (g->down().size()) {
			move_to(g->down()[0]);
			save_branch(f);
			move_to(g);
		}
		f << ")\n";
	}
}


Game * go::read_sgf(istream & f)
{
	return SGFWalk().load_sgf(f);
} 

void go::write_sgf(ostream & f, Game * g)
{
	SGFWalk().save_sgf(f, g);
}
