/* ccgo: go/node.hh
 *
 * Copyright (C) 2006 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
#ifndef GO_NODE_HH
#define GO_NODE_HH
#include <go/delta.hh>
#include <vector>
#include <map>
#include <set>
#include <string>
namespace go
{
	class Game;
	class WalkBase;
	class Node
	{
		Game * game; // point to the game node (root of the tree)
		Node * ulink; // link pointing up
		std::vector<Node *> dlink; // down pointing links
		unsigned level; // distance from the root
		bool linked;

		// position of the game, properties fixed upon creating of a Node
		Delta delta; // change to the board, only the board change is mandatory, all following are advisory
		Move move;
		Turn turn;
		unsigned capb; // captured by black so far
		unsigned capw; // captured by white so far
		std::set<Loc> ko; // locations of ko (forbidden) sites
		unsigned move_num; // move number (not really essential)

		std::set<WalkBase *> lot; // WalkBase's staying at this Node

		// additional properties
		std::string text;
		std::map<Loc, Mark> marking;
		std::map<Loc, std::string> labels;
	public:
		Node(Node * parent, const Delta & d, Move mv, Turn tn, unsigned cb, unsigned cw, const std::set<Loc> & k);
		// Node(Node * parent); // * Node is not linked by the parent upon creation. It need to be setup before link().
		virtual ~Node();

		void link(); // actually link the node to parent, move, delta, capw, capb, and ko should be set
		void unlink(); // delete all child and unlink self from parent
		void link_child(Node * node);
		bool can_unlink() const;

		// Access for WalkBase
		void park(WalkBase * walk);
		void unpark(WalkBase * walk);

		// tree geometry
		Game * get_game();
		unsigned get_level();
		Node * up();
		const std::vector<Node *> & down();
		unsigned get_order();

		// game properties
		const Delta & get_delta() const;
		Move get_move() const;
		Turn get_turn() const;
		unsigned get_move_num() const;
		unsigned get_capb() const;
		unsigned get_capw() const;
		const std::set<Loc> & get_ko() const;

		// special modifications to the advisory part of the position
		void flip_turn();
		void set_move_num(unsigned num);

		// markup properties
		const std::string & get_text() const;
		const std::map<Loc, Mark> & get_marking() const;
		const std::map<Loc, std::string> & get_labels() const;

		void set_text(const std::string & txt);
		void set_marking(const std::map<Loc, Mark> & mkng);
		void set_labels(const std::map<Loc, std::string> & lbls);
		void set_mark(Loc loc, Mark mark);
		void set_label(Loc loc, const std::string & lbl);

		// backdoors
		friend class Walk;
		friend class Game;
	};
}
#endif // GO_NODE_HH
