/* ccgo: go/node.cc
 * 
 * Copyright (C) 2006 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
#include <go/node.hh>
#include <go/walk_base.hh>
#include <debug.hh>
#include <iostream>
#include <sstream>
using namespace go;
using namespace std;

// members of class Node
Node::Node(Node * parent, const Delta & d, Move mv, Turn tn, unsigned cb, unsigned cw, const set<Loc> & k) :
	linked(false),
	delta(d),
	move(mv),
	turn(tn),
	capb(cb),
	capw(cw),
	ko(k)
{
	if (! parent) {
		// game = dynamic_cast<Game *>(this);
		game = 0; // need to be set by game
		ulink = 0;
		move_num = 0;
		level = 0;
	}
	else {
		game = parent->game;
		ulink = parent;
		move_num = parent->move_num;
		level = parent->level + 1;
	}
	if (move != MoveNone) move_num ++;
}

Node::~Node()
{
	assert(! linked);
}

void Node::link()
{
	assert(ulink && ! linked);
	linked = true;
	ulink->link_child(this);
}

void Node::unlink()
{
	// delete all children
	while (dlink.size()) {
		Node * n = dlink.back();
		n->unlink();
		delete n;
	}

	if (ulink) {
		// unlink self from parent
		for (vector<Node *>::iterator i = ulink->dlink.begin(); i != ulink->dlink.end(); i ++) if (this == * i) {
			ulink->dlink.erase(i);
			break;
		}
		linked = false;
	}

	// notify parked WalkBase
	unsigned l = lot.size(); // safty count... prevent infinite loop
	while (lot.size()) {
		assert(l);
		(* lot.begin())->note_gone();
		l --;
	}
}

void Node::link_child(Node * node)
{
	assert(node && node->ulink == this);
	dlink.push_back(node);
	if (dlink.size() == 1) {
		set<WalkBase *> ll = lot;
		for (set<WalkBase *>::iterator i = ll.begin(); i != ll.end(); i ++) (* i)->note_stem();
	}
}

bool Node::can_unlink() const
{
	for (set<WalkBase *>::const_iterator i = lot.begin(); i != lot.end(); i ++) if (! (* i)->is_movable()) return false;
	for (vector<Node *>::const_iterator i = dlink.begin(); i != dlink.end(); i ++) if (! (* i)->can_unlink()) return false;
	return true;
}

void Node::park(WalkBase * walk)
{
	lot.insert(walk);
}

void Node::unpark(WalkBase * walk)
{
	assert(lot.find(walk) != lot.end());
	lot.erase(walk);
}

Game * Node::get_game()
{
	return game;
}

unsigned Node::get_level()
{
	return level;
}

Node * Node::up()
{
	return ulink;
}

const vector<Node *> & Node::down()
{
	return dlink;
}

unsigned Node::get_order()
{
	assert(ulink);
	unsigned ord = 0;
	for (vector<Node *>::const_iterator i = ulink->down().begin(); i != ulink->down().end(); i ++) {
		if (this == * i) break;
		ord ++;
	}
	return ord;
}

const Delta & Node::get_delta() const
{
	return delta;
}

Move Node::get_move() const
{
	return move;
}

Turn Node::get_turn() const
{
	return turn;
}

unsigned Node::get_move_num() const
{
	return move_num;
}

unsigned Node::get_capb() const
{
	return capb;
}

unsigned Node::get_capw() const
{
	return capw;
}

const set<Loc> & Node::get_ko() const
{
	return ko;
}

void Node::flip_turn()
{
	turn = flip(turn);
	// notify parked WalkBase
	for (set<WalkBase *>::iterator i = lot.begin(); i != lot.end(); i ++) (* i)->note_vary();
}

void Node::set_move_num(unsigned num)
{
	move_num = num;
	// notify parked WalkBase
	for (set<WalkBase *>::iterator i = lot.begin(); i != lot.end(); i ++) (* i)->note_vary();
}

const string & Node::get_text() const
{
	return text;
}

const map<Loc, Mark> & Node::get_marking() const
{
	return marking;
}

const map<Loc, string> & Node::get_labels() const
{
	return labels;
}

void Node::set_text(const string & txt)
{
	text = txt;
	for (set<WalkBase *>::iterator i = lot.begin(); i != lot.end(); i ++) (* i)->note_vary();
}

void Node::set_marking(const map<Loc, Mark> & mkng)
{
	marking = mkng;
	for (set<WalkBase *>::iterator i = lot.begin(); i != lot.end(); i ++) (* i)->note_vary();
}

void Node::set_labels(const map<Loc, string> & lbls)
{
	labels = lbls;
	for (set<WalkBase *>::iterator i = lot.begin(); i != lot.end(); i ++) (* i)->note_vary();
}

void Node::set_mark(Loc loc, Mark mark)
{
	if (mark) marking[loc] = mark;
	else marking.erase(loc);
}

void Node::set_label(Loc loc, const string & lbl)
{
	if (lbl.size()) labels[loc] = lbl;
	else labels.erase(loc);
}
