/* ccgo: go/go.hh
 *
 * Copyright (C) 2004 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#ifndef GO_GO0_HH
#define GO_GO0_HH

#include <vector>
#include <map>
#include <set>
#include <string>

namespace ccgo
{
	class Error
	{ // exceptions
	public:
		Error(std::string message);
		const std::string & get_message();
	private:
		std::string s;
	};

	// implement: conventional square board
	class Board
	{
	public:
		enum State {
			STATE_EMPTY = 0,
			STATE_BLACK = 1,
			STATE_WHITE = 2
		};
		enum Terri {
			TERRI_NEUTRAL = 0,
			TERRI_BLACK = 1,
			TERRI_WHITE = 2
		};
		enum {
			NO_LOC = - 1
		};

		Board(); // create a Board of no size
		Board(int size);
		Board(const Board & board); // copy constructor
		virtual ~Board();

		// modifications
		void copy(const Board & board);
		virtual void clear(); // clear board
		virtual void set_state(int loc, State state); // set State at "loc" to "state"
		virtual void set_handi(int handi); // place "handi" handicap stones on the board

		// queries
		int get_size() const;
		State get_state(int loc) const; // get State at "loc"
		std::vector<int> neighbors(int loc) const; // return neighboring sites of "loc"
		int loc_range() const; // return valid range of Loc

		// math
		int flip_axis(int x) const;
		int xy_to_loc(int x, int y) const;
		void loc_to_xy(int loc, int & x, int & y) const;

		// housekeeping functions
		void get_kill(int loc, std::vector<int> & kill, State put_state) const;
		// put sites with killed stones when place "put_state" at "loc" in "kill"
		int qi_group(int loc, std::vector<int> & group, State state, bool * marking_field) const;
		// return number of qi for the "group" including site at "loc" with "state", skip
		// sites which has been marked in "marking_field"

		// scoring functions
		void score_terri(Terri * terri) const;
		void find_terri(int loc, std::vector<int> & group, bool & w_seen, bool & b_seen, Terri * terri, bool * marks) const;
		void hang_group(int loc, State state, std::vector<int> & qi, Terri * terri, bool * marks) const;
		void get_group(int loc, std::vector<int> & group) const;
		void dead_group(int loc, std::vector<int> & group) const;
		void terri_group(int loc, State state, std::vector<int> & group, bool * marks) const;

		// functions provides a compact data structure of current board configuration
		int compact_size() const;
		void compact_get(void * data) const;

		bool loc_is_star(int loc) const;
		std::string loc_to_str(int loc) const;

		bool is_valid() const;
	private:
		int z;     // size
		State * b; // board
	};

	enum Move {
		MOVE_PASS = - 1,
		MOVE_NONE = - 100
	};

	class Position : // Board with turn
		public Board
	{
	public:
		enum Turn {
			TURN_BLACK = 1,
			TURN_WHITE = 2
		};

		Position();
		Position(int size);
		Position(const Position & position);
		virtual ~Position();

		void copy(const Position & position);
		virtual void clear();

		virtual void set_turn(Turn turn);
		virtual void set_handi(int handi); // handicap implies white turn
		virtual void flip_turn();

		Turn get_turn() const;

		// math
		static Turn flip_turn(Turn turn);
	private:
		Turn turn; // turn
	};

	class Delta // changes to a Position
	{
	public:
		Delta();
		Delta(const Delta & delta);

		void add_change(int loc, Board::State old_state, Board::State new_state);
		void del_change(int loc);
		void set_turn_flip(bool turn_flip);

		void apply(Position & position) const;
		void revert(Position & position) const;

		bool get_turn_flip() const;
		bool is_null() const;
		bool is_changed(int loc) const;
		Board::State get_new_state(int loc) const;

		int num_change() const; // change in (# of black) - (# of white)
	private:
		struct D {
			int loc;
			Board::State os; // old state
			Board::State ns; // new state
		};
		std::vector<D> dl; // change list
		bool tf; // turn flip?
	};

}
#endif // GO_GO0_HH
