/* ccgo: go/go.cc
 * 
 * Copyright (C) 2004 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "go0.hh"
#include <debug.hh>
#include <iostream>
#include <sstream>
using namespace ccgo;
using namespace std;

// members of class Error
Error::Error(string message)
{
	s = message;
}

const string & Error::get_message()
{
	return s;
}

// members of class Board
Board::Board() :
	z(0),
	b(0)
{
}

Board::Board(int size) :
	z(0),
	b(0)	
{
	assert(size >= 0); // if (size < 0) throw Error("negative board size");
	z = size;
	if (z) b = new State[z * z];
	for (int i = z * z - 1; i >= 0; i --) b[i] = STATE_EMPTY;
}

Board::Board(const Board & board) : // copy constructor
	z(0),
	b(0)
{
	z = board.z;
	if (z) {
		b = new State[z * z];
		for (int i = z * z - 1; i >= 0; i --) b[i] = board.b[i];
	}
}

Board::~Board()
{
	if (b) delete[] b;
}

void Board::copy(const Board & board)
{
	if (b) delete[] b;
	b = 0;
	z = board.z;
	if (z) {
		b = new State[z * z];
		for (int i = z * z - 1; i >= 0;i --) b[i] = board.b[i];
	}
}

void Board::clear()
{
	assert(b); // if (! b) throw Error("no board to clear");
	for (int i = z * z - 1; i >= 0; i --) b[i] = STATE_EMPTY;
}

void Board::set_state(int loc, State state)
{
	assert(loc >=0 && loc < z * z); // if (loc < 0 || loc >= z * z) throw Error("out or range loc");
	b[loc] = state;
}

void Board::set_handi(int handi)
{
	// handi < 0 => reset to no handicap
	// handi == 0 =>no handicap stone, white to start
	assert(handi > 1 || ! handi); //  if (handi && handi <= 1) throw Error("too few handicap stones");
	assert(handi <= 9); // if (handi > 9) throw Error("too many handicap stones");
	assert(z >= 5); // if (z < 5) throw Error("board too small for handicap");
	assert(handi <= 4 || (z % 2 && z >= 9)); // if (handi > 4 && (z % 2 == 0 || z < 9)) throw Error("too many handicap stones for the board");
	static const int hx[] = {- 1, 1, - 1, 1, - 1, 1, 0, 0};
	static const int hy[] = {- 1, 1, 1, - 1, 0, 0, - 1, 1};
	int n = handi;
	clear(); // clear board
	if (n > 4) {
		if (n % 2 == 1) { // odd number handicap,put one at center
			-- n;
			set_state(z / 2 + (z / 2) * z, Board::STATE_BLACK);
		}
	}
	// place the rest handi stones in order
	int d = z > 9 ? z - 7 : z - 5;
	for (int i = 0; i < n; i ++) {
		int x = (z - 1 + d * hx[i]) / 2;
		int y = (z - 1 + d * hy[i]) / 2;
		set_state(x + y * z, Board::STATE_BLACK);
	}
}

int Board::get_size() const
{
	return z;
}

Board::State Board::get_state(int loc) const
{
	assert(loc >= 0 && loc < z * z); // if (loc < 0 || loc >= z * z) throw Error("out or range loc");
	return b[loc];
}

vector<int> Board::neighbors(int loc) const
{
	assert(loc >= 0 && loc < z * z); // if (loc < 0 || loc >= z * z) throw Error("out or range loc");
	vector<int> n;
	if (loc % z > 0) n.push_back(loc - 1);
	if (loc % z < z - 1) n.push_back(loc + 1);
	if (loc / z > 0) n.push_back(loc - z);
	if (loc / z < z - 1) n.push_back(loc + z);
	return n;
}

int Board::loc_range() const
{
	return z * z;
}

int Board::flip_axis(int x) const
{
	assert(x >= 0 && x < z); // if (x <0 || x >= z) throw Error("out of range x");
	return z - x - 1;
}

int Board::xy_to_loc(int x, int y) const 
{
	if (x >= 0 && x < z && y >= 0 && y < z) return x + y * z;
	return NO_LOC;
}

void Board::loc_to_xy(int loc, int & x, int & y) const
{
	assert(loc >= 0 && loc < z * z); // if (loc < 0 || loc >= z * z) throw Error("out of range loc");
	x = loc % z;
	y = loc / z;
}

void Board::get_kill(int loc, vector<int> & kill, State put_state) const
{
	assert(loc >= 0 && loc < z * z); // if (loc < 0 || loc >= z * z) throw Error("out of range loc");
	int zz = z * z;
	bool m[zz]; // marking field
	State s = State(3 - put_state); // check other stones around
	vector<int> n = neighbors(loc); // checking the neighbors of l
	kill.clear();
	int bn = 0; // number of killed stones
	while (n.size()) {
		for (int i = 0; i < zz; i ++) m[i] = false; // clear marking field
		m[loc] = true; // fill the point!
		int q = qi_group(n.back(), kill, s, m);
		n.pop_back();
		for (vector<int>::iterator i = n.begin(); i != n.end(); i ++) if (m[* i]) {
			// remove checked neighbor from the list
			n.erase(i);
			i --;
		}
		if (q > 0) kill.resize(bn); // group is alive, remove from k
		else bn = kill.size();
	}
	if (bn > 0) return; // kill something

	// killed no one,  check for suicide
	for (int i = 0; i < zz; i ++) m[i] = false; // clear marking field
	m[loc] = true;
	kill.push_back(loc);
	int q = 0;
	n = neighbors(loc);
	s = put_state; // check same stones around
	while (n.size()) {
		q += qi_group(n.back(), kill, s, m);
		n.pop_back();
	}
	if (q) kill.clear(); // not suicide, clear k
}

int Board::qi_group(int loc, vector<int> & group, State state, bool * mark_field) const
{
	assert(loc >= 0 && loc < z * z); // if (loc < 0 || loc >= z * z) throw Error("out of range loc");
	if (mark_field[loc]) return 0;
	mark_field[loc] = true; // mark the loc
	if (get_state(loc) == STATE_EMPTY) return 1;
	if (get_state(loc) != state) return 0;
	group.push_back(loc); // add the loc
	int q = 0;
	vector<int> n = neighbors(loc);
	while (n.size()) {
		q += qi_group(n.back(), group, state, mark_field); // recursion
		n.pop_back();
	}
	return q;
}

void Board::score_terri(Terri * terri) const
// terri should be setup by caller, with killed stones marked as territory
{
	assert(terri);
	int r = loc_range();
	bool * marks = new bool[r];
	for (int i = 0; i < r; i ++) {
		// clear TERRI except for the stones
		if (terri[i] && ! b[i]) terri[i] = TERRI_NEUTRAL;
		// clear marking board
		marks[i] = false;
	}
	for (int i = 0; i < r; i ++) if (! marks[i] && (! b[i] || terri[i])) { // empty || killed nonmarked site
		std::vector<int> l; // l should be empty
		bool w_seen = false; // white encountered
		bool b_seen = false; // black encountered
		find_terri(i, l, w_seen, b_seen, terri, marks); // explore the empty space
		Terri t = TERRI_NEUTRAL;
		if (w_seen && ! b_seen) t = TERRI_WHITE; // see white only -> white territory
		else if (! w_seen && b_seen) t = TERRI_BLACK; // -> black_territory
		for (std::vector<int>::const_iterator i = l.begin(); i != l.end(); i ++) if (! b[* i]) terri[* i] = t; // set terri except for stones
	}
	// the above is an over estimate, some empty points linking to hanging groups need to be filled (fake eyes)
        bool rr; // any changes? (iteratively eliminate fake eyes...
        do {
                rr = false;
                // for blacks
		for (int i = 0; i < r; i ++) marks[i] = false; // clear the marking board
                for (int i = 0; i < r; i ++) if (! marks[i] && b[i] == STATE_BLACK && ! terri[i]) { // a live black stone
			std::vector<int> qi; // territory qi points encounterred
			hang_group(i, STATE_WHITE, qi, terri, marks);
			if (qi.size() == 1 && ! b[qi[0]]) { // one qi only... need to be connected
				// unless it's removed stone point, then we will leave it along
				terri[qi[0]] = TERRI_NEUTRAL;
				rr = true;
			}
			while (qi.size() > 0) { // unmark the territory qi points, as they can be shared by groups
				marks[qi.back()] = false;
				qi.pop_back();
			}
		}
		// do the same for white
		for (int i = 0; i < r; i ++) marks[i] = false; // clear the marking board
		for (int i = 0; i < r; i ++) if (! marks[i] && b[i] == STATE_WHITE && ! terri[i]) {
			std::vector<int> qi;
			hang_group(i, STATE_BLACK, qi, terri, marks);
			if (qi.size() == 1 && ! b[qi[0]]) {
				terri[qi[0]] = TERRI_NEUTRAL;
				rr = true;
			}
			while (qi.size() > 0) {
				marks[qi.back()] = false;
				qi.pop_back();
			}
		}
	} while (rr);
	delete marks;
}

void Board::find_terri(int loc, std::vector<int> & group, bool & w_seen, bool & b_seen, Terri * terri, bool * marks) const
{
	if (b[loc] && ! terri[loc]) {
		if (b[loc] == STATE_WHITE) w_seen = true;
		else b_seen = true;
		return;
	}
	marks[loc] = true;
	group.push_back(loc);
	std::vector<int> n = neighbors(loc);
	for (std::vector<int>::iterator i = n.begin(); i != n.end(); i ++) {
		if (marks[* i]) continue;
		find_terri(* i, group, w_seen, b_seen, terri, marks);
	}
}

void Board::hang_group(int loc, State state, std::vector<int> & qi, Terri * terri, bool * marks) const
{
	if (terri[loc]) {
		qi.push_back(loc);
		marks[loc] = true;
		return;
	}
	if (b[loc] == state) return;
	marks[loc] = true;
	std::vector<int> n = neighbors(loc);
	for (std::vector<int>::iterator i = n.begin(); i != n.end(); i ++) {
		if (marks[* i]) continue;
		hang_group(* i, state, qi, terri, marks);
	}
}

void Board::get_group(int loc, std::vector<int> & group) const
{
	if (! b[loc]) return;
	int r = loc_range();
	bool * marks = new bool[r];
	for (int i = 0; i < r; i ++) marks[i] = false;
	qi_group(loc, group, b[loc], marks);
	delete marks;
}

void Board::dead_group(int loc, std::vector<int> & group) const
{
	assert(b[loc]);
	int r = loc_range();
	bool * marks = new bool[r];
	for (int i = 0; i < r; i ++) marks[i] = false;
	terri_group(loc, b[loc], group, marks);
	delete marks;
}

void Board::terri_group(int loc, State state, std::vector<int> & group, bool * marks) const
{
	marks[loc] = true;
	if (b[loc] == 3 - state) return;
	if (b[loc] == state) group.push_back(loc);
	std::vector<int> n = neighbors(loc);
	for (std::vector<int>::iterator i = n.begin(); i != n.end(); i ++) if (! marks[* i]) terri_group(* i, state, group, marks);
}

int Board::compact_size() const
{
	return (z * z + 4) / 5;
}

void Board::compact_get(void * data) const
{
	unsigned char * d = (unsigned char *) data;
	d --;
	int zz = z * z;
	for (int i = 0; i < zz; i ++) {
		if (i % 5 == 0) {
			d ++;
			* d = 0;
		}
		* d *= 3;
		* d += b[i];
	}
}

bool Board::loc_is_star(int loc) const
{
	assert(loc >= 0 && loc < z * z); // if (loc < 0 || loc >= z * z) throw Error("out of range loc");
	int x = loc % z;
	int y = loc / z;
	if (z > 9) return (x == 3 || x == z - 4 || 2 * x == z - 1) && (y == 3 || y == z - 4 || 2 * y == z - 1);
	if (z == 9) return (x == 2 || x == z - 3 || 2 * x == z - 1) && (y == 2 || y == z - 3 || 2 * y == z - 1);
	// only 4 stars for board size 5~9
	if (z > 4) return (x == 2 || x == z - 3) && (y == 2 || y == z - 3);
	// no star for board of size 4 and below
	return false;
}

string Board::loc_to_str(int loc) const
{
	ostringstream s;
	char c = 'a' + loc % z;
	if (c >= 'i') c ++;
	s << c << 1 + loc / z;
	return s.str();
}

bool Board::is_valid() const
{
	return z && b;
}

// members of class Position
Position::Position() :
	turn(TURN_BLACK)
{
}

Position::Position(int size) :
	Board(size),
	turn(TURN_BLACK)
{
}

Position::Position(const Position & position) : // copy constructor
	Board(position),
	turn(position.turn)
{
	// k = position.k;
}

Position::~Position()
{
}

void Position::copy(const Position & position)
{
	Board::copy(position);
	turn = position.turn;
	// k = position.k;
}

void Position::clear()
{
	Board::clear();
	turn = TURN_BLACK;
	// k.clear();
}

void Position::set_turn(Turn t)
{
	turn = t;
}

void Position::set_handi(int handi)
{
	Board::set_handi(handi);
	turn = TURN_WHITE;
}

void Position::flip_turn()
{
	turn = Turn(3 - turn);
}

Position::Turn Position::get_turn() const
{
	return turn;
}

Position::Turn Position::flip_turn(Turn t)
{
	return Turn(3 - t);
}

// members of class Delta
Delta::Delta() :
	tf(false)
{
}

Delta::Delta(const Delta & delta)
{
	dl = delta.dl;
	tf = delta.tf;
}

void Delta::add_change(int loc, Board::State old_state, Board::State new_state)
{
	for (vector<D>::iterator i = dl.begin(); i != dl.end(); i ++) if (i->loc == loc) {
		msg(DBG_DEBUG) << "multiple add_change to the same site (" << i->loc << ")\n";
		assert(i->ns == old_state);
		if (i->os == new_state) { // back to old state, erase it!
			dl.erase(i);
		} else i->ns = new_state;
		return;
	}
	D d;
	d.loc = loc;
	d.os = old_state;
	d.ns = new_state;
	dl.push_back(d);
}

void Delta::del_change(int loc)
{
	for (vector<D>::iterator i = dl.begin(); i != dl.end(); i ++) if (i->loc == loc) {
		dl.erase(i);
		return;
	}
}

void Delta::set_turn_flip(bool turn_flip)
{
	tf = turn_flip;
}

void Delta::apply(Position & position) const
{
	for (vector<D>::const_iterator i = dl.begin(); i != dl.end(); i ++) {
		position.set_state(i->loc, i->ns);
	}
	if (tf) position.flip_turn();
}

void Delta::revert(Position & position) const
{
// 	if (ok.size() || nk.size()) position.set_ko(ok);
	if (tf) position.flip_turn();
	for (vector<D>::const_reverse_iterator i = dl.rbegin(); i != dl.rend(); i ++) {
		position.set_state(i->loc, i->os);
	}
}


bool Delta::get_turn_flip() const
{
	return tf;
}

bool Delta::is_null() const
{
	return ! (dl.size() || tf);
}

bool Delta::is_changed(int loc) const
{
	for (vector<D>::const_iterator i = dl.begin(); i != dl.end(); i ++) if (i->loc == loc) return true;
	return false;
}

Board::State Delta::get_new_state(int loc) const
{
	// assert(is_change(loc));
	for (vector<D>::const_iterator i = dl.begin(); i != dl.end(); i ++) if (i->loc == loc) return i->ns;
	throw Error("No change to loc");
}

int Delta::num_change() const
{
	int d = 0;
	for (vector<D>::const_iterator i = dl.begin(); i != dl.end(); i ++) {
		d -= ((i->os + 1) % 3 - 1);
		d += ((i->ns + 1) % 3 - 1);
	}
	return d;
}
