/* ccgo: go/game.hh
 *
 * Copyright (C) 2004 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#ifndef GO_GAME_HH
#define GO_GAME_HH

#include "prop.hh"
#include <vector>
#include <map>
#include <set>
#include <string>

namespace ccgo
{
	class Game;

	class Node
	{
	public:
		Node(Node * parent);
		virtual ~Node();

		// modifications
		void copy(const Node & node);
		void set_move(int move, const Delta & delta);
		void clear_move();
		void set_prop(int id, Prop * prop);
		int add_child(Node * node);
		void del_child(Node * node);

		// tree functions
		Node * up(); // the parent
		unsigned down_num(); // number of down links
		Node * down(unsigned index); // the children
		Game * get_game(); // game node (root of a tree)

		int get_move() const;
		const Delta * get_delta() const;
		Delta * get_delta();
		const Prop * get_prop(int id) const;
		unsigned get_index() const;
		unsigned get_level() const;

		// derivative properties 
		int get_num() const;
		const std::set<int> & get_ko() const;
		int get_capw() const;
		int get_capb() const;

		// maintenance functions for derivative properties
		void set_num(int num);
		void set_ko(const std::set<int> & ko);
		void add_capw(int capw);
		void add_capb(int capb);

	private:
		Node * game; // Game
		Node * ul; // up link
		std::vector<Node *> dl; // down links
		unsigned level;

		// accumulative properties, effects should be added up
		//  the following describe a move and the change to the board
		int mv; // Move
		Delta * ch; // Changes by the move

		// derivative properties, can be obtained from path of the game
		int mn; // Move number
		std::set<int> ko; // ko (forbidden empty sites) points on the board
		int capw; // number of stones captured by white
		int capb; // number of stones captured by black

		// substitutive properties, independent of other nodes
		std::map<int, Prop *> pp;

		bool lock;
	};

	class WalkBase // tree traverser
	{
	public:
		WalkBase();
		virtual ~WalkBase();
		virtual void set_node(Node * node); // attach the walk to a node
		Node * where() const;
		virtual void up(); // move to the parent Node (but not necessary end up at the parent)
		virtual void down(unsigned index);
		virtual void down(Node * node);

		// be notified of changes to the tree
		virtual void do_stem(); // notified growth
		virtual void do_geom(); // notified geometry change

		// get on and off the current node
		virtual void node_off();
		virtual void node_on();
	private:
		Node * node;
	};

	class Path : // keeper of a route to a leaf
		public WalkBase
	{
	public:
		void set_node(Node * node);
		void up();
		void down(unsigned index); // shouldn't be used at all...
		void down(Node * node);
		void do_stem();

		Node * get_path(unsigned level);
	private:
		std::vector<Node *> path;
	};

	class Walk : // traverser with a Position, game state
		public WalkBase,
		virtual public Position
	{
	public:
		Walk();
		virtual ~Walk();


		virtual void set_node(Node * node); // attach the walk to a node
		virtual void up(); // move to the parent Node (but not necessary end up at the parent)
		virtual void down(unsigned index);
		virtual void down(Node * node);

		virtual void node_off();
		virtual void node_on();

		bool game_can_init(); // check if initialization of the game can be changed
		int game_add_move(int move); // add a move node and return the index
		int game_add_setup(); // add a non-move node
		int game_add_setup(const Position & position); // add setup
		void game_set_handi(int handi); // set handicap of the game
		void game_add_handi(int loc); // add a handicap stone
		bool game_del_node(); // delete a node

		bool node_can_setup(); // check if setup of the node can be changed
		void node_set_turn(Turn turn);
		void node_set_site(int loc, State state);

		int get_b_cap();
		int get_w_cap();

		Node * make_move(int move); // make a move node
		Node * make_extra_move(int move); // make a move node of previous turn
		Node * make_setup(const Position & position);
		Node * make_setup(const Delta & delta);
	protected:
		virtual void init_game(Game * game);
		virtual void apply_node(Node * node);
		virtual void revert_node(Node * node);
	private:
		int cap_b;
		int cap_w;

		bool node_state;
	};

	class PathWalk : // a game state that has a destinated leaf
		public Walk
	{
	public:
		PathWalk();
		~PathWalk();
		virtual void set_node(Node * node);
		virtual void down(unsigned index);
		virtual void down(Node * node);
		virtual void down();
		unsigned path_depth();
		Node * get_path(unsigned level);
	private:
		Path path;
	};

	class Match :
		virtual public Walk
	{
	public:
		~Match();
	};

	struct Rule
	{
		Rule();
		bool opt_overwrite_stone;
		bool opt_ignore_ko;
		bool opt_allow_suicide;
		enum Ko {
			KO_SIMPLE = 0,
			KO_NONE = 1,
			KO_POSITIONAL = 2,
			KO_SITUATIONAL = 3
		};
		Ko opt_ko_style;
	};

	class Game :
		public Node
	{
	public:
		Game();
		~Game();

		void copy(const Game & game);
		
		void set_init(const Position & position);
		const Position * get_init() const;
		Position * get_init();
		bool check(const Walk & walk, int move);
		Delta make_delta(const Walk & walk, int move);
		const Rule & get_rule() const;
		void set_rule(const Rule & rule);
		void add_walk(WalkBase * walk);
		void del_walk(WalkBase * walk);

		void set_name(std::string name);
		const std::string & get_name() const;
		void set_komi(double komi);
		double get_komi() const;

		void node_clear(Node * node);
		void node_stem(Node * node); // notify peers a leaf becomes stem
		void node_lock(WalkBase * walk);
		void node_free(WalkBase * walk);
		void game_geom(); // notify all walk of geometry change

	private:
		Position * p; // initial position
		Rule r;
		bool h; // handicap
		friend class WalkBase;
		std::vector<WalkBase *> walk_list;

		// Game parameters
		std::string nm;
		double km;
	};

}
#endif // GO_GAME_HH
