/* ccgo: go/go.cc
 * 
 * Copyright (C) 2004 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "game0.hh"
#include <debug.hh>
#include <iostream>
#include <sstream>
using namespace ccgo;
using namespace std;

// members of class Node
Node::Node(Node * parent) :
	mv(MOVE_NONE),
	ch(0),
	lock(false)
{
	ul = parent;
	if (ul) {
		game = ul->game; // copy game node pointer
		level = ul->level + 1; // one level deeper
		mn = ul->mn; // same move number
		ko = ul->ko; // same ko (no ko change if same move number)
		capw = ul->capw;
		capb = ul->capb;
	} else {
		game = this;
		level = 0;
		mn = 0;
		// ko.clear();
		capw = 0;
		capb = 0;
	}
}

Node::~Node()
{
	while (dl.size()) {
		delete dl.back();
		dl.pop_back();
	}
	if (ch) delete ch;
	for (map<int, Prop *>::iterator i = pp.begin(); i != pp.end(); i ++) delete i->second;
}

void Node::copy(const Node & node) // will ever need to do this?
{
	while (dl.size()) {
		delete dl.back();
		dl.pop_back();
	}
	for (vector<Node *>::const_iterator i = node.dl.begin(); i != node.dl.end(); i ++) {
		dl.push_back(new Node(this));
		dl.back()->copy(* (* i));
	}
	mv = node.mv;
	if (ch) delete ch;
	ch = new Delta(* node.ch);
	mn = node.mn;
	for (map<int, Prop *>::iterator i = pp.begin(); i != pp.end(); i ++) delete i->second;
	pp.clear();
	for (map<int, Prop *>::const_iterator i = node.pp.begin(); i != node.pp.end(); i ++) pp[i->first] = i->second->dup();
}

void Node::set_move(int move, const Delta & delta)
{
	if (mv != MOVE_NONE) mn --;
	mv = move;
	if (mv != MOVE_NONE) mn ++;
	if (ch) delete ch;
	ch = new Delta(delta);
	ko.clear();
}

void Node::clear_move()
{
	mv = MOVE_NONE;
	if (ch) delete ch;
	ch = 0;
}

void Node::set_prop(int id, Prop * prop)
{
	map<int, Prop *>::iterator i = pp.find(id);
	if (i != pp.end()) {
		delete i->second;
		pp.erase(i);
	}
	if (prop) pp[prop->get_id()] = prop;
}

int Node::add_child(Node * node)
{
	dl.push_back(node);
	return dl.size() - 1;
}

void Node::del_child(Node * node)
{
	for (vector<Node *>::iterator i = dl.begin(); i != dl.end(); i ++) if (* i == node) {
		delete * i;
		dl.erase(i);
		break;
	}
}

Node * Node::up()
{
	return ul;
}

unsigned Node::down_num()
{
	return dl.size();
}

Node * Node::down(unsigned index)
{
	return dl[index];
}

Game * Node::get_game()
{
	return dynamic_cast<Game *>(game);
}


int Node::get_move() const
{
	return mv;
}

const Delta * Node::get_delta() const
{
	return ch;
}

Delta * Node::get_delta()
{
	return ch;
}


const Prop * Node::get_prop(int id) const
{
	map<int, Prop *>::const_iterator i = pp.find(id);
	if (i != pp.end()) return i->second;
	return 0;
}

unsigned Node::get_index() const
{
	assert(ul);
	for(unsigned i = 0; i < ul->dl.size(); i ++) if (ul->dl[i] == this) return i;
	throw Error("can find index for node!");
}

unsigned Node::get_level() const
{
	return level;
}

int Node::get_num() const
{
	return mn;
}

const std::set<int> & Node::get_ko() const
{
	return ko;
}

int Node::get_capw() const
{
	return capw;
}

int Node::get_capb() const
{
	return capb;
}

void Node::set_num(int num)
{
	mn = num;
}

void Node::set_ko(const std::set<int> & k)
{
	ko = k;
}

void Node::add_capw(int cap)
{
	capw += cap;
}

void Node::add_capb(int cap)
{
	capb += cap;
}

// members of class WalkBase
WalkBase::WalkBase() :
	node(0)
{
}

WalkBase::~WalkBase()
{
	if (node) node->get_game()->del_walk(this);
}

void WalkBase::set_node(Node * nd)
{
	if (! nd) {
		if (node) {
			node->get_game()->del_walk(this);
			node = 0;
		}
		return;
	}

	if (node && node->get_game() == nd->get_game()) {
		node = nd;
	} else {
		if (node) node->get_game()->del_walk(this);
		nd->get_game()->add_walk(this);
		node = nd;
	}
}

Node * WalkBase::where() const
{
	return node;
}

void WalkBase::up()
{
	assert(node->up());
	node = node->up();
}

void WalkBase::down(unsigned index)
{
	assert(index < node->down_num());
	node = node->down(index);
}

void WalkBase::down(Node * n)
{
	assert(n->up() == where());
	node = n;
}

void WalkBase::do_stem()
{
}

void WalkBase::do_geom()
{
}

void WalkBase::node_off()
{
}

void WalkBase::node_on()
{
}

// members of class Path
void Path::set_node(Node * n)
{
	msg(DBG_DEBUG) << "Path::set_node(" << n << ")\n";
	if (! n) {
		WalkBase::set_node(0);
		path.clear();
		return;
	}

	unsigned l = n->get_level();
	if (where() && path.size() >= l && n == (l ? path[l - 1] : (Node *) where()->get_game())) return;

	WalkBase::set_node(n);
	Node * m = where();
	assert(m == n);
	path.resize(l, 0);

	while (l && path[l - 1] != m) {
		l --;
		path[l] = m;
		m = m->up();
	}

	while (where()->down_num()) {
		WalkBase::down(0u);
		path.push_back(where());
	}
}

void Path::up()
{
	Node * n = where();
	WalkBase::up();
	path.pop_back();
	for (unsigned i = 0; i < where()->down_num(); i ++) if (where()->down(i) != n) {
		// decend to leaf
		WalkBase::down(i);
		path.push_back(where());
		while (where()->down_num()) {
			WalkBase::down(0u);
			path.push_back(where());
		}
		break;
	}
}

void Path::down(unsigned index)
{
	assert(false);
}

void Path::down(Node * node)
{
	assert(false);
}

void Path::do_stem()
{
	while(where()->down_num()) {
		WalkBase::down(0u);
		path.push_back(where());
	}
}

Node * Path::get_path(unsigned level)
{
	assert(level < path.size());
	return path[level];
}

// members of class Walk
Walk::Walk() :
	cap_b(0),
	cap_w(0)
{
}

Walk::~Walk()
{
	msg(DBG_DEBUG) << "deleting walk " << this << "\n";
}

void Walk::set_node(Node * n)
{
	msg(DBG_DEBUG) << "Walk::set_node(" << n << ")\n";
	if (! n) {
		WalkBase::set_node(0);
		init_game(0);
		return;
	}

	if (where() && where()->get_game() == n->get_game()) { // same tree, find common ancestor
		int i = where()->get_level() - n->get_level();
		Node * m = n;
		vector<Node *> s;
		while (i > 0) {
			i --;
			revert_node(where());
			WalkBase::up();
		}
		while (i < 0) {
			i ++;
			s.push_back(m);
			m = m->up();
		}
		while (m != where()) {
			revert_node(where());
			WalkBase::up();
			s.push_back(m);
			m = m->up();
			assert(m);
		}
		while (s.size()) {
			WalkBase::down(s.back());
			apply_node(s.back());
			s.pop_back();
		}
	} else { // different tree, jump to it
		WalkBase::set_node(n);
		Node * m = n;
		vector<Node *> s;
		init_game(where()->get_game());
		while (m->up()) {
			s.push_back(m);
			m = m->up();
		}
		while (s.size()) {
			apply_node(s.back());
			s.pop_back();
		}
	}
	node_state = true;
}

void Walk::up()
{
	if (node_state) revert_node(where());
	else node_state = true;
	WalkBase::up();
}

void Walk::down(unsigned index)
{
	assert(node_state);
	WalkBase::down(index);
	apply_node(where());
}

void Walk::down(Node * n)
{
	assert(node_state);
	WalkBase::down(n);
	apply_node(where());
}

void Walk::node_off()
{
	assert(node_state);
	revert_node(where());
	node_state = false;
}

void Walk::node_on()
{
	assert(! node_state);
	apply_node(where());
	node_state = true;
}

bool Walk::game_can_init()
{
	Node * n = where();
	if (n != (Node *) n->get_game()) return false;
	if (n->down_num()) return false;
	return true;
}

int Walk::game_add_move(int move) // return child index
{
	Node * nn = make_move(move);
	if (! nn) return - 1;
	Node * n = where();
	int i = n->add_child(nn);
	if (! i) n->get_game()->node_stem(n);
	n->get_game()->game_geom();
	return i;
}

int Walk::game_add_setup()
{
	Node * n = where();
	int i = n->add_child(new Node(n));
	if (! i) n->get_game()->node_stem(n);
	n->get_game()->game_geom();
	return i;
}

int Walk::game_add_setup(const Position & position)
{
	return 0;
}

void Walk::game_set_handi(int handi)
{
	if (! game_can_init()) throw Error("can not change handicap");
	Game * g = where()->get_game();
	g->node_lock(this);
	if (handi < 0) {
		g->get_init()->clear(); // clear handicap
		clear();
	} else {
		g->get_init()->set_handi(handi);
		set_handi(handi);
	}
	g->node_free(this);
}

void Walk::game_add_handi(int loc)
{
	if (! game_can_init()) throw Error("can not add handicap");
	Game * g = where()->get_game();
	g->node_lock(this);
	g->get_init()->set_state(loc, STATE_BLACK);
	set_state(loc, STATE_BLACK);
	g->node_free(this);
}

bool Walk::game_del_node()
{
	Node * n = where();
	if (n->down_num() || ! n->get_level()) return false; // not a leaf node
	up();
	n->get_game()->node_clear(n);
	where()->del_child(n);
	// n->get_game()->game_geom();
	return true;
}

bool Walk::node_can_setup()
{
	Node * n = where();
	return ! n->down_num() && n->get_move() == MOVE_NONE;
}

void Walk::node_set_turn(Turn turn)
{
	assert(node_can_setup());
	if (get_turn() == turn) return;
	Node * n = where();
	Game * g = n->get_game();
	Delta * d = n->get_delta();
	g->node_lock(this);
	if (! d) { // no Delta before
		if (! n->get_level()) { // Game node, set turn to init
			g->get_init()->set_turn(turn);
			flip_turn();
			g->node_free(this);
			return;
		}
		n->set_move(MOVE_NONE, Delta()); // add a Delta
		d = n->get_delta();

		// don't change ko
// 		d->set_ko_old(get_ko());
// 		d->set_ko_new(get_ko());
	}
	d->set_turn_flip(! d->get_turn_flip());
	flip_turn();
	if (d->is_null()) n->clear_move();
	g->node_free(this);
}

void Walk::node_set_site(int loc, State state)
{
	assert(node_can_setup());
	if (get_state(loc) == state) return;
	Node * n = where();
	Game * g = n->get_game();
	Delta * d = n->get_delta();
	g->node_lock(this);
	if (! d) {
		if (! n->get_level()) { // Game node, set site to init
			g->get_init()->set_state(loc, state);
			set_state(loc, state);
			g->node_free(this);
			return;
		}
		n->set_move(MOVE_NONE, Delta());
		d = n->get_delta();
	}
	d->add_change(loc, get_state(loc), state);
	set_state(loc, state);
	if (d->is_null()) n->clear_move();
	g->node_free(this);
}

int Walk::get_b_cap()
{
	return cap_b;
}

int Walk::get_w_cap()
{
	return cap_w;
}

Node * Walk::make_move(int move)
{
	if (! where()->get_game()->check(* this, move)) {
		msg(DBG_ERROR) << "Can not make illegal move.";
		return 0; // should throw an error
	}

	// make Delta
	Delta d;
	vector<int> k;
	bool suicide = false;
	if (move >= 0) { // put a stone
		d.add_change(move, get_state(move), State(get_turn()));
		get_kill(move, k, State(get_turn()));
		if (k.size()) {
			vector<int>::iterator i = k.begin();
			if (* i == move) { // suicide, change it back
				d.add_change(move, State(get_turn()), STATE_EMPTY);
				i ++;
				suicide = true;
			}
			while (i != k.end()) {
				d.add_change(* i, get_state(* i), STATE_EMPTY);
				i ++;
			}
		}
	}
	d.set_turn_flip(true);

	Node * n = new Node(where());
	n->set_move(move, d);

	// calculate captures
	switch (get_turn()) {
	case TURN_BLACK: // move is by black
		if (suicide) n->add_capw(k.size());
		else n->add_capb(k.size());
		break;
	case TURN_WHITE:
		if (suicide) n->add_capb(k.size());
		else n->add_capb(k.size());
		break;
	}

	// calculate ko
	assert(! n->get_ko().size());
	if (k.size() == 1 && ! suicide) {
		msg(DBG_DEBUG) << "checking simple ko\n";
		vector<int> p = neighbors(move);
		bool surrounded = true;
		State enemy = State(flip_turn(get_turn()));
		for (vector<int>::iterator i = p.begin(); i != p.end(); i ++) {
			if (get_state(* i) != enemy) {
				surrounded = false; // same stone around
				break;
			}
		}
		if (surrounded) { // site surrouneded -> simple ko!
			set<int> ko;
			ko.insert(k[0]);
			n->set_ko(ko);
			msg(DBG_DEBUG) << "ko at "  << k[0] << '\n';
		}
	}

	return n;
}

Node * Walk::make_extra_move(int move)
{
	flip_turn();
	Node * n = make_move(move);
	if (n) n->get_delta()->set_turn_flip(false);
	flip_turn();
	return n;
}

Node * Walk::make_setup(const Position & position)
{
	return 0;
}

Node * Walk::make_setup(const Delta & delta)
{
	Node * n = new Node(where());
	n->set_move(MOVE_NONE, delta);
	return n;
}

void Walk::init_game(Game * game)
{
	if (! (game && game->get_init())) { // no board
		copy(Position());
		return;
	};
	copy(* game->get_init());
	cap_b = 0;
	cap_w = 0;
}

void Walk::apply_node(Node * node)
{
	Delta * d = node->get_delta();
	if (d) {
		d->apply(* this);
		if (node->get_move() >= 0) { // regular move, calculate captures
			int c = d->num_change();
			switch (get_turn()) {
			case TURN_WHITE:
				if (c > 1) cap_b += (c - 1);
				else cap_w += (1 - c);
				break;
			case TURN_BLACK:
				if (c < - 1) cap_w -= (c + 1);
				else cap_b += (c + 1);
				break;
			}
		}
	}
}

void Walk::revert_node(Node * node)
{
	Delta * d = node->get_delta();
	if (d) {
		if (node->get_move() >= 0) { // regular move, calculate captures
			int c = d->num_change();
			switch (get_turn()) {
			case TURN_WHITE:
				if (c > 1) cap_b -= (c - 1);
				else cap_w -= (1 - c);
				break;
			case TURN_BLACK:
				if (c < - 1) cap_w += (c + 1);
				else cap_b -= (c + 1);
				break;
			}
		}
		d->revert(* this);
	}
}

// members of PathWalk

PathWalk::PathWalk()
{
}

PathWalk::~PathWalk()
{
}

void PathWalk::set_node(Node * node)
{
	Walk::set_node(node);
	path.set_node(node);
}

void PathWalk::down(unsigned index)
{
	Walk::down(index);
	path.set_node(where());
}

void PathWalk::down(Node * n)
{
	Walk::down(n);
	path.set_node(where());
}

void PathWalk::down()
{
	Walk::down(path.get_path(where()->get_level()));
}

unsigned PathWalk::path_depth()
{
	msg(DBG_DEBUG) << "path where() = " << path.where() << '\n';
	return path.where()->get_level();
}

Node * PathWalk::get_path(unsigned level)
{
	return path.get_path(level);
}

// members of class Match
Match::~Match()
{
}

// members of struct Rule
Rule::Rule()
{
	opt_overwrite_stone = false;
	opt_ignore_ko = false;
	opt_allow_suicide = false;
	opt_ko_style = KO_SIMPLE;
}

// members of class Game
Game::Game() :
	Node(0),
	p(0),
	h(- 1),
	km(0)
{
}

Game::~Game()
{
	vector<WalkBase *> ww = walk_list;
	msg(DBG_INFO) << "deleting game with " << ww.size() << " walks on!\n";
	for (vector<WalkBase *>::iterator i = ww.begin(); i != ww.end(); i ++) {
		(* i)->set_node(0);
	}
	assert(walk_list.size() == 0);
	if (p) delete p;
}

void Game::copy(const Game & game)
{
	Node::copy(game);
}

void Game::set_init(const Position & position)
{
	p = new Position(position);
}

Position * Game::get_init()
{
	return p;
}

bool Game::check(const Walk & walk, int move)
{
	if (move == MOVE_PASS) return true; // pass is always allowed
	if (move == MOVE_NONE) return false; // must be a move
	if (! r.opt_overwrite_stone && walk.get_state(move) != Board::STATE_EMPTY) return false;
	if (! r.opt_ignore_ko) {
		if (walk.where()->get_ko().count(move)) return false;
	}
	if (! r.opt_allow_suicide) {
		vector<int> k;
		walk.get_kill(move, k, Board::State(walk.get_turn()));
		if (k.size() && k[0] == move) return false;
	}
	return true;
}

Delta Game::make_delta(const Walk & walk, int move)
{
	Delta a;
	vector<int> k;
	if (move >= 0) { // put a stone
		a.add_change(move, walk.get_state(move), Board::State(walk.get_turn()));
		walk.get_kill(move, k, Board::State(walk.get_turn()));
		if (k.size()) {
			vector<int>::iterator i = k.begin();
			if (* i == move) { // suicide, change it back
				a.add_change(move, Board::State(walk.get_turn()), Board::STATE_EMPTY);
				i ++;
			}
			while (i != k.end()) {
				a.add_change(* i, walk.get_state(* i), Board::STATE_EMPTY);
				i ++;
			}
		}
	}
	// turn flips
	a.set_turn_flip(true);
	return a;
}

const Rule & Game::get_rule() const
{
	return r;
}

void Game::set_rule(const Rule & rule)
{
	r = rule;
}

void Game::add_walk(WalkBase * walk)
{
	walk_list.push_back(walk);
}

void Game::del_walk(WalkBase * walk)
{
	msg(DBG_DEBUG) << "removing walk : " << walk << " from game list\n";
	for (vector<WalkBase *>::iterator i = walk_list.begin(); i != walk_list.end(); i ++) if (walk == * i) {
		walk_list.erase(i);
		return;
	}
	throw Error("can not find Walk in Game");
}

void Game::set_name(std::string name)
{
	nm = name;
}

const std::string & Game::get_name() const
{
	return nm;
}

void Game::set_komi(double komi)
{
	km = komi;
}

double Game::get_komi() const
{
	return km;
}

void Game::node_clear(Node * node)
{
	msg(DBG_ERROR) << "clearing node " << node << '\n';
	for (std::vector<WalkBase *>::iterator i = walk_list.begin(); i != walk_list.end(); i ++)
		if (node == (* i)->where()) (* i)->up();
}

void Game::node_stem(Node * node)
{
	for (std::vector<WalkBase *>::iterator i = walk_list.begin(); i != walk_list.end(); i ++)
		if (node == (* i)->where()) (* i)->do_stem();
}

void Game::game_geom()
{
	for (std::vector<WalkBase *>::iterator i = walk_list.begin(); i != walk_list.end(); i ++) (* i)->do_geom();
}

void Game::node_lock(WalkBase * walk)
{
	Node * node = walk->where();
	for (std::vector<WalkBase *>::iterator i = walk_list.begin(); i != walk_list.end(); i ++)
		if (node == (* i)->where() && walk != * i) (* i)->node_off();
}

void Game::node_free(WalkBase * walk)
{
	Node * node = walk->where();
	for (std::vector<WalkBase *>::iterator i = walk_list.begin(); i != walk_list.end(); i ++)
		if (node == (* i)->where() && walk != * i) (* i)->node_on();
}
