/* ccgo: go/board.hh
 *
 * Copyright (C) 2006 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
#ifndef GO_BOARD_HH
#define GO_BOARD_HH
#include <go/delta.hh>
#include <vector>
#include <map>
#include <set>
#include <string>
namespace go
{
	class Board // square board, an array of states
	{
	private:
		unsigned z; // board size
		State * b;

		void get_kill(Loc loc, Turn turn, std::vector<Loc> & kill) const;
		unsigned qi_group(Loc loc, std::vector<Loc> & group, State state, bool * marking_field) const;
	public:
		Board(unsigned sz = 0);
		Board(const Board & board);
		virtual ~Board();

		void resize(unsigned size); // change the size of the board
		void copy(const Board & board);
		void operator = (const Board & board);
		void clear(); // clear board
		void operator += (const Delta & delta);
		void operator -= (const Delta & delta);
		void set_state(Loc loc, State state); // set state at a loc

		// for children to override
		virtual void new_size();
		virtual void new_state(Loc loc);

		// constant functions
		unsigned get_size() const;
		Loc range() const; // number of sites on the board
		State state(Loc loc) const; // get state at a loc
		std::set<Loc> neighbors(Loc loc) const; // return neighboring sites of "loc"
		bool loc_is_star(Loc loc) const;

		Delta make_delta(Move move, Turn turn) const; // make Delta from the board with given move and turn
		Delta operator - (const Board & board) const;
		bool operator == (const Board & board) const;

		std::vector<Loc> handi_sites(unsigned handi) const;

		// scoring functions
		void score_terri(State * terri) const; // terri should be setup by caller, with killed stones marked as territory
		void get_group(Loc loc, std::vector<Loc> & group) const; // get connected group
		void dead_group(Loc loc, std::vector<Loc> & group) const; // get dead group in the same territory
	private:
		void find_terri(Loc loc, std::vector<Loc> & group, bool & w_seen, bool & b_seen, State * terri, bool * marks) const;
		void hang_group(Loc loc, State state, std::vector<Loc> & qi, State * terri, bool * marks) const;
		void terri_group(Loc loc, State state, std::vector<Loc> & group, bool * marks) const;
	};
}
#endif // GO_BOARD_HH
