/* ccgo: go/board.cc
 * 
 * Copyright (C) 2006 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
#include <go/board.hh>
#include <debug.hh>
#include <iostream>
#include <sstream>
#include <set>
using namespace go;
using namespace std;

void Board::get_kill(Loc loc, Turn turn, vector<Loc> & kill) const
{
	assert(loc < range());
	bool * m = new bool [range()]; // marking field
	State s = State(3 - turn); // state of other stones
	set<Loc> n = neighbors(loc); // checking the neighbors of l
	kill.clear();
	unsigned bn = 0; // number of killed stones
	while (n.size()) {
		for (Loc i = 0; i < range(); i ++) m[i] = false; // clear marking field
		m[loc] = true; // mark the site
		unsigned q = qi_group(* n.begin(), kill, s, m);
		n.erase(n.begin());
		// remove already-checked neighbors from the list
		set<Loc>::iterator ii = n.begin();
		while (ii != n.end()) {
			set<Loc>::iterator next = ii;
			next ++;
			if (m[* ii]) n.erase(ii);
			ii = next;
		}
		if (q > 0) kill.resize(bn); // group is alive, remove from kill
		else bn = kill.size(); // increase the number of killed
	}
	if (bn > 0) { // killed something, the stone itself must be alive
		delete [] m;
		return;
	}

	// killed no one, check for suicide
	for (Loc i = 0; i < range(); i ++) m[i] = false; // clear marking field
	m[loc] = true;
	kill.push_back(loc);
	unsigned q = 0;
	n = neighbors(loc);
	s = turn; // check same stones around
	while (n.size()) {
		q += qi_group(* n.begin(), kill, s, m);
		n.erase(n.begin());
	}
	if (q) kill.clear(); // not suicide, clear k
	delete [] m;
}

unsigned Board::qi_group(Loc loc, vector<Loc> & group, State st, bool * mark_field) const
{
	assert(loc < range());
	if (mark_field[loc]) return 0;
	mark_field[loc] = true; // mark the loc
	if (state(loc) == Empty) return 1; // adjecent empty site, qi += 1
	if (state(loc) != st) return 0; // blocked site
	group.push_back(loc); // connected group, add the loc
	unsigned q = 0;
	set<Loc> n = neighbors(loc);
	while (n.size()) {
		q += qi_group(* n.begin(), group, st, mark_field); // recursion
		n.erase(n.begin());
	}
	return q;
}

Board::Board(unsigned sz) :
	z(sz)
{
	if (z) b = new State[z * z];
	for (Loc i = 0; i < range(); i ++) b[i] = Empty;
}

Board::Board(const Board & board) :
	z(board.z)
{
	if (z) b = new State[z * z];
	for (Loc i = 0; i < range(); i ++) b[i] = Empty;
	for (Loc i = 0; i < range(); i ++) if (board.state(i) != Empty) set_state(i, board.state(i));
}

Board::~Board()
{
	if (z) delete[] b;
}

void Board::resize(unsigned size)
{
	if (z) delete [] b;
	z = size;
	if (z) b = new State[z * z];
	for (Loc i = 0; i < range(); i ++) b[i] = Empty;
	new_size();
}

void Board::copy(const Board & board)
{
	if (z != board.z) resize(board.z);
	assert(range() == board.range());
	for (Loc i = 0; i < range(); i ++) if (state(i) != board.state(i)) set_state(i, board.state(i));
}

void Board::operator = (const Board & board)
{
	copy(board);
}

void Board::clear()
{
	for (Loc i = 0; i < range(); i ++) if (state(i) != Empty) set_state(i, Empty);
}

void Board::operator += (const Delta & delta)
{
	for (map<Loc, unsigned char>::const_iterator i = delta.dlist.begin(); i != delta.dlist.end(); i ++) {
		set_state(i->first, (state(i->first) + i->second) % 3);
	}
}

void Board::operator -= (const Delta & delta)
{
	for (map<Loc, unsigned char>::const_iterator i = delta.dlist.begin(); i != delta.dlist.end(); i ++) {
		set_state(i->first, (state(i->first) + 3 - i->second) % 3);
	}
}

void Board::set_state(Loc loc, State state)
{
	assert(loc < range());
	b[loc] = state;
	new_state(loc);
}

void Board::new_size()
{
}

void Board::new_state(Loc loc)
{
}

unsigned Board::get_size() const
{
	return z;
}

Loc Board::range() const
{
	return z * z;
}

State Board::state(Loc loc) const
{
	assert(loc < range());
	return b[loc];
}

set<Loc> Board::neighbors(Loc loc) const
{
	assert(loc < range());
	set<Loc> nb;
	if (loc % z > 0) nb.insert(loc - 1);
	if (loc % z < z - 1) nb.insert(loc + 1);
	if (loc / z > 0) nb.insert(loc - z);
	if (loc / z < z - 1) nb.insert(loc + z);
	return nb;
}

bool Board::loc_is_star(Loc loc) const
{
	assert(loc >= 0 && loc < z * z); // if (loc < 0 || loc >= z * z) throw Error("out of range loc");
	size_t x = loc % z;
	size_t y = loc / z;
	if (z > 9) return (x == 3 || x == z - 4 || 2 * x == z - 1) && (y == 3 || y == z - 4 || 2 * y == z - 1);
	if (z == 9) return (x == 2 || x == z - 3 || 2 * x == z - 1) && (y == 2 || y == z - 3 || 2 * y == z - 1);
	// only 4 stars for board size 5~9
	if (z > 4) return (x == 2 || x == z - 3) && (y == 2 || y == z - 3);
	// no star for board of size 4 and below
	return false;
}

Delta Board::make_delta(Move move, Turn turn) const
{
	Delta d;
	if (move < 0) return d;
	vector<Loc> k;
	get_kill(Loc(move), turn, k);
	if (k.size()) {
		vector<Loc>::iterator i = k.begin();
		if (* i != Loc(move)) {
			unsigned char dd = (turn + 3 - state(Loc(move))) % 3;
			if (dd) d.dlist[Loc(move)] = dd;
		}
		while (i != k.end()) {
			unsigned char dd = (3 - state(* i)) % 3;
			if (dd) d.dlist[* i] = dd;
			i ++;
		}
	}
	else {
		unsigned char dd = (turn + 3 - state(Loc(move))) % 3;
		if (dd) d.dlist[Loc(move)] = dd;
	}
	return d;
}

Delta Board::operator - (const Board & board) const
{
	Delta d;
	assert(range() == board.range());
	for (Loc i = 0; i < range(); i ++) if (b[i] != board.b[i]) {
		d.dlist[i] = (b[i] + 3 - board.b[i]) % 3;
	}
	return d;
}

bool Board::operator == (const Board & board) const
{
	for (Loc i = 0; i < range(); i ++) if (b[i] != board.b[i]) return false;
	return true;
}

vector<Loc> Board::handi_sites(unsigned handi) const
{
	vector<Loc> hs;
	assert(handi > 1 || ! handi); // too few handicap stones?
	assert(handi <= 9); // too many handicap stones?
	assert(z >= 5); // board too small for handicap?
	assert(handi <= 4 || (z % 2 && z >= 9)); // too many handicap stones for the board?
	static const int hx[] = {- 1, 1, - 1, 1, - 1, 1, 0, 0};
	static const int hy[] = {- 1, 1, 1, - 1, 0, 0, - 1, 1};
	unsigned n = handi;
	if (n > 4) {
		if (n % 2) { // odd number handicap,put one at center
			-- n;
			hs.push_back(Loc(z / 2 + (z / 2) * z));
		}
	}
	int d = z > 9 ? z - 7 : z - 5;
	for (unsigned i = 0; i < n; i ++) {
		unsigned x = unsigned(z - 1 + d * hx[i]) / 2;
		unsigned y = unsigned(z - 1 + d * hy[i]) / 2;
		hs.push_back(Loc(x + y * z));
	}
	return hs;
}

void Board::score_terri(State * terri) const
{
	assert(terri);
	unsigned r = range();
	bool * marks = new bool[r];
	for (Loc i = 0; i < r; i ++) {
		// clear TERRI except for the stones
		if (! b[i]) terri[i] = Empty;
		// clear marking board
		marks[i] = false;
	}
	for (Loc i = 0; i < r; i ++) if (! marks[i] && (! b[i] || terri[i])) { // empty || killed nonmarked site
		vector<Loc> l; // l should be empty
		bool w_seen = false; // white encountered
		bool b_seen = false; // black encountered
		find_terri(i, l, w_seen, b_seen, terri, marks); // explore the empty space
		State t = Empty;
		if (w_seen && ! b_seen) t = White; // see white only -> white territory
		else if (! w_seen && b_seen) t = Black; // -> black_territory
		for (vector<Loc>::const_iterator i = l.begin(); i != l.end(); i ++) if (! b[* i]) terri[* i] = t; // set terri except for stones
	}
	// the above is an over estimate, some empty points linking to hanging groups need to be filled (fake eyes)
        bool rr; // any changes? (iteratively eliminate fake eyes...)
        do {
                rr = false;
                // for blacks
		for (Loc i = 0; i < r; i ++) marks[i] = false; // clear the marking board
                for (Loc i = 0; i < r; i ++) if (! marks[i] && b[i] == Black && ! terri[i]) { // a live black stone
			vector<Loc> qi; // territory qi points encounterred
			hang_group(i, White, qi, terri, marks);
			if (qi.size() == 1 && ! b[qi[0]]) { // one qi only... need to be connected
				// FIXME unless it's removed stone point, then we will leave it along
				terri[qi[0]] = Empty;
				rr = true;
			}
			while (qi.size() > 0) { // unmark the territory qi points, as they can be shared by groups
				marks[qi.back()] = false;
				qi.pop_back();
			}
		}
		// do the same for white
		for (Loc i = 0; i < r; i ++) marks[i] = false; // clear the marking board
		for (Loc i = 0; i < r; i ++) if (! marks[i] && b[i] == White && ! terri[i]) {
			vector<Loc> qi;
			hang_group(i, Black, qi, terri, marks);
			if (qi.size() == 1 && ! b[qi[0]]) {
				terri[qi[0]] = Empty;
				rr = true;
			}
			while (qi.size() > 0) {
				marks[qi.back()] = false;
				qi.pop_back();
			}
		}
	} while (rr);
	delete marks;
}

void Board::get_group(Loc loc, vector<Loc> & group) const
{
	if (! b[loc]) return;
	unsigned r = range();
	bool * marks = new bool[r];
	for (Loc i = 0; i < r; i ++) marks[i] = false;
	qi_group(loc, group, b[loc], marks);
	delete marks;
}

void Board::dead_group(Loc loc, vector<Loc> & group) const
{
	assert(b[loc]);
	unsigned r = range();
	bool * marks = new bool[r];
	for (Loc i = 0; i < r; i ++) marks[i] = false;
	terri_group(loc, b[loc], group, marks);
	delete marks;
}
void Board::find_terri(Loc loc, vector<Loc> & group, bool & w_seen, bool & b_seen, State * terri, bool * marks) const
// recursively find connected empty space including killed stones
{
	if (b[loc] && ! terri[loc]) { // an occupied site not marked dead
		if (b[loc] == White) w_seen = true;
		else b_seen = true;
		return;
	}
	marks[loc] = true;
	group.push_back(loc);
	set<Loc> n = neighbors(loc);
	for (set<Loc>::iterator i = n.begin(); i != n.end(); i ++) {
		if (marks[* i]) continue;
		find_terri(* i, group, w_seen, b_seen, terri, marks);
	}
}

void Board::hang_group(Loc loc, State state, vector<Loc> & qi, State * terri, bool * marks) const
// find breath of a hanging group surrounded by state
{
	if (terri[loc]) { // a territory, count it as a breath
		qi.push_back(loc);
		marks[loc] = true;
		return;
	}
	marks[loc] = true;
	if (b[loc] == state) return;
	set<Loc> n = neighbors(loc);
	for (set<Loc>::iterator i = n.begin(); i != n.end(); i ++) {
		if (marks[* i]) continue;
		hang_group(* i, state, qi, terri, marks);
	}
}

void Board::terri_group(Loc loc, State state, vector<Loc> & group, bool * marks) const
{
	marks[loc] = true;
	if (b[loc] == State(flip(Turn(state)))) return;
	if (b[loc] == state) group.push_back(loc);
	set<Loc> n = neighbors(loc);
	for (set<Loc>::iterator i = n.begin(); i != n.end(); i ++) if (! marks[* i]) terri_group(* i, state, group, marks);
}
