/* ccgo: gmp/player.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include <iostream>
#include "player.hh"
#include "../handicap.hh"
using namespace gmp;

unsigned Player::answer(Query q)
{
	if (! arena) return 0;
	const go::Game * g = arena->get_game();
	unsigned ans;
	switch (q) {
	case QUERY_GAME:
		ans = 1; // go
		break;
	case QUERY_BUFFER:
		ans = (BufferSize - 4) / 16;
		if (ans > 255) ans = 255;
		break;
	case QUERY_VERSION:
		ans = 0;
		break;
	case QUERY_STONES:
		ans = 0;
		for (unsigned x = 0; x < g->get_board_size(); x ++) for (unsigned y = 0; y < g->get_board_size(); y ++) if (g->get_site(go::Loc(x, y)) != go::STATE_EMPTY) ans ++;
		break;
	case QUERY_BLACK_TIME:
		ans = 0;
		break;
	case QUERY_WHITE_TIME:
		ans = 0;
		break;
	case QUERY_CHARSET:
		ans = 0;
		break;
	case QUERY_RULE:
		// std::cerr << "query: what' s the rule? -> japanese" << std::endl;
		ans = 1;
		break;
	case QUERY_HANDICAP:
		// std::cerr << "query: handicap -> " << g->get_handicap() << std::endl;
		ans = g->get_handicap();
		break;
	case QUERY_BOARD_SIZE:
		// std::cerr << "query: board_size -> " << g->get_board_size() << std::endl;
		ans = g->get_board_size();
		break;
	case QUERY_TIME_LIMIT:
		ans = 0;
		break;
	case QUERY_COLOR:
		// std::cerr << "query: color -> " << (play_turn == go::TURN_WHITE ? "white" : "black" ) << std::endl;
		ans = (play_turn == go::TURN_WHITE ? 2 : 1);
		break;
	case QUERY_WHO:
		ans = 0;
		break;
	default:
		ans = 0;
	}
	return ans;
}

bool Player::move_in(unsigned v)
{
	if (! playing) {
		return false;
	}
	v &= 0x1ff;
	return (v ? arena->request_put(this, go::Loc::distribute(arena->get_game()->get_board_size(), v - 1)) : arena->request_pass(this));
}

bool Player::undo_in(unsigned v)
{
	return false;
}

Player::~Player()
{
	// std::cerr << "deleting gmp::Player" << std::endl;
}

void Player::set_arena(go::Arena * a)
{
	go::Player::set_arena(a);
	if (arena->check_player(go::TURN_WHITE, this)) play_turn = go::TURN_WHITE;
	if (arena->check_player(go::TURN_BLACK, this)) play_turn = go::TURN_BLACK;
}

void Player::enable_player()
{
	go::Player::enable_player();
}

void Player::disable_player()
{
	go::Player::disable_player();
}

void Player::notify_setup(const go::Board & b)
{
}

void Player::notify_handicap(unsigned n)
{
	// std::cerr << "handicap notified! (shouldn't happened)" << std::endl;
}

void Player::notify_put(const go::Loc & l)
{
	unsigned i = l.expand(arena->get_game()->get_board_size());
	// std::cerr << "notifying put " << l.to_string() << std::endl;
	i ++;
	if (arena->get_game()->get_turn() == go::TURN_WHITE) i |= 0x200;
	queue_command(CMD_MOVE, i);
}

void Player::notify_pass()
{
	unsigned i = 0;
	if (arena->get_game()->get_turn() == go::TURN_WHITE) i |= 0x200;
	queue_command(CMD_MOVE, i);
}

void Player::notify_undo()
{
	queue_command(CMD_UNDO, 2);
	arena->request_undo(this);
}
