/* ccgo: gmp/device.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "device.hh"
#include <iostream>
#include <cstdlib>
#include <csignal>
extern "C" {
#include <sys/fcntl.h>
#include <sys/wait.h>
#include <unistd.h>
}

using namespace gmp;

bool Device::read_child(Glib::IOCondition c)
{
	if (! child_pid) return false;
	int s;
	if (waitpid(child_pid, & s, WNOHANG) != 0) {
		child_pid = 0;
		std::cerr << "gmp device hung up!" << std::endl;
		read_child_connection.disconnect();
		child_timeout_connection.disconnect();
		device_ok = false;
		child_exit.emit();
		return false;
	}
	unsigned char b[256];
	while ((s = read(from_child, b, 255)) > 0) {
		// std::cerr << "** read " << s << " bytes" << std::endl;
		for (int i = 0; i < s; i ++) {
			process(b[i]);
		}
	}
	if (conflict) {
		conflict = false;
		return true;
	}
	process_queue();
	if (explicit_ok) {
		if (! waiting_ok) {
			// std::cerr << "doing explicit OK" << std::endl;
			send(Packet(last_mine.get_serial(), last_peer.get_serial(), CMD_OK, 1023));
			explicit_ok = false;
		} else {
			// std::cerr << "sending last command" << std::endl;
			// send(sending);
			// waiting_time = 0;
		}
	} else {
		// std::cerr << "no explicit OK required" << std::endl;
	}
	return true;
}

bool Device::child_timeout()
{
	if (! waiting_ok) {
		process_queue();
		return true;
	}
	if (waiting_time > 30) { // timeout, resend last command
		// std::cerr << "resending command" << std::endl;
		send(sending);
		waiting_time = 0;
	} else waiting_time ++;
	return true;
}

void Device::process(unsigned char c)
{
	// std::cerr << unsigned(c) << ' ';
	if ((c & 0xfc) == 0 && ! extension) { // start byte
		buffer[0] = c;
		buffer_top = 1;
		return;
	}
	if (buffer_top == BufferSize) { // overflow
		extension = 0;
		buffer_top =0;
		return;
	}
	buffer[buffer_top] = c;
	++ buffer_top;
	if (buffer_top == 4 + extension) { // a packet
		explicit_ok = false;
		// std::cerr << "<= command" << std::endl;
		if ((unsigned char)((buffer[0] + buffer[2] + buffer[3]) | 0x80) != buffer[1]) { // checksum
			// dump packet
			std::cerr << "fail checksum" << std::endl;
			extension = 0;
			buffer_top = 0;
			return;
		}
		unsigned peer_serial = buffer[0] & 1;
		unsigned mine_serial = (buffer[0] & 2) >> 1;
		// std::cerr << "serial : " << peer_serial << " " << mine_serial << std::endl;
		Cmd cmd = Cmd((buffer[2] >> 4) & 0x07);
		unsigned value = ((buffer[2] & 0x07) << 7) + (buffer[3] & 0x7f);
		// std::cerr << "cmd = " << cmd << "  value = " << value << std::endl;
		if (cmd == CMD_EXTENDED) {
			if (extension == 0) {
				extension = value;
				return;
			}
			extension = 0;
			unsigned char sum = buffer[4];
			for (unsigned i = 2; i < extension;  i ++) sum += buffer[i + 4];
			sum |= 0x80;
			if (sum != buffer[5]) {
				// failed checksum
				std::cerr << " fail checksum ext" << std::endl;
				buffer_top = 0;
				return;
			}
		}
		Packet p(peer_serial, mine_serial, cmd, value);
		if (cmd == CMD_EXTENDED) p.set_ext(buffer + 4);
		if (! waiting_ok) { // not sending a command
			if (mine_serial != last_mine.get_serial()) { // doesn't match mine_serail -> not possible
				std::cerr << "serial doesn't match mine" << mine_serial << " " << last_mine.get_serial() << std::endl;
				buffer_top = 0;
				return;
			}
			if (peer_serial == last_peer.get_serial()) { // same peer_serail
				// repeated cmd, might need to resend OK
				if (p == last_peer) {
					std::cerr << "repeated cmd" << std::endl;
					explicit_ok = true;
				} else {
					std::cerr << "inconsistent last peer!!!" << std::endl;
				}
				buffer_top =0;
				return;
			}
			if (! cmd) { // repeated OK, discard
				buffer_top = 0;
				return;
			}
			// new command received ok
			// std::cerr << "new command from peer" << std::endl;
			last_peer = p;
			buffer_top = 0;
			explicit_ok = true;
       			process_peer();
			return;
		}
		// waiting for OK
		if (! cmd) { // a peer OK
			if (peer_serial != last_peer.get_serial() || mine_serial != sending.get_serial()) { // not possible
				std::cerr << "drop impossible OK!" << std::endl;
				buffer_top = 0;
				return;
			}
			// a good peer OK
			// std::cerr << "a good peer OK" << std::endl;
			last_mine = sending;
			cmd_queue.pop();
			waiting_ok = false;
			return;
		}
		// a peer command
		if (mine_serial == sending.get_serial()) { // seen my sending
			if (peer_serial == last_peer.get_serial()) { // old command, not possible
				// discard
				std::cerr << "old peer, seen mine" << std::endl;
				buffer_top = 0;
				return;
			}
			// new peer command
			last_mine = sending;
			cmd_queue.pop();
			waiting_ok = false;
			last_peer = p;
			buffer_top = 0;
			explicit_ok = true;
			process_peer();
			return;
		}
		// not seen my sending
		if (peer_serial == last_peer.get_serial()) { // old command, my ok/cmd not seen
			// might need to resend OK
			std::cerr << "old command, not seen my OK" << std::endl;
			explicit_ok = true;
			buffer_top = 0;
			return;
		}
		std::cerr << "conflict!!" << std::endl;
		// new command, not seen my sending -> this is a conflict!!!
		conflict = true;
		waiting_ok = false;
		explicit_ok = false;
		buffer_top = 0;
		std::cerr << "peer cmd = " << cmd << ", value = " << value << std::endl;
		std::cerr << "mine cmd = " << sending.get_cmd() << ", value = " << sending.get_value() << std::endl;
		return;
	}
}

void Device::process_peer()
{
	// std::cerr << "peer cmd = "  << last_peer.get_cmd() << "; my queue size = " << cmd_queue.size() << std::endl;
	switch (last_peer.get_cmd()) {
	case CMD_DENY:
		denied();
		break;
	case CMD_NEWGAME:
		newgame();
		// std::cerr << explicit_ok << std::endl;
		break;
	case CMD_QUERY:
		queue_command(CMD_ANSWER, answer(Query(last_peer.get_value())));
		break;
	case CMD_ANSWER:
		respond(last_peer.get_value());
		break;
	case CMD_MOVE:
		if (! move_in(last_peer.get_value())) queue_command(CMD_DENY, 0);
		break;
	case CMD_UNDO:
		if (! undo_in(last_peer.get_value())) queue_command(CMD_DENY, 0);
		break;
	case CMD_EXTENDED:
		if (! process_ext()) queue_command(CMD_DENY, 0);
		break;
	default: // no way
		;
	}
}

void Device::send(const Packet & p)
{
	if (! device_ok) return;
	unsigned char buf[4];
	buf[0] = 0x00 | (p.get_peer() << 1) | p.get_serial();
	buf[2] = 0x80 | ((unsigned char)(p.get_cmd()) << 4) | (p.get_value() >> 7);
	buf[3] = 0x80 | (p.get_value() & 0x7f);
	buf[1] = (buf[0] + buf[2] + buf[3])  | 0x80;
	// std::cerr << "sending cmd = " << p.get_cmd() << "  value = " << p.get_value() << std::endl;
	write(to_child, buf, 4);
	if (p.get_cmd() != CMD_EXTENDED) return;
	write(to_child, p.get_ext(), p.get_value());
}

unsigned Device::answer(Query q)
{
	// std::cerr << "QUERY (" << q << ")" << std::endl;
	return 0;
}

void Device::denied()
{
	std::cerr << "DENY from peer, clear cmd_queue" << std::endl;
	while (! cmd_queue.empty()) cmd_queue.pop();
	if (waiting_ok) {
		std::cerr << "still waiting_ok -> not possible!" << std::endl;
		waiting_ok = false;
	}
}

void Device::respond(unsigned r)
{
	std::cerr << "response from peer -> " << r << std::endl;
}

bool Device::move_in(unsigned m)
{
	std::cerr << "MOVE from peer : " << m << std::endl;
	return false;
}

bool Device::undo_in(unsigned n)
{
	std::cerr << "UNDO from peer : " << n << std::endl;
	return false;
}

void Device::newgame()
{
	std::cerr << "NEWGAME from peer" << std::endl;
}

bool Device::process_ext()
{
	// std::cerr << "EXTENDED from peer -> deny" << std::endl;
	return false;
}

void Device::process_queue()
{
	if (waiting_ok || cmd_queue.empty()) return;
	Packet p = cmd_queue.front();
	p.set_serial(last_mine.get_serial() ^ 0x01);
	p.set_peer(last_peer.get_serial());
	sending = p;
	send(sending);
	waiting_ok = true;
	waiting_time = 0;
	explicit_ok = false;
}

void Device::queue_command(Cmd c, unsigned v, unsigned char * e)
{
	// std::cerr << "queue command: " << c << std::endl;
	Packet p(0, 0, c, v);
	if (e) p.set_ext(e);
	cmd_queue.push(p);
	process_queue();
}

Device::Device() :
	device_ok(false),
	buffer(0)
{
}

Device::~Device()
{
	// std::cerr << "deleting gmp::Device" << std::endl;
	if (child_pid) { // child is still alive
		kill(child_pid, SIGHUP); // kill child
		int s;
		waitpid(child_pid, & s, 0); // read the return value
		// std::cerr << "->child return " << s << std::endl;
		read_child_connection.disconnect();
		child_timeout_connection.disconnect();
		// std::cerr << "signals disconnected" << std::endl;
	} else {
		// std::cerr << "->child dead already." << std::endl;
	}
	if (buffer) delete[] buffer;
}

bool Device::fork_child(const std::string & n)
{
	device_ok = false;
	// std::cerr << "forking child" << std::endl;
	int c2p[2];
	int p2c[2];
	pipe(c2p);
	pipe(p2c);
	child_pid = fork();
	if (child_pid == -1) return false;
	if (child_pid == 0) { // for child
		close(0); // close cin
		dup(p2c[0]); // from parent
		close(1); // close cout
		dup(c2p[1]); // to parent
		close(p2c[0]);
		close(p2c[1]);
		close(c2p[0]);
		close(c2p[1]);
		// close(2); // close cerr
		const char * shell = "/bin/bash";
		std::string c = std::string("exec ") + n;
		execl(shell, shell, "-c", c.c_str(), (void *) 0);
		// exec failed!
		std::cerr << "Fail to run " << n << "!" << std::endl;
		exit(EXIT_FAILURE);
	}
	close(p2c[0]);
	to_child = p2c[1];
	from_child = c2p[0];
	close(c2p[1]);
	// set nonblock on reading child
	fcntl(from_child, F_SETFL, O_NONBLOCK);
	// std::cerr << "establish connections" << std::endl;
	read_child_connection = Glib::signal_io().connect(sigc::mem_fun(* this, & Device::read_child), from_child, Glib::IO_IN | Glib::IO_HUP);
	child_timeout_connection = Glib::signal_timeout().connect(sigc::mem_fun(* this, & Device::child_timeout), 100);
	if (! buffer) buffer = new unsigned char[BufferSize];
	buffer_top = 0;
	device_ok = true;
	waiting_ok = false;
	explicit_ok = false;
	conflict = false;
	extension = 0;
	return true;
}

Device::operator bool() const
{
	return device_ok;
}
