/* ccgo: game.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "game.hh"

using namespace go;

Game::QiData::QiData(unsigned z)
{
	size = z;
	mark = new bool[size * size];
}

Game::QiData::~QiData()
{
	delete[] mark;
}

void Game::QiData::clear()
{
	for (unsigned i = 0; i < size * size; i ++) mark[i] = false;
}

bool & Game::QiData::operator[] (const Loc & p)
{
	return mark[p.expand(size)];
}

int Game::qi_group(const Loc & l, std::vector<Loc> & g, QiData & qd) const
{
	if (! l.in_range(get_board_size()) || qd[l]) return 0;
	qd[l] = true;
	if ((* board)[l] == Board::other(qd.state)) return 0;
	if ((* board)[l] == STATE_EMPTY) return 1; // one qi
	g.push_back(l);
	int q = 0;
	q += qi_group(l.dir1(), g, qd);
	q += qi_group(l.dir2(), g, qd);
	q += qi_group(l.dir3(), g, qd);
	q += qi_group(l.dir4(), g, qd);
	return q;
}

Game::ScoreData::ScoreData(unsigned z, Terri * t)
{
	size = z;
	terri = t;
	mark = new bool[z * z];
}

Game::ScoreData::~ScoreData()
{
	delete[] mark;
}

Terri & Game::ScoreData::t(const Loc & l)
{
	return terri[l.expand(size)];
}

bool & Game::ScoreData::m(const Loc & l)
{
	return mark[l.expand(size)];
}

void Game::ScoreData::m_clear()
{
	for (unsigned i = 0; i < size * size; i ++) mark[i] = false;
}

void Game::terri_group(const Loc & l, std::vector<Loc> & g, bool & w, bool & b, ScoreData & sd) const
{
	if (! l.in_range(get_board_size()) || sd.m(l)) return;
	if (get_site(l) && ! sd.t(l)) {
		if (get_site(l) == STATE_WHITE) w = true;
		else b = true;
		return;
	}
	sd.m(l) = true;
	g.push_back(l);
	terri_group(l.dir1(), g, w, b, sd);
	terri_group(l.dir2(), g, w, b, sd);
	terri_group(l.dir3(), g, w, b, sd);
	terri_group(l.dir4(), g, w, b, sd);
}

void Game::hang_group(const Loc & l, State o, std::vector<Loc> & q, ScoreData & sd) const
{
	if (! l.in_range(get_board_size()) || sd.m(l)) return;
	if (sd.t(l)) {
		q.push_back(l);
		sd.m(l) = true;
		return;
	}
	if (get_site(l) == o) return;
	sd.m(l) = true;
	hang_group(l.dir1(), o, q, sd);
	hang_group(l.dir2(), o, q, sd);
	hang_group(l.dir3(), o, q, sd);
	hang_group(l.dir4(), o, q, sd);
}

Game::Game()
{
	board = NULL;
	rule = rule_igs;
	komi = 0;
	handicap = 0; // black decide
	move_num = 0;
	turn = TURN_BLACK;
	cap_w = 0;
	cap_b = 0;
	ko = Loc();
	last_move = Loc();
}

Game::Game(const Game & s)
{
	rule = s.rule;
	board = s.board->dup();
	komi = s.komi;
	move_num = s.move_num;
	turn = s.turn;
	cap_w = s.cap_w;
	cap_b = s.cap_b;
	ko = s.ko;
	last_move = s.last_move;
}

Game::~Game()
{
	if (board != NULL) delete board;
	board = NULL;
	// std::cerr << "deleting Game" << std::endl;
}

const Rule & Game::get_rule() const
{
	return rule;
}

void Game::set_rule(const Rule & r)
{
	rule = r;
}

double Game::get_komi() const
{
	return komi;
}

void Game::set_komi(double k)
{
	komi = k;
}

unsigned Game::get_handicap() const
{
	return handicap;
}

void Game::set_handicap(unsigned h)
{
	handicap = h;
}

unsigned Game::get_board_size() const
{
	if (board == NULL) return 0;
	return board->get_size();
}

void Game::set_board_size(unsigned s)
{
	if (board != NULL) delete board;
	board = new Board(s);
	move_num = 0;
	turn = TURN_BLACK;
	cap_w = 0;
	cap_b = 0;
	ko = Loc();
	last_move = Loc();
}

const std::string & Game::get_name() const
{
	return name;
}

void Game::set_name(const std::string & n)
{
	name = n;
}

void Game::set_board(const Board * b)
{
	if (! b || ! board) return;
	board->copy(* b);
}

unsigned Game::get_move_num() const
{
	return move_num;
}

void Game::set_move_num(unsigned n)
{
	move_num = n;
}

Turn Game::get_turn() const
{
	return turn;
}

void Game::set_turn(Turn t)
{
	turn = t;
}

const Loc & Game::get_ko() const
{
	return ko;
}

void Game::set_ko(const Loc & l)
{
	ko = l;
}

const Loc & Game::get_last_move() const
{
	return last_move;
}

void Game::set_last_move(const Loc & l)
{
	last_move = l;
}

State Game::get_site(const Loc & l) const
{
	return board->get_site(l);
}

void Game::set_site(const Loc & l, State s)
{
	board->set_site(l, s);
}

unsigned Game::get_cap_w() const
{
	return cap_w;
}

void Game::set_cap_w(unsigned c)
{
	cap_w = c;
}

unsigned Game::get_cap_b() const
{
	return cap_b;
}

void Game::set_cap_b(unsigned c)
{
	cap_b = c;
}

void Game::get_group(const Loc & p, std::vector<Loc> & g) const
{
	if (! get_site(p)) return;
	int z = get_board_size();
	QiData qd(z);
	qd.clear();
	qd.state = get_site(p);
	qi_group(p, g, qd);
}

void Game::get_kill(const Loc & p, std::vector<Loc> & k) const
{
	int z = get_board_size();
	QiData qd(z);
	qd.state = Board::stone(flip(turn));
	if (k.size() > 0) k.clear();
	if (p.dir1().in_range(z)) k.push_back(p.dir1());
	if (p.dir2().in_range(z)) k.push_back(p.dir2());
	if (p.dir3().in_range(z)) k.push_back(p.dir3());
	if (p.dir4().in_range(z)) k.push_back(p.dir4());
	for (int nc = k.size(); nc > 0;) {
		Loc pp = k.front();
		k.erase(k.begin());
		int bn = k.size();
		qd.clear();
		qd[p] = true;
		int q = qi_group(pp, k, qd);
		nc --;
		for (int i = 0; i < nc; ++ i) if (qd[k[i]]) {
			k.erase(k.begin() + i);
			i --;
			nc --;
			bn --;
		}
		if (q > 0) k.resize(bn);
	}
}

bool Game::check_space(const Loc & l) const
{
	if (l == ko) return false;
	return (* board)[l] == STATE_EMPTY;
}

bool Game::check_eye(const Loc & l) const
{
	State s = Board::stone(flip(turn));
	int z = board->get_size();
	return ! ((l.dir1().in_range(z) && (* board)[l.dir1()] != s) ||
		  (l.dir2().in_range(z) && (* board)[l.dir2()] != s) ||
		  (l.dir3().in_range(z) && (* board)[l.dir3()] != s) ||
		  (l.dir4().in_range(z) && (* board)[l.dir4()] != s));
}

int Game::check_fill(const Loc & l, std::vector<Loc> & k) const
{
	int z = board->get_size();
	QiData qd(z);
	qd.state = Board::stone(turn);
	qd.clear();
	qd[l] = true;
	k.clear();
	k.push_back(l);
	int q = 0;
	q += qi_group(l.dir1(), k, qd);
	q += qi_group(l.dir2(), k, qd);
	q += qi_group(l.dir3(), k, qd);
	q += qi_group(l.dir4(), k, qd);
	return q;
}

void Game::score_terri(Terri * t) const
	// t should be setup by caller, with killed stones marked as territory
{
	unsigned z = get_board_size();
	ScoreData sd(z, t);
	for (Loc x(0, 0); x; x.inc(z)) if (sd.t(x) && ! get_site(x)) sd.t(x) = TERRI_NEUTRAL; // clear TERRI except for the stones
	sd.m_clear(); // marking board
	for (Loc x(0, 0); x; x.inc(z)) if (! sd.m(x) && (! get_site(x) || sd.t(x))) {  // empty|killed nonmarked site
		std::vector<Loc> l; // l should be empty
		bool w_seen = false; // white encountered
		bool b_seen = false; // black encountered
		terri_group(x, l, w_seen, b_seen, sd); // explore the empty space
		if (w_seen && ! b_seen) for (std::vector<Loc>::const_iterator i = l.begin(); i != l.end(); i ++) sd.t(* i) = TERRI_WHITE;
		// see white but not black->white territory
		if (! w_seen && b_seen) for (std::vector<Loc>::const_iterator i = l.begin(); i != l.end(); i ++) sd.t(* i) = TERRI_BLACK;
		// see black but not white->black territory
	}

	// the above is an over estimate, some empty points linking to hanging groups need to be filled (fake eyes)
	if (! rule.no_fake_eyes) return; // don't eliminate fake eyes

	bool rr; // any changes? (iteratively eliminate fake eyes...
	do {
		rr = false;
		// for blacks
		sd.m_clear(); // clear the marking board
		for (Loc x(0, 0); x; x.inc(z)) if (! sd.m(x) && get_site(x) == STATE_BLACK && ! sd.t(x)) { // start with black stone
			std::vector<Loc> q; // territory qi points encounterred
			hang_group(x, STATE_WHITE, q, sd);
			if (q.size() == 1 && ! get_site(q[0])) { // one qi only... need to be connected
				// unless it's removed stone point, then we want to keep it
				sd.t(q[0]) = TERRI_NEUTRAL;
				rr = true;
			}
			while (q.size() > 0) { // unmark the territory qi points, as they can be shared by groups
				sd.m(q.back()) = false;
				q.pop_back();
			}
		}
		// do the same for white
		sd.m_clear();
		for (Loc x(0, 0); x; x.inc(z)) if (! sd.m(x) && get_site(x) == STATE_WHITE && ! sd.t(x)) {
			std::vector<Loc> q;
			hang_group(x, STATE_BLACK, q, sd);
			if (q.size() == 1 && ! get_site(q[0])) {
				sd.t(q[0]) = TERRI_NEUTRAL;
				rr = true;
			}
			while (q.size() > 0) {
				sd.m(q.back()) = false;
				q.pop_back();
			}
		}
	} while (rr);
}
