/* ccgo: ccgo.cc
 * 
 * Copyright (C) 2002-2008 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include <gtk/gtk.hh>
#include <go/sgf0.hh>
#include <debug.hh>

#include <config.hh>
extern "C" {
#include <gettext.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
}
#include <cstdio>
#include <cstdlib>
#include <game_setup.hh>
#include <igs/win.hh>
#include <setting_win.hh>
#include <arg.hh>

#include <gtkmm/box.h>
#include <gtkmm/window.h>
#include <gtkmm/main.h>
#include <gtkmm/fileselection.h>
#include <gtkmm/stock.h>
#include <gtkmm/image.h>

#include <gconfmm/init.h>

#include <iostream>
#include <fstream>

#define _(String) Glib::locale_to_utf8(gettext(String))

class ccGo :
	public Gtk::Window
{
public:
	ccGo();
	~ccGo();

	void open_sgf_file(std::string sgf_file);
	void set_control(int fd);
private:
	struct OpenView {
		ccgo::Game * game;
		gtk::View view;
	};

	Gtk::FileSelection * filesel;
	go::GameSetup * setup;
	go::SettingWin * setting;
	void open_sgf();
	void open_sgf_done();
	void view_closed(OpenView * ov);
	void new_game();
	void open_igs();
	void set_properties();
	void clear_properties();

	// remote control
	Glib::RefPtr<Glib::IOChannel> control_io;
	bool control_callback(Glib::IOCondition io_condition);

	std::vector<OpenView *> view_list;
	ccgo::List game_list;
};

ccGo::ccGo() :
	filesel(0),
	setup(0),
	setting(0)
{

	Gtk::VBox * vb;
	Gtk::Button *bt;
	vb = Gtk::manage(new Gtk::VBox);
	add(* vb);

	std::string ip = go::settings.get_image_path();

	if (Glib::file_test(ip + "/banner.png", Glib::FILE_TEST_EXISTS)) {
		Gtk::Image * im;
		im = Gtk::manage(new Gtk::Image(Gdk::Pixbuf::create_from_file(ip + "/banner.png")));
		vb->pack_start(* im, Gtk::PACK_SHRINK);
	} else {
		Gtk::Label * lb;
		lb = Gtk::manage(new Gtk::Label(_("ccGo")));
		vb->pack_start(* lb, Gtk::PACK_SHRINK);
	}

	set_title("ccGo");
	if (Glib::file_test(ip + "/main_icon.png", Glib::FILE_TEST_EXISTS)) {
		set_icon(Gdk::Pixbuf::create_from_file(ip + "/main_icon.png"));
	}

	bt = Gtk::manage(new Gtk::Button(_("Load sgf file")));
	vb->pack_start(* bt, Gtk::PACK_SHRINK);
	bt->signal_clicked().connect(mem_fun(* this, & ccGo::open_sgf));

	bt = Gtk::manage(new Gtk::Button(_("New game")));
	vb->pack_start(* bt, Gtk::PACK_SHRINK);
	bt->signal_clicked().connect(mem_fun(* this, & ccGo::new_game));

	bt = Gtk::manage(new Gtk::Button(_("Connect to IGS")));
	vb->pack_start(* bt, Gtk::PACK_SHRINK);
	bt->signal_clicked().connect(mem_fun(* this, & ccGo::open_igs));

	bt = Gtk::manage(new Gtk::Button(Gtk::Stock::PROPERTIES));
	vb->pack_start(* bt, Gtk::PACK_SHRINK);
	bt->signal_clicked().connect(mem_fun(* this, & ccGo::set_properties));

	bt = Gtk::manage(new Gtk::Button(Gtk::Stock::QUIT));
	vb->pack_start(* bt, Gtk::PACK_SHRINK);
	bt->signal_clicked().connect(mem_fun(* this, & Window::hide));

}

ccGo::~ccGo()
{
	if (filesel) delete filesel;
	if (setup) delete setup;
}

void ccGo::open_sgf_file(std::string fn)
{
	msg(DBG_INFO) << "opening sgf file: " << fn << '\n';
	ccgo::Game * game = game_list.load_game(fn);
	if (game) {
		OpenView * ov = new OpenView;
		// ov->game = game;
		ov->view.set_list(& game_list);
		ov->view.set_node(game);
		view_list.push_back(ov);
		ov->view.signal_hide().connect(bind(mem_fun(* this, & ccGo::view_closed), ov));
		ov->view.show_all();
	}
}

void ccGo::set_control(int fd)
{
	Glib::signal_io().connect(sigc::mem_fun(* this, & ccGo::control_callback), fd, Glib::IO_IN);
	control_io = Glib::IOChannel::create_from_fd(fd);
}

void ccGo::open_sgf()
{
	if (! filesel) {
		filesel = new Gtk::FileSelection(_("load sgf file"));
		filesel->hide_fileop_buttons();
		filesel->get_cancel_button()->signal_clicked().connect(mem_fun(* filesel, & Widget::hide));
		filesel->get_ok_button()->signal_clicked().connect(mem_fun(* this, & ccGo::open_sgf_done));
	}
	filesel->show_all();
	filesel->complete("*.sgf");
}

void ccGo::open_sgf_done()
{
	if (! filesel) return;
	std::string fn = filesel->get_filename();
	filesel->hide();
	open_sgf_file(fn);
}

void ccGo::view_closed(OpenView * ov)
{
	for (std::vector<OpenView *>::iterator i = view_list.begin(); i != view_list.end(); i ++) if (* i == ov) {
		// game_list.del_game(ov->game);
		delete ov;
		view_list.erase(i);
		return;
	}
	// error!!!
}

void ccGo::new_game()
{
	if (! setup) {
		setup = new go::GameSetup();
	}
	setup->show_all();
}

void ccGo::open_igs()
{
	new igs::Win();
}

void ccGo::set_properties()
{
	if (! setting) {
		setting = new go::SettingWin;
		setting->reset.connect(mem_fun(* this, & ccGo::clear_properties));
	}
	setting->show();
}

void ccGo::clear_properties()
{
	if (! setting) return;
	setting->hide();
	delete setting;
	go::settings.reset();
	setting = 0;
	set_properties();
}


bool ccGo::control_callback(Glib::IOCondition io_condition)
{
	if ((io_condition & Glib::IO_IN) == 0) {
		msg(DBG_ERROR) << "Invalid fifo response\n";
		return true;
	}

	Glib::ustring buf;
	control_io->read_line(buf);
	std::string cmd = buf;
	if (cmd[cmd.length() - 1] == '\n') {
		cmd = cmd.substr(0, cmd.length() - 1);
	}
	if (cmd == "QUIT") {
		hide();
	} else if (cmd.substr(0, 8) == "OPENSGF ") {
		open_sgf_file(cmd.substr(8));
	} else if (cmd == "WAKEUP") {
		present();
	} else {
		msg(DBG_ERROR) << "Unknown command: " << cmd << "\n";
	}
	return true;
}

int main(int argc, char * argv[], char * envp[])
{
	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
	arg::Parser p;

	// setting up command line options
	p.add_help(_("Usage: ccgo [OPTION...] [SGF_FILE]"));
	p.add_help(_("Launch the ccGo program for playing the game of Go [and open the game record, SGF_FILE]"));
	p.add_help("");

	p.add_opt('d', "debug")
		.store(arg::SV(debug_level))
		.help(_("setting debug level to NUM"), _("NUM"));

	std::string cmd_str;
	p.add_opt('c', "command")
		.store(arg::SV(cmd_str))
		.help(_("send CMD as remote command to existing ccgo process"), _("CMD"));

	p.add_opt_help();
	p.add_opt_version(PACKAGE_STRING);

	p.add_help("");
	char buf[100];
	snprintf(buf, 100, _("Please report bugs to %s").c_str(), PACKAGE_BUGREPORT);
	p.add_help(buf);

	try {
		p.parse(argc, argv);
	} catch (arg::Error e) {
		std::cerr << e.get_msg() << '\n';
		return - 1;
	}

	std::string sgf_file;
	for (std::vector<std::string>::const_iterator i = p.args().begin(); i != p.args().end(); i ++) {
		const std::string & s = * i;
		if (s.length() < 4) continue;
		if (s.substr(s.length() - 4) == ".sgf") {
			sgf_file = s;
			break;
		}
	}

	// make command socket name "ccgo_${DISPLAY}"
	std::string socket_name = std::string("/tmp/ccgo_");
	char * str = getenv("DISPLAY");
	if (str) {
		std::string display = str;
		// should canonicalize hostname
		socket_name += display;
	}
	msg(DBG_INFO) << "fifo name = " << socket_name << '\n';
	// check socket
	struct stat sock_stat;
	stat(socket_name.c_str(), & sock_stat);
	if (S_ISFIFO(sock_stat.st_mode)) { // sock exists, send command & exit
		int fd = open(socket_name.c_str(), O_WRONLY | O_NONBLOCK);
		if (fd >= 0) {
			msg(DBG_INFO) << "sending command through fifo\n";
			if (sgf_file.size() && ! cmd_str.size()) {
				cmd_str = "OPENSGF ";
				cmd_str += sgf_file;
			}
			if (! cmd_str.size()) {
				cmd_str = "WAKEUP";
			}
			cmd_str += '\n';
			write(fd, cmd_str.c_str(), cmd_str.size());
			close(fd);
			return 0;
		}
		assert(errno == ENXIO);
		msg(DBG_INFO) << "remove dangling command pipe\n";
		if (unlink(socket_name.c_str())) {
			std::cerr << "error removing fifo\n";
			return - 1;
		}
	}

	if (cmd_str.size()) { // command but no existing ccgo process
		std::cerr << "no fifo for existing ccgo process\n";
		return -1;
	}

	// create sock
	msg(DBG_INFO) << "creating fifo\n";
	if (mkfifo(socket_name.c_str(), 0660)) {
		std::cerr << "can not created fifo\n";
		return - 1;
	}

	msg(DBG_INFO) << "opening fifo\n";
	int control_fd = open(socket_name.c_str(), O_RDONLY | O_NONBLOCK);
	if (control_fd == -1) {
		std::cerr << "fail to open fifo\n";
		return - 1;
	}
	// hack to prevent POLLHUP. Is there a way to clear HUP condition
	// after the remote end close the FIFO?
	int write_fd = open(socket_name.c_str(), O_WRONLY);

	msg(DBG_INFO) << "initializing\n";
	Gtk::Main m(& argc, & argv);

	go::settings.init();
	go::settings.command_line(argc, argv, envp);

	ccGo ccgo;
	ccgo.show_all();

	if (sgf_file.size()) ccgo.open_sgf_file(sgf_file);

	msg(DBG_INFO) << "setting control\n";
	ccgo.set_control(control_fd);
	Gtk::Main::run(ccgo);

	close(control_fd);
	close(write_fd);
	
	if (unlink(socket_name.c_str())) {
		std::cerr << "error removing fifo\n";
		return - 1;
	}
	return 0;
}
