/* arg.hh
 *
 * Copyright (C) 2004-2008 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * 
 * See arg.cc for a list of changes
 */
#ifndef ARG_HH
#define ARG_HH 1
#include <vector>
#include <map>
#include <string>
#include <sstream>
#include <typeinfo>
namespace arg {
	// proxy to values of command line options, need to know where to store the values
	class Value
	{
	public:
		virtual ~Value();
		virtual void set(const std::string & str); // convert the str to value and put it in storage
		virtual std::string to_str() const; // convert the value to a string
	};

	// signature for callback functions
	typedef bool (CallBack)(int, const std::string & , void *);

	class Option
	{
		int key;
		std::string name;

		Value * store_ptr; // pointer to storage space
		bool store_optional; // if value string is optional
		std::string store_str; // default value string

		int * set_var; // variable to set
		int set_value; // value to set
		bool set_once; // if can only set once
		int set_init; // initial value, 

		CallBack * call_func; // callback function
		void * call_data; // data to pass to callback function

		std::string help_text;
		std::string help_var;
		bool help_default; // whether to show default value of store
	public:
		Option(int key, const std::string & name);
		~Option();

		Option & store(Value * ptr = 0); // store value to "* ptr", the Value will be released by the Option
		Option & optional(const std::string & str = ""); // value is optional defaulting to "str"
		Option & set(int * var, int value = - 1); // set "* var" to "value"
		Option & once(int init = 0); // can only be set once, with distinct value, "init"
		Option & call(CallBack * func, void * data); // call function "* func" with "data" as extra argument
		Option & help(const std::string & text, const std::string & var = ""); // help text
		Option & help_word(const std::string & var);
		Option & show_default(bool do_show = true);

		bool take_value();
		bool need_value();
		int get_key();
		const std::string & get_name();

		enum HelpFormat {
			HF_REGULAR,
			HF_NODASH
		};
		std::string get_help(HelpFormat format = HF_REGULAR);

		void process();
		void process(const std::string & str);
	};

	class Parser
	{
		std::string header_text;
		std::string version_info;
	protected:
		std::vector<Option *> opt_list;
		std::vector<std::string> arg_list;
		struct HelpLine {
			std::string msg;
			Option * opt;
		};
		std::vector<HelpLine> help_list;
	public:
		~Parser();
		void add_help(const std::string & msg);
		Option & add_opt(int key, const std::string & name, bool hide = false);
		Option & add_opt(const std::string & name, bool hide = false);
		std::vector<std::string> & args();
		void parse(int argc, char * argv[], bool ignore_unknown = false);
		void set_header(const std::string & text);
		const std::string & get_header() const;

		// find existing options
		Option * find(int key);
		Option * find(const std::string & name);

		// removing options
		void remove(int key);
		void remove(const std::string & name);
		void remove_all(); // remove all options

		std::string get_help();
		// default options
		Option & add_opt_help();
		Option & add_opt_version(const std::string & version);
	};

	class SubParser :
		public Value,
		public Parser
	{
	public:
		Option & add_opt(const std::string & name);
		void set(const std::string & str);

		std::string get_help();

		// default options
		Option & add_opt_help();
	};

	// Errors:

	class Error
	{
	protected:
		std::string msg;
	public:
		Error();
		Error(const std::string & msg);
		std::string get_msg();
	};

	class OptError : // option processing error
		public Error
	{
	protected:
		std::string opt;
	public:
		OptError(const std::string & opt);
		OptError(const std::string & opt, const std::string & msg);
	};

	class ConvError : // conversion error
		public Error
	{
	public:
		ConvError(const std::string & str, const std::string & type);
	};

	class UnknError : // unknow option error
		public Error
	{
	public:
		UnknError(const std::string & opt);
	};

	// Templates:

	// value types that have << and >> defined for istream/ostream
	template <typename T>
	class StreamableValue :
		public Value
	{
		T & ptr;
	public:
		StreamableValue(T & t) :
			ptr(t)
		{
		}

		void set(const std::string & str)
		{
			std::istringstream s(str);
			T tmp;
			s >> tmp;
			if (s.bad() || ! s.eof()) throw ConvError(str, typeid(T).name());
			ptr = tmp;
		}

		std::string to_str() const
		{
			std::ostringstream s;
			s << ptr;
			return s.str();
		}
	};

	template <typename T>
	StreamableValue<T> * SV(T & t)
	{
		return new StreamableValue<T>(t);
	}

	// Extensions:

	// Set of names in strings
	class SetValue :
		public Value
	{
		int & var;
		std::string help_title;
		bool help_default;
		struct Element {
			std::string name;
			int value;
			std::string help;
		};
		std::vector<Element> set_list;
	public:
		SetValue(int & var);
		void add_help(const std::string & title = "Available values:");
		void add_help(const std::string & title, int value);
		void add(const std::string & name, const std::string & help = "");
		void add(const std::string & name, int value, const std::string & help = "");

		void set(const std::string & str);
		std::string to_str() const;

		// additional access to set
		int get_value(const std::string & name) const;
		const std::string & get_name(int value) const;
		const std::string & get_help(const std::string & name) const;
		const std::string & get_help(int value) const;
		std::string get_help() const;
	};

	// List of values seperated by comma
	template <typename T>
	class ListValue :
		public Value
	{
		std::vector<T> & plist;
		char sep;
	public:
		ListValue(std::vector<T> & list, char seperator = ',') :
			plist(list),
			sep(seperator)
		{
		}

		void set(const std::string & str)
		{
			plist.clear();
			std::string::size_type n = 0;
			while (n < str.length()) {
				std::string::size_type m = str.find(sep, n);
				if (m == std::string::npos) m = str.length();
				std::istringstream i(str.substr(n, m - n));
				T v;
				i >> v;
				plist.push_back(v);
				n = m + 1;
			}
		}

		std::string to_str() const
		{
			std::ostringstream o;
			for (typename std::vector<T>::const_iterator i = plist.begin(); i != plist.end(); i ++) {
				if (i != plist.begin()) o << sep;
				o << * i;
			}
			return o.str();
		}
	};

	// double that can be relative (if it begins with '+' sign)
	class RelValue :
		public Value
	{
		double & v;
		bool & rel;
	public:
		RelValue(double & var, bool & is_relative);
		void set(const std::string & str);
		std::string to_str() const;
	};
}
#endif // ARG_HH
