/* ccgo: arena.cc
 * 
 * Copyright (C) 2002,2003 Chun-Chung Chen <cjj@u.washington.edu>
 * 
 * This file is part of ccGo.
 * 
 * ccGo is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */

#include "arena.hh"

using namespace go;

Arena::Arena()
{
	white = NULL;
	black = NULL;
	in_action = false;
	pass_count = 0;
}

Arena::~Arena()
{
	// std::cerr << "deleting arena" << std::endl;
}

void Arena::set_player(Turn t, Player * p)
{
	if (t == TURN_WHITE) {
		white = p;
	} else {
		black = p;
	}
	if (p) p->set_arena(this);
}

bool Arena::check_player(Turn t, Player *p)
{
	return (t == TURN_WHITE) ? white == p : black == p;
}

const Game * Arena::get_game() const
{
	return  NULL;
}

void Arena::start_action()
{
	if (! get_game()) return;
	if (white) white->enable_player();
	if (black) black->enable_player();
	in_action = true;
}

void Arena::stop_action()
{
	// if (! in_action) return;
	if (white) white->disable_player();
	if (black) black->disable_player();
	in_action = false;
}

void Arena::enter_score()
{
	if (in_action) stop_action();
	in_scoring = true;
	if (white) white->enter_scoring();
	if (black) black->enter_scoring();
}

void Arena::done_score()
{
	in_scoring = false;
	if (white) white->done_scoring();
	if (black) black->done_scoring();
}

bool Arena::request_handicap(Player * p, unsigned n)
{
	if (! in_action || p != (get_game()->get_turn() == TURN_WHITE ? white : black)) return false;
	if (! action_handicap(n)) return false;
	(get_game()->get_turn() == TURN_WHITE ? white : black)->notify_handicap(n);
	pass_count = 0;
	return true;
}

bool Arena::request_put(Player * p, const Loc & l)
{
	if (! in_action ||p != (get_game()->get_turn() == TURN_WHITE ? white : black)) return false;
	Loc loc = l;
	if (! action_put(loc)) return false;
	(get_game()->get_turn() == TURN_WHITE ? white : black)->notify_put(loc);
	pass_count = 0;
	return true;
}

bool Arena::request_pass(Player * p)
{
	if (! in_action || p != (get_game()->get_turn() == TURN_WHITE ? white : black)) return false;
	if (! action_pass()) return false;
	(get_game()->get_turn() == TURN_WHITE ? white : black)->notify_pass();
	pass_count ++;
	if (pass_count == 3) {
		stop_action();
		enter_score();
		pass_count = 0;
	}
	return true;
}

bool Arena::request_undo(Player * p) // dual action
{
	if (! in_action || p != (get_game()->get_turn() == TURN_WHITE ? white : black)) return false;
	if (! action_undo()) return false;
	(get_game()->get_turn() == TURN_WHITE ? white : black)->notify_undo();
	pass_count = 0;
	return true;
}

bool Arena::request_terri(Player * p, const Loc & l)
{
	if (! in_scoring) return false;
	Loc loc = l;
	if (! action_terri(loc)) return false;
	white->notify_terri(loc);
	black->notify_terri(loc);
	return true;
}

bool Arena::request_reset(Player * p)
{
	if (! in_scoring) return false;
	if (! action_reset()) return false;
	if (white && p != white) white->notify_reset();
	if (black && p != black) black->notify_reset();
	return true;
}

bool Arena::request_done(Player * p)
{
	if (! in_scoring) return false;
	if (! action_done()) return false;
	if (white && p != white) white->notify_done();
	if (black && p != black) black->notify_done();
	return true;
}

bool Arena::action_handicap(unsigned n)
{
	return false;
}

bool Arena::action_put(const Loc & l)
{
	return false;
}

bool Arena::action_pass()
{
	return false;
}

bool Arena::action_undo()
{
	return false;
}

bool Arena::action_terri(const Loc & l)
{
	return false;
}

bool Arena::action_reset()
{
	return false;
}

bool Arena::action_done()
{
	return false;
}

void Arena::ask_terri(Terri *)
{
}
