local cassowary = require("cassowary")

describe('cassowary.Constraint', function ()

  it('should create expression equations', function ()
    local ex = cassowary.Expression(10)
    local c1 = cassowary.Equation(ex)
    assert.same(c1.expression, ex)
  end)

  it('can create expressions cassowary.Variable instances', function ()
    local x = cassowary.Variable({ value = 167 })
    local y = cassowary.Variable({ value = 2 })
    local cly = cassowary.Expression(y)
    cly:addExpression(x)
  end)

  it('can create equations from variables and expressions', function ()
    local x = cassowary.Variable({ name = 'x', value = 167 })
    local cly = cassowary.Expression(2)
    local eq = cassowary.Equation(x, cly)
    assert.is.truthy(eq.expression:equals(cly:minus(x)))
  end)

  -- it('should handle strengths correctly', function ()
  --   local solver = cassowary.SimplexSolver()
  --   local x = cassowary.Variable({ name = 'x', value = 10 })
  --   local y = cassowary.Variable({ name = 'y', value = 20 })
  --   local z = cassowary.Variable({ name = 'z', value = 1 })
  --   local w = cassowary.Variable({ name = 'w', value = 1 })

  --   -- Default weights.
  --   local e0 = cassowary.Equation(x, y)
  --   solver.addStay(y)
  --   solver.addConstraint(e0)
  --   assert.is.truthy(cassowary.approx(x, 20))
  --   assert.is.truthy(cassowary.approx(y, 20))

  --   -- Weak.
  --   local e1 = cassowary.Equation(x, z, cassowary.Strength.weak)
  --   -- console.log('x:', x.value)
  --   -- cassowary.trace = true
  --   solver.addStay(x)
  --   solver.addConstraint(e1)
  --   assert.is.truthy(cassowary.approx(x, 20))
  --   assert.is.truthy(cassowary.approx(z, 20))

  --   -- Strong.
  --   local e2 = cassowary.Equation(z, w, cassowary.Strength.strong)
  --   solver.addStay(w)
  --   solver.addConstraint(e2)
  --   assert.deepEqual(w.value, 1)
  --   assert.deepEqual(z.value, 1)
  -- end)

  it('can use numbers in place of variables', function ()
    local v = cassowary.Variable({ name = 'v', value = 22 })
    local eq = cassowary.Equation(v, 5)
    assert.is.truthy(eq.expression:equals(cassowary.minus(5, v)))
  end)

  it('can use equations in place of variables', function ()
    local e = cassowary.Expression(10)
    local v = cassowary.Variable({ name = 'v', value = 22 })
    local eq = cassowary.Equation(e, v)
    assert.is.truthy(eq.expression:equals(cassowary.minus(10, v)))
  end)

  it('works with nested expressions', function ()
    local e1 = cassowary.Expression(10)
    local e2 = cassowary.Expression(cassowary.Variable({ name = 'z', value = 10 }), 2, 4)
    local eq = cassowary.Equation(e1, e2)
    assert.is.truthy(eq.expression:equals(e1:minus(e2)))
  end)

  it('instantiates inequality expressions correctly', function ()
    local e = cassowary.Expression(10)
    local ieq = cassowary.Inequality(e)
    assert.is.same(ieq.expression, e)
  end)

  it('handles inequality constructors with operator arguments', function ()
    local v1 = cassowary.Variable({ name = 'v1', value = 10 })
    local v2 = cassowary.Variable({ name = 'v2', value = 5 })
    local ieq = cassowary.Inequality(v1, ">=", v2)
    assert.is.truthy(ieq.expression:equals(cassowary.minus(v1, v2)))
    ieq = cassowary.Inequality(v1, "<=", v2)
    assert.is.truthy(ieq.expression:equals(cassowary.minus(v2, v1)))
  end)

  it('handles expressions with variables, operators, and numbers', function ()
    local v = cassowary.Variable({ name = 'v', value = 10 })
    local ieq = cassowary.Inequality(v, ">=", 5)
    assert.is.truthy(ieq.expression:equals(cassowary.minus(v, 5)))
    local ieq2 = cassowary.Inequality(v, "<=", 5)
    assert.is.truthy(ieq2.expression:equals(cassowary.minus(5, v)))
  end)

  it('handles inequalities with reused variables', function ()
    local e1 = cassowary.Expression(10)
    local e2 = cassowary.Expression(cassowary.Variable({ name = 'c', value = 10 }), 2, 4)
    local ieq = cassowary.Inequality(e1, ">=", e2)
    assert.is.truthy(ieq.expression:equals(e1:minus(e2)))
    local ieq2 = cassowary.Inequality(e1, "<=", e2)
    assert.is.truthy(ieq2.expression:equals(e2:minus(e1)))
  end)

  -- it('handles constructors with variable/operator/expression args', function ()
  -- 	local v = cassowary.Variable({ name = 'v', value = 10 })
  -- 	local e = cassowary.Expression(cassowary.Variable({ name = 'x', value = 5 }), 2, 4)
  -- 	local ieq = cassowary.Inequality(v, cassowary.GEQ, e)
  -- 	assert.is.truthy(ieq.expression.equals(cassowary.minus(v, e)))
  -- 	ieq = cassowary.Inequality(v, cassowary.LEQ, e)
  -- 	assert.is.truthy(ieq.expression.equals(e.minus(v)))
  -- end)

  -- it('handles constructors with expression/operator/variable args', function ()
  -- 	local v = cassowary.Variable({ name = 'v', value = 10 })
  -- 	local e = cassowary.Expression(cassowary.Variable({ name = 'x', value = 5 }), 2, 4)
  -- 	local ieq = cassowary.Inequality(e, cassowary.GEQ, v)
  -- 	assert.is.truthy(ieq.expression.equals(e.minus(v)))
  -- 	ieq = cassowary.Inequality(e, cassowary.LEQ, v)
  -- 	assert.is.truthy(ieq.expression.equals(cassowary.minus(v, e)))
  -- end)

end)
