/*-
 * Copyright (c) 2006, 2008 Iain Hibbert
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
#include <sys/types.h>
#include <sys/un.h>

#include <bluetooth.h>
#include <err.h>
#include <qapplication.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qprogressbar.h>
#include <qsocketnotifier.h>
#include <qspinbox.h>
#include <qtimer.h>

#include "btclient.h"
#include "pindialog.h"

BtClient::BtClient( int & argc, char ** argv ) : QApplication ( argc, argv )
{
	struct sockaddr_un un;
	int ch, s, len;

	memset(&un, 0, sizeof(un));
	un.sun_len = sizeof(un);
	un.sun_family = AF_LOCAL;
	strlcpy(un.sun_path, BTHCID_SOCKET_NAME, sizeof(un.sun_path));

	bdaddr_copy(&localMatch, BDADDR_ANY);
	bdaddr_copy(&remoteMatch, BDADDR_ANY);
	defaultPIN = NULL;
	randomLength = 4;

	while ((ch = getopt(argc, argv, "a:d:l:p:s:")) != EOF) {
		switch (ch) {
		case 'a':
			if (!bt_aton(optarg, &remoteMatch)) {
			    	struct hostent  *he = NULL;

			    	if ((he = bt_gethostbyname(optarg)) == NULL)
				    	errx(EXIT_FAILURE, "Invalid address %s", optarg);

				bdaddr_copy(&remoteMatch, (bdaddr_t *)he->h_addr);
			}
			break;

		case 'd':
    			if (!bt_devaddr(optarg, &localMatch))
				errx(EXIT_FAILURE, "Invalid address %s", optarg);

			break;

		case 'l':
			len = atoi(optarg);
			if (len < 1 || len > HCI_PIN_SIZE)
				errx(EXIT_FAILURE, "Invalid length: %s\n", optarg);

			randomLength = len;
			break;

		case 'p':
			if (strlen(optarg) > HCI_PIN_SIZE)
				errx(EXIT_FAILURE, "Invalid PIN %s", optarg);

			defaultPIN = optarg;
			break;

		case 's':
			strlcpy(un.sun_path, optarg, sizeof(un.sun_path));
			break;

		default:
			errx(EXIT_FAILURE, "unknown option '%c'", ch);
		}
	}

	s = socket(PF_LOCAL, SOCK_STREAM, 0);
	if (s < 0)
		err(EXIT_FAILURE, "socket");

	if (::connect(s, (struct sockaddr *)&un, sizeof(un)) < 0)
		err(EXIT_FAILURE, "connect(\"%s\")", un.sun_path);

	QSocketNotifier* n = new QSocketNotifier(s, QSocketNotifier::Read);
	connect( n, SIGNAL(activated(int)), SLOT(trigger(int)) );
}

BtClient::~BtClient()
{
}

void BtClient::trigger(int s)
{
	PINDialog d;
	QTimer t;
	bthcid_pin_request_t req;
	bthcid_pin_response_t rsp;
	size_t len;

	len = recv(s, &req, sizeof(req), MSG_WAITALL);
	if (len != sizeof(req)
	    || (!bdaddr_any(&localMatch) && !bdaddr_same(&localMatch, &req.laddr))
	    || (!bdaddr_any(&remoteMatch) && !bdaddr_same(&remoteMatch, &req.raddr)))
	    	return;

	d.localAddr->setText(bt_ntoa(&req.laddr, NULL));
	d.remoteAddr->setText(bt_ntoa(&req.raddr, NULL));

	d.randomLength->setValue(randomLength);

	srandom(time(NULL));
	if (defaultPIN)
	    	d.pinLineEdit->setText(defaultPIN);
	else
		d.randompin();

	d.pinLineEdit->selectAll();

	d.progressBar->setMaximum(req.time * 10);
	connect( &t, SIGNAL(timeout()), &d, SLOT(tickle()) );
	t.start(100);

	d.exec();

	randomLength = d.randomLength->value();

	if (d.result() != PINDialog::Accepted)
		return;

	memset(&rsp, 0, sizeof(rsp));
	bdaddr_copy(&rsp.laddr, &req.laddr);
	bdaddr_copy(&rsp.raddr, &req.raddr);
	strncpy((char *)rsp.pin,
		qPrintable(d.pinLineEdit->text().trimmed()),
		HCI_PIN_SIZE);

	if (send(s, &rsp, sizeof(rsp), 0) != sizeof(rsp))
		err(EXIT_FAILURE, "send failed");
}
