// distribution boxbackup-0.11.1 (svn version: 2821_2827)
// Box Backup, http://www.boxbackup.org/
// 
// Copyright (c) 2003-2010, Ben Summers and contributors.
// All rights reserved.
// 
// Note that this project uses mixed licensing. Any file with this license
// attached, or where the code LICENSE-DUAL appears on the first line, falls
// under this license. See the file COPYING.txt for more information.
// 
// This file is dual licensed. You may use and distribute it providing that you
// comply EITHER with the terms of the BSD license, OR the GPL license. It is
// not necessary to comply with both licenses, only one.
// 
// The BSD license option follows:
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  
// 3. Neither the name of the Box Backup nor the names of its contributors may
//    be used to endorse or promote products derived from this software without
//    specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 
// [http://en.wikipedia.org/wiki/BSD_licenses#3-clause_license_.28.22New_BSD_License.22.29]
// 
// The GPL license option follows:
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
// 
// [http://www.gnu.org/licenses/old-licenses/gpl-2.0.html#SEC4]
// --------------------------------------------------------------------------
//
// File
//		Name:    RaidFileUtil.h
//		Purpose: Utilities for the raid file classes
//		Created: 2003/07/11
//
// --------------------------------------------------------------------------

#ifndef RAIDFILEUTIL__H
#define RAIDFILEUTIL__H

#include "RaidFileController.h"
#include "RaidFileException.h"

// note: these are hardcoded into the directory searching code
#define RAIDFILE_EXTENSION			".rf"
#define RAIDFILE_WRITE_EXTENSION	".rfw"

// --------------------------------------------------------------------------
//
// Class
//		Name:    RaidFileUtil
//		Purpose: Utility functions for RaidFile classes
//		Created: 2003/07/11
//
// --------------------------------------------------------------------------
class RaidFileUtil
{
public:
	typedef enum 
	{
		NoFile = 0,
		NonRaid = 1,
		AsRaid = 2,
		AsRaidWithMissingReadable = 3,
		AsRaidWithMissingNotRecoverable = 4
	} ExistType;
	
	enum
	{
		Stripe1Exists = 1,
		Stripe2Exists = 2,
		ParityExists = 4
	};
	
	static ExistType RaidFileExists(RaidFileDiscSet &rDiscSet, const std::string &rFilename, int *pStartDisc = 0, int *pExisitingFiles = 0, int64_t *pRevisionID = 0);
	
	static int64_t DiscUsageInBlocks(int64_t FileSize, const RaidFileDiscSet &rDiscSet);
	
	// --------------------------------------------------------------------------
	//
	// Function
	//		Name:    std::string MakeRaidComponentName(RaidFileDiscSet &, const std::string &, int)
	//		Purpose: Returns the OS filename for a file of part of a disc set
	//		Created: 2003/07/11
	//
	// --------------------------------------------------------------------------	
	static inline std::string MakeRaidComponentName(RaidFileDiscSet &rDiscSet, const std::string &rFilename, int Disc)
	{
		if(Disc < 0 || Disc >= (int)rDiscSet.size())
		{
			THROW_EXCEPTION(RaidFileException, NoSuchDiscSet)
		}
		std::string r(rDiscSet[Disc]);
		r += DIRECTORY_SEPARATOR_ASCHAR;
		r += rFilename;
		r += RAIDFILE_EXTENSION;
		return r;
	}
	
	// --------------------------------------------------------------------------
	//
	// Function
	//		Name:    std::string MakeWriteFileName(RaidFileDiscSet &, const std::string &)
	//		Purpose: Returns the OS filename for the temporary write file
	//		Created: 2003/07/11
	//
	// --------------------------------------------------------------------------	
	static inline std::string MakeWriteFileName(RaidFileDiscSet &rDiscSet, const std::string &rFilename, int *pOnDiscSet = 0)
	{
		int livesOnSet = rDiscSet.GetSetNumForWriteFiles(rFilename);
		
		// does the caller want to know which set it's on?
		if(pOnDiscSet) *pOnDiscSet = livesOnSet;
		
		// Make the string
		std::string r(rDiscSet[livesOnSet]);
		r += DIRECTORY_SEPARATOR_ASCHAR;
		r += rFilename;
		r += RAIDFILE_WRITE_EXTENSION;
		return r;
	}
};

#endif // RAIDFILEUTIL__H

