/*
 *  bktr2jpeg - write jpeg image captured from /dev/bktr to file
 *
 * TODO: - initialise the card (you still need fxtv to do this after reboot)
 *
 */

/* Copyright (c) 1999, 2000, 2001, 2002 Thomas Runge (coto@core.de)
 *
 * Based on grab.c by Roger Hardiman, videocapture.c by Amancio Hasty,
 *  libjpeg.doc by the Independent JPEG Group
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of its contributors
 *    may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/param.h>
#include <sys/time.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>

#ifdef __cplusplus
extern "C"
{
#endif

#include <jpeglib.h>

#ifdef __cplusplus
}
#endif

#ifdef __NetBSD__
#include <dev/ic/bt8xx.h>
#endif /* NetBSD */

#ifdef __FreeBSD__
#include <machine/ioctl_meteor.h>
#include <machine/ioctl_bt848.h>
#endif /* __FreeBSD__ */

/* which device */
#define VIDEO_DEV  "/dev/bktr"
#define TUNER_DEV  "/dev/tuner"

/* PAL is 768 x 576. NTSC is 640 x 480 */
#define PAL_MAXCOLS 768
#define PAL_MAXROWS 576
#define NTSC_MAXCOLS 640
#define NTSC_MAXROWS 480

/* device needs some time to settle down */
#define SETTLE 1

int BRIGHT_MIN       = BT848_BRIGHTMIN;
int BRIGHT_MAX       = (BT848_BRIGHTMIN + BT848_BRIGHTRANGE);
int BRIGHT_CENTER    = BT848_BRIGHTCENTER;
int BRIGHT_RANGE     = BT848_BRIGHTRANGE;
int BRIGHT_DRV_MIN   = BT848_BRIGHTREGMIN;
int BRIGHT_DRV_RANGE = (BT848_BRIGHTREGMAX - BT848_BRIGHTREGMIN + 1);

int CONTRAST_MIN       = BT848_CONTRASTMIN;
int CONTRAST_MAX       = (BT848_CONTRASTMIN + BT848_CONTRASTRANGE);
int CONTRAST_CENTER    = BT848_CONTRASTCENTER;
int CONTRAST_RANGE     = BT848_CONTRASTRANGE;
int CONTRAST_DRV_MIN   = BT848_CONTRASTREGMIN;
int CONTRAST_DRV_RANGE = (BT848_CONTRASTREGMAX - BT848_CONTRASTREGMIN + 1);

int keep_running;
int verbose = 0;

struct capjpeg
{
	char *filename;
	char *tfilename;
	struct jpeg_error_mgr jerr;
	struct jpeg_compress_struct cinfo;
	JSAMPLE *rgb_buffer;
	char *copybuf;
	int cols, rows;
};

static void error(char *prefix, char *what)
{
	fprintf(stderr, "failure while saving jpeg\n");
	fprintf(stderr, "%s: %s\n\n", prefix, what);
	exit(EXIT_FAILURE);
}

static void done(int sig)
{
	keep_running = 0;
}

static void jpeg_init(struct capjpeg *ctx, int quality)
{
	ctx->cinfo.err = jpeg_std_error(&ctx->jerr);
	jpeg_create_compress(&ctx->cinfo);

	ctx->cinfo.image_width  = ctx->cols;
	ctx->cinfo.image_height = ctx->rows;
	ctx->cinfo.input_components = 3;
	ctx->cinfo.in_color_space = JCS_RGB;

	jpeg_set_defaults(&ctx->cinfo);

	jpeg_set_quality(&ctx->cinfo, quality, TRUE);
}

static void jpeg_encode(struct capjpeg *ctx)
{
	FILE *fp = stdout;
	JSAMPROW row_pointer[1];
	JSAMPLE *src = ctx->rgb_buffer;
	unsigned char *p;
	int i;

	if(ctx->tfilename)
	{
		if(verbose)
			fprintf(stderr, "opening savefile %s\n", ctx->tfilename);
		if((fp = fopen(ctx->tfilename, "wb")) == NULL)
			error("fopen", strerror(errno));
	}

	jpeg_stdio_dest(&ctx->cinfo, fp);

	jpeg_start_compress(&ctx->cinfo, TRUE);

	while(ctx->cinfo.next_scanline < ctx->cinfo.image_height)
	{
		p = ctx->copybuf;
		for(i = ctx->cols; i; i--)
		{
			*p++ = src[2];
			*p++ = src[1];
			*p++ = src[0];
			src += 4;
		}
		row_pointer[0] = ctx->copybuf;
		jpeg_write_scanlines(&ctx->cinfo, row_pointer, 1);
	}

	jpeg_finish_compress(&ctx->cinfo);

	if(verbose)
		fprintf(stderr, "wrote jpeg data\n");

	fflush(fp);
	if(ctx->tfilename)
	{
		fclose(fp);
		if(verbose)
			fprintf(stderr, "save file closed\n");
		if(rename(ctx->tfilename, ctx->filename) == -1)
			error("rename", strerror(errno));
		if(verbose)
			fprintf(stderr, "save file renamed to %s\n", ctx->filename);
	}
}

void setValues(int tuner, int value, unsigned long request,
				int min, int max, int range,
				int drvmin, int drvrange)
{
	int arg;

	value = MAX(min, MIN(max, value));
	if(verbose)
		fprintf(stderr, "%d", value);

	arg = (value - min) / (range + 0.01) * drvrange + drvmin;
	arg = MAX(drvmin, MIN(drvmin + drvrange - 1, arg));

	if(verbose)
		fprintf(stderr, " (%d)\n", arg);
	if(ioctl(tuner, request, &arg ) < 0 )
		error("ioctl", strerror(errno));
}

void camsleep(long sleep)
{
	struct timeval timeout;
	int i;

	timeout.tv_sec=1;
	timeout.tv_usec=0;

	for(i = 0; i < sleep; i++)
	{
		if(verbose)
		{
			fprintf(stderr, "\rsleeping %d secs.", sleep-i);
			fflush(stderr);
		}
		if(keep_running)
			select(0, NULL, NULL, NULL, &timeout);
	}
}

void usage(char *prg)
{
	fprintf(stderr, "Usage: %s [-f filename] [-s input] [-d device_number] [-w width] [-h height] [-q quality] [-c seconds] [-v]\n", prg);
	fprintf(stderr, "  -f: if you don't provide a filename, the jpeg will be written to stdout\n");
	fprintf(stderr, "  -s: which input, tuner is 1 (default), video is 0\n");
	fprintf(stderr, "  -d: which device number, 0 is default (which is your first grabber card)\n");
	fprintf(stderr, "  -w, -h: width and height, default is 320x200\n");
	fprintf(stderr, "  -b: Brightness (%d <-> %d  default: %d)\n", BRIGHT_MIN, BRIGHT_MAX, BRIGHT_CENTER);
	fprintf(stderr, "  -o: Contrast (%d <-> %d  default: %d)\n", CONTRAST_MIN, CONTRAST_MAX, CONTRAST_CENTER);
	fprintf(stderr, "  -q: jpeg quality (default is 75%%, which should be okay)\n");
	fprintf(stderr, "  -c is for continuous capturing and the argument specifies the time to sleep between two captures in seconds.\n");
	fprintf(stderr, "  -a channel to switch to.\n");
	fprintf(stderr, "  -v verbose output.\n");
	exit(EXIT_SUCCESS);
}

int main(int argc, char **argv)
{
	struct capjpeg actx, *ctx = &actx;

	char dev_video[MAXPATHLEN];
	char dev_tuner[MAXPATHLEN];
	int unit    = 0;
	int width   = 320;
	int height  = 240;
	int sleep   = -1;
	int quality = 75;
	int input   = 1;
	int channel = -1;
	int brightness = BRIGHT_CENTER;
	int contrast   = CONTRAST_CENTER;
	int single  = METEOR_CAP_SINGLE;
	int fmt, ch;

	int video    = -1;
	int tuner    = -1;

	struct meteor_geomet geo;

	ctx->filename = NULL;
	ctx->tfilename = NULL;
	keep_running = 1;

	bzero(ctx, sizeof(*ctx));

	signal(SIGPIPE, done);
	signal(SIGINT,  done);

	if((argc == 2) && !strcmp(argv[1], "-h"))
		usage(argv[0]);

	while((ch = getopt(argc, argv, "f:s:d:w:h:q:c:u:b:o:a:v")) != -1)
	{
		switch(ch)
		{
			case 'f':
				ctx->filename = optarg;
				break;
			case 's':
				input = atoi(optarg);
				break;
			case 'd':
				unit = atoi(optarg);
				break;
			case 'w':
				width = atoi(optarg);
				break;
			case 'h':
				height = atoi(optarg);
				break;
			case 'c':
				sleep = atoi(optarg);
				break;
			case 'b':
				brightness = atoi(optarg);
				if((brightness < BRIGHT_MIN) || (brightness > BRIGHT_MAX))
				{
					fprintf(stderr, "brightness out of range");
					exit(EXIT_FAILURE);
				}
				break;
			case 'o':
				contrast = atoi(optarg);
				if((contrast < CONTRAST_MIN) || (contrast > CONTRAST_MAX))
				{
					fprintf(stderr, "contrast out of range");
					exit(EXIT_FAILURE);
				}
				break;
			case 'q':
				quality = atoi(optarg);
				if(quality < 0 || quality > 100)
				{
					fprintf(stderr, "quality out of range");
					exit(EXIT_FAILURE);
				}
				break;
			case 'a':
				channel = atoi(optarg);
				break;
			case 'v':
				verbose = 1;
				break;
			default:
				usage(argv[0]);
				break;
		}
	}

	switch(input)
	{
		case 0:  input = METEOR_INPUT_DEV0; break;
		case 1:  input = METEOR_INPUT_DEV1; break;
		case 2:  input = METEOR_INPUT_DEV2; break;
		case 3:  input = METEOR_INPUT_DEV3; break;
		default: input = METEOR_INPUT_DEV1; break;
	}

	if(ctx->filename)
	{
		ctx->tfilename = (char*)malloc(strlen(ctx->filename + 2));
		sprintf(ctx->tfilename, "%s~", ctx->filename);
	}

	ctx->cols = width;
	ctx->rows = height;
	ctx->copybuf = (char*)malloc(ctx->cols*3);
	if(ctx->copybuf == NULL)
		error("malloc", strerror(errno));

	if((unsigned) ctx->cols > PAL_MAXCOLS || (unsigned) ctx->cols <= 0 ||
	   (unsigned) ctx->rows > PAL_MAXROWS || (unsigned) ctx->rows <= 0)
	{
		ctx->cols = PAL_MAXCOLS;
		ctx->rows = PAL_MAXROWS;
	}

	geo.columns = ctx->cols;
	geo.rows    = ctx->rows;
	geo.frames  = 1;
	geo.oformat = METEOR_GEO_RGB24;

	snprintf(dev_video, sizeof(dev_video), "%s%d", VIDEO_DEV, unit);
	snprintf(dev_tuner, sizeof(dev_tuner), "%s%d", TUNER_DEV, unit);

	if(verbose)
	{
		fprintf(stderr, "video device: %s\n", dev_video);
		fprintf(stderr, "tuner device: %s\n", dev_tuner);
	}

	if(verbose)
		fprintf(stderr, "opening video device\n");

	if((video = open(dev_video, O_RDONLY)) == -1)
		error("open video", strerror(errno));

	if(verbose)
		fprintf(stderr, "opening tuner device\n");

	if((tuner = open(dev_tuner, O_RDONLY)) == -1)
		error("open tuner", strerror(errno));

	if(ioctl(video, METEORGFMT, &fmt) == -1)
		error("ioctl METEORGFMT", "video GFMT");

	if(fmt == METEOR_FMT_NTSC)
	{
		if(verbose)
			fprintf(stderr, "detected NTSC format\n");
		if(geo.rows <= NTSC_MAXROWS / 2)
			geo.oformat |= METEOR_GEO_ODD_ONLY; 
	}

	if(fmt == METEOR_FMT_PAL)
	{
		if(verbose)
			fprintf(stderr, "detected PAL format\n");
		if(geo.rows <= PAL_MAXROWS / 2)
			geo.oformat |= METEOR_GEO_ODD_ONLY; 
	}

	if(ioctl(video, METEORSETGEO, &geo) == -1)
		error("ioctl METEORSETGEO", "video SETGEO");

	if(ioctl(video, METEORSFMT, &fmt) == -1)
		error("ioctl METEORSFMT", "video SFORMAT");

	if(ioctl(video, METEORSINPUT, &input) == -1)
		error("ioctl METEORSINPUT", "video SINPUT");

	if(verbose)
		fprintf(stderr, "setting brightness to: ");
	setValues(tuner, brightness, BT848_SBRIG,
				BRIGHT_MIN, BRIGHT_MAX, BRIGHT_RANGE,
				BRIGHT_DRV_MIN, BRIGHT_DRV_RANGE);

	if(verbose)
		fprintf(stderr, "setting contrast to: ");
	setValues(tuner, contrast, BT848_SCONT,
				CONTRAST_MIN, CONTRAST_MAX, CONTRAST_RANGE,
				CONTRAST_DRV_MIN, CONTRAST_DRV_RANGE);

	if(ioctl(tuner, TVTUNER_SETCHNL, &channel) < 0 )
		error("set_channel", strerror(errno));

	ctx->rgb_buffer = (JSAMPLE*) mmap(NULL, ctx->rows * ctx->cols * 4,
										PROT_READ, MAP_SHARED, video, 0);

	if(ctx->rgb_buffer == MAP_FAILED)
		error("mmap", strerror(errno));

	/* wait for the device to settle down */
	if(verbose)
		fprintf(stderr, "waiting for device to settle down\n");
	camsleep(SETTLE);

	jpeg_init(ctx, quality);

	if(sleep == -1)
	{
		ioctl(video, METEORCAPTUR, &single);
		jpeg_encode(ctx);
	}
	else
	{
		while(keep_running)
		{
			ioctl(video, METEORCAPTUR, &single);
			jpeg_encode(ctx);
			camsleep(sleep);
		}
	}

	jpeg_destroy_compress(&ctx->cinfo);

	if(ctx->tfilename)
		free(ctx->tfilename);

	if(verbose)
		fprintf(stderr, "closing tuner device\n");
	close(tuner);

	if(verbose)
		fprintf(stderr, "closing video device\n");
	close(video);

	exit(EXIT_SUCCESS);
}

