/*
 * Copyright (C) 2003 Clemens Fuchslocher <clfuit00@fht-esslingen.de>
 *
 * config.c - 17.04.2003 - v0.5 - sort order
 *            05.04.2003 - v0.4 - edit with middle mouse click
 *            05.04.2003 - v0.4 - bugfix - URL handler with special characters are now working
 *            05.04.2003 - v0.4 - default browser type
 *            07.03.2003 - v0.3 - dialog options
 *            01.03.2003 - v0.2 - "Recently Opened Documents"
 *            15.02.2003 - v0.1
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h>

#include <gtk/gtk.h>

#include <libxml/parser.h>
#include <libxml/xpath.h>

#include "config.h"

#include "bk_edit_misc.h"
#include "bk_edit_url_handler.h"

static void use_default_values (config *c);
static void set_property (config *c, char *xpath, char *property, char *value);
static char *get_property (config *c, char *xpath);


config *config_new (char *directory, char *filename)
{
	struct stat st;

	config *c = (config *) malloc (sizeof (config));
	if (c == NULL)
	{
		return NULL;
	}

	if (directory == NULL)
	{
		directory = bk_edit_misc_get_user_home_dir ();
	}

	if (filename == NULL)
	{
		filename = ".rc";
	}

	c->filename = g_strdup_printf ("%s/%s", directory, filename);
	if (stat (c->filename, &st) == -1)
	{
		use_default_values (c);
		return c;
	}

	xmlXPathInit ();

	c->doc = xmlParseFile (c->filename);
	if (c->doc == NULL)
	{
		free (c->filename);
		free (c);

		return NULL;
	}

	return c;
}


int config_save (config *c)
{
	if (xmlSaveFormatFile (c->filename, c->doc, 0) == -1)
	{
		return CONFIG_KO;
	}

	return CONFIG_OK;
}


void config_delete (config *c)
{
	xmlFreeDoc (c->doc);

	free (c->filename);
	free (c);
}


int config_recent_documents_list_size_get (config *c)
{
	return atoi (get_property (c, "string (/child::bk_edit/child::recently_opened_documents/attribute::size)"));
}


void config_recent_documents_list_size_set (config *c, int size)
{
	char *s = g_strdup_printf ("%d", size);
	set_property (c, "/child::bk_edit/child::recently_opened_documents", "size", s);
	free (s);
}


GList *config_recent_documents_list_get (config *c)
{
	GList *list = NULL;

	int n;

	xmlXPathContextPtr path_ctx = xmlXPathNewContext (c->doc);
	xmlXPathObjectPtr path = xmlXPathEvalExpression ("/child::bk_edit/child::recently_opened_documents/child::document", path_ctx);

	if (path->type != XPATH_NODESET)
	{
		return NULL;
	}

	if (path->nodesetval == NULL)
	{
		return NULL;
	}

	for (n = 0; n < path->nodesetval->nodeNr; n++)
	{
		xmlNodeSetPtr node_set = path->nodesetval;
		xmlNodePtr node = node_set->nodeTab[n]->children;

		config_recent_document *doc = (config_recent_document *) malloc (sizeof (config_recent_document));
		if (doc == NULL)
		{
			continue;
		}

		while (node != NULL)
		{
			if (node->type == XML_ELEMENT_NODE)
			{
				if (!strcmp (node->name, "filename"))
				{
					if (node->children != NULL)
					{
						doc->filename = strdup (node->children->content);
					}
					else
					{
						doc->filename = strdup ("");
					}
				}
				else if (!strcmp (node->name, "plugin"))
				{
					if (node->children != NULL)
					{
						doc->plugin = strdup (node->children->content);
					}
					else
					{
						doc->plugin = strdup ("");
					}
				}
			}

			node = node->next;
		}

		list = g_list_append (list, doc);
	}

	xmlXPathFreeObject (path);
	xmlXPathFreeContext (path_ctx);

	return list;
}


void config_recent_documents_list_set (config *c, GList *list)
{
	int n;

	GList *l = list;

	xmlXPathContextPtr path_ctx = xmlXPathNewContext (c->doc);
	xmlXPathObjectPtr path = xmlXPathEvalExpression ("/child::bk_edit/child::recently_opened_documents/child::*", path_ctx);
	xmlNodePtr node;

	if (path->type != XPATH_NODESET)
	{
		return;
	}

	if (path->nodesetval == NULL)
	{
		return;
	}

	for (n = 0; n < path->nodesetval->nodeNr; n++)
	{
		xmlNodeSetPtr node_set = path->nodesetval;
		xmlNodePtr node = node_set->nodeTab[n];

		xmlUnlinkNode (node);
		xmlFreeNode (node);
	}

	xmlXPathFreeObject (path);
	path = xmlXPathEvalExpression ("/child::bk_edit/child::recently_opened_documents", path_ctx);

	while (l != NULL)
	{
		node = xmlNewChild (path->nodesetval->nodeTab[0], NULL, "document", NULL);
		xmlNewChild (node, NULL, "filename", ((config_recent_document *) l->data)->filename);
		xmlNewChild (node, NULL, "plugin", ((config_recent_document *) l->data)->plugin);

		l = l->next;
	}

	xmlXPathFreeObject (path);
	xmlXPathFreeContext (path_ctx);
}


void config_recent_documents_list_free (config *c, GList *list)
{
	while (list != NULL)
	{
		free (((config_recent_document *) list->data)->filename);
		free (((config_recent_document *) list->data)->plugin);
		free ((config_recent_document *) list->data);

		list = list->next;
	}

	g_list_free (list);
}


GList *config_url_handlers_get (config *c)
{
	GList *url_handlers = NULL;

	int n;

	xmlXPathContextPtr path_ctx = xmlXPathNewContext (c->doc);
	xmlXPathObjectPtr path = xmlXPathEvalExpression ("/child::bk_edit/child::url_handlers/child::*", path_ctx);

	if (path->type != XPATH_NODESET)
	{
		return NULL;
	}

	if (path->nodesetval == NULL)
	{
		return NULL;
	}

	for (n = 0; n < path->nodesetval->nodeNr; n++)
	{
		xmlNodeSetPtr node_set = path->nodesetval;
		xmlNodePtr node = node_set->nodeTab[n]->children;

		config_url_handler *url_handler = (config_url_handler *) malloc (sizeof (url_handler));
		if (url_handler == NULL)
		{
			continue;
		}

		url_handler->default_handler = CONFIG_URL_HANDLER_NORMAL;
		if (node_set->nodeTab[n]->properties != NULL)
		{
			xmlAttrPtr attr = node_set->nodeTab[n]->properties;
			while (attr != NULL)
			{
				if (!strcmp (attr->name, "default"))
				{
					url_handler->default_handler = CONFIG_URL_HANDLER_DEFAULT;
				}

				attr = attr->next;
			}
		}

		while (node != NULL)
		{
			if (node->type == XML_ELEMENT_NODE)
			{
				if (!strcmp (node->name, "name"))
				{
					if (node->children != NULL)
					{
						url_handler->name = bk_edit_misc_utf8_to_latin1 (node->children->content);
					}
					else
					{
						url_handler->name = strdup ("");
					}
				}
				else if (!strcmp (node->name, "command"))
				{
					if (node->children != NULL)
					{
						url_handler->command = bk_edit_misc_utf8_to_latin1 (node->children->content);
					}
					else
					{
						url_handler->command = strdup ("");
					}
				}
			}

			node = node->next;
		}

		url_handlers = g_list_append (url_handlers, url_handler);

	}

	xmlXPathFreeObject (path);
	xmlXPathFreeContext (path_ctx);

	return url_handlers;
}


void config_url_handlers_set (config *c, GList *url_handlers)
{
	int n;

	GList *url_handler = url_handlers;

	xmlXPathContextPtr path_ctx = xmlXPathNewContext (c->doc);
	xmlXPathObjectPtr path = xmlXPathEvalExpression ("/child::bk_edit/child::url_handlers/child::*", path_ctx);
	xmlNodePtr node;

	if (path->type != XPATH_NODESET)
	{
		return;
	}

	if (path->nodesetval == NULL)
	{
		return;
	}

	for (n = 0; n < path->nodesetval->nodeNr; n++)
	{
		xmlNodeSetPtr node_set = path->nodesetval;
		xmlNodePtr node = node_set->nodeTab[n];

		xmlUnlinkNode (node);
		xmlFreeNode (node);
	}

	xmlXPathFreeObject (path);
	path = xmlXPathEvalExpression ("/child::bk_edit/child::url_handlers", path_ctx);

	while (url_handler != NULL)
	{
		char *name = bk_edit_misc_xml_entities_encode (((config_url_handler *) url_handler->data)->name);
		char *command = bk_edit_misc_xml_entities_encode (((config_url_handler *) url_handler->data)->command);

		node = xmlNewChild (path->nodesetval->nodeTab[0], NULL, "url_handler", NULL);

		xmlNewChild (node, NULL, "name", name);
		xmlNewChild (node, NULL, "command", command);

		free (name);
		free (command);

		if (((config_url_handler *) url_handler->data)->default_handler == CONFIG_URL_HANDLER_DEFAULT)
		{
			xmlNewProp (node, "default", "yes");
		}

		url_handler = url_handler->next;
	}

	xmlXPathFreeObject (path);
	xmlXPathFreeContext (path_ctx);
}


void config_url_handlers_free (config *c, GList *url_handlers)
{
	while (url_handlers != NULL)
	{
		free (((config_url_handler *) url_handlers->data)->name);
		free (((config_url_handler *) url_handlers->data)->command);
		free ((config_url_handler *) url_handlers->data);

		url_handlers = url_handlers->next;
	}

	g_list_free (url_handlers);
}


gboolean config_dialog_options_url_dropzone_always_on_top_get (config *c)
{
	return get_property (c, "string (/child::bk_edit/child::dialog_options/child::url_dropzone/attribute::always_on_top)")[0] == 'y' ? TRUE : FALSE;
}


void config_dialog_options_url_dropzone_always_on_top_set (config *c, gboolean state)
{
	set_property (c, "/child::bk_edit/child::dialog_options/child::url_dropzone", "always_on_top", state == TRUE ? "yes" : "no");
}


gboolean config_dialog_options_url_dropzone_sticky_get (config *c)
{
	return get_property (c, "string (/child::bk_edit/child::dialog_options/child::url_dropzone/attribute::sticky)")[0] == 'y' ? TRUE : FALSE;
}


void config_dialog_options_url_dropzone_sticky_set (config *c, gboolean state)
{
	set_property (c, "/child::bk_edit/child::dialog_options/child::url_dropzone", "sticky", state == TRUE ? "yes" : "no");
}


gboolean config_dialog_options_url_dropzone_size_state_get (config *c)
{
	return get_property (c, "string (/child::bk_edit/child::dialog_options/child::url_dropzone/child::size/attribute::use)")[0] == 'y' ? TRUE : FALSE;
}


void config_dialog_options_url_dropzone_size_state_set (config *c, gboolean state)
{
	set_property (c, "/child::bk_edit/child::dialog_options/child::url_dropzone/child::size", "use", state == TRUE ? "yes" : "no");
}


void config_dialog_options_url_dropzone_size_get (config *c, int *width, int *height)
{
	*width = atoi (get_property (c, "string (/child::bk_edit/child::dialog_options/child::url_dropzone/child::size/attribute::width)"));
	*height = atoi (get_property (c, "string (/child::bk_edit/child::dialog_options/child::url_dropzone/child::size/attribute::height)"));
}


void config_dialog_options_url_dropzone_size_set (config *c, int width, int height)
{
	char *s;

	s = g_strdup_printf ("%d", width);
	set_property (c, "/child::bk_edit/child::dialog_options/child::url_dropzone/child::size", "width", s);
	free (s);

	s = g_strdup_printf ("%d", height);
	set_property (c, "/child::bk_edit/child::dialog_options/child::url_dropzone/child::size", "height", s);
	free (s);
}


gboolean config_dialog_options_url_dropzone_position_state_get (config *c)
{
	return get_property (c, "string (/child::bk_edit/child::dialog_options/child::url_dropzone/child::position/attribute::use)")[0] == 'y' ? TRUE : FALSE;
}


void config_dialog_options_url_dropzone_position_state_set (config *c, gboolean state)
{
	set_property (c, "/child::bk_edit/child::dialog_options/child::url_dropzone/child::position", "use", state == TRUE ? "yes" : "no");
}


gboolean config_middle_mouse_get (config *c)
{
	return get_property (c, "string (/child::bk_edit/child::middle_mouse/attribute::use)")[0] == 'y' ? TRUE : FALSE;
}


void config_middle_mouse_set (config *c, gboolean state)
{
	set_property (c, "/child::bk_edit/child::middle_mouse", "use", state == TRUE ? "yes" : "no");
}


void config_dialog_options_url_dropzone_position_get (config *c, int *x, int *y)
{
	*x = atoi (get_property (c, "string (/child::bk_edit/child::dialog_options/child::url_dropzone/child::position/attribute::x)"));
	*y = atoi (get_property (c, "string (/child::bk_edit/child::dialog_options/child::url_dropzone/child::position/attribute::y)"));
}


void config_dialog_options_url_dropzone_position_set (config *c, int x, int y)
{
	char *s;

	s = g_strdup_printf ("%d", x);
	set_property (c, "/child::bk_edit/child::dialog_options/child::url_dropzone/child::position", "x", s);
	free (s);

	s = g_strdup_printf ("%d", y);
	set_property (c, "/child::bk_edit/child::dialog_options/child::url_dropzone/child::position", "y", s);
	free (s);
}


char *config_browser_type_get (config *c)
{
	return get_property (c, "string (/child::bk_edit/child::browser_type/attribute::type)");
}


void config_browser_type_set (config *c, char *type)
{
	set_property (c, "/child::bk_edit/child::browser_type", "type", type);
}


char *config_sort_order_get (config *c)
{
	return get_property (c, "string (/child::bk_edit/child::sort_order/attribute::order)");
}


void config_sort_order_set (config *c, char *order)
{
	set_property (c, "/child::bk_edit/child::sort_order", "order", order);
}


static void use_default_values (config *c)
{
	xmlNodePtr root, node, url_handlers, recently_opened_documents, dialog_options, url_dropzone, browser_type, middle_mouse, sort_order;

	c->doc = xmlNewDoc ("1.0");

	root = xmlNewNode (NULL, "bk_edit");
	xmlDocSetRootElement (c->doc, root);

	/* url handlers */
	url_handlers = xmlNewChild (root, NULL, "url_handlers", NULL);

	node = xmlNewChild (url_handlers, NULL, "url_handler", NULL);
	xmlNewChild (node, NULL, "name", "Mozilla (New Window)");
	xmlNewChild (node, NULL, "command", "mozilla -remote 'openURL(%u,new-window)'");

	node = xmlNewChild (url_handlers, NULL, "url_handler", NULL);
	xmlNewChild (node, NULL, "name", "Mozilla (Existing)");
	xmlNewChild (node, NULL, "command", "mozilla -remote 'openURL(%u)'");

	node = xmlNewChild (url_handlers, NULL, "url_handler", NULL);
	xmlNewChild (node, NULL, "name", "Mozilla (New Tab)");
	xmlNewChild (node, NULL, "command", "mozilla -remote 'openURL (%u,new-tab)'");
	xmlNewProp (node, "default", "yes");

	node = xmlNewChild (url_handlers, NULL, "url_handler", NULL);
	xmlNewChild (node, NULL, "name", "www.archive.org (Mozilla)");
	xmlNewChild (node, NULL, "command", "mozilla -remote 'openURL(http://web.archive.org/web/*/%u,new-tab)'");

	node = xmlNewChild (url_handlers, NULL, "url_handler", NULL);
	xmlNewChild (node, NULL, "name", "Phoenix (New Tab)");
	xmlNewChild (node, NULL, "command", "phoenix -remote 'openURL (%u,new-tab)'");

	node = xmlNewChild (url_handlers, NULL, "url_handler", NULL);
	xmlNewChild (node, NULL, "name", "Opera (New Window)");
	xmlNewChild (node, NULL, "command", "opera -remote 'openURL(%u,new-window)'");

	node = xmlNewChild (url_handlers, NULL, "url_handler", NULL);
	xmlNewChild (node, NULL, "name", "Galeon (New Tab)");
	xmlNewChild (node, NULL, "command", "galeon --new-tab %u");

	node = xmlNewChild (url_handlers, NULL, "url_handler", NULL);
	xmlNewChild (node, NULL, "name", "Netscape (Existing)");
	xmlNewChild (node, NULL, "command", "netscape -remote 'openURL(%u)'");

	node = xmlNewChild (url_handlers, NULL, "url_handler", NULL);
	xmlNewChild (node, NULL, "name", "Netscape (New Window)");
	xmlNewChild (node, NULL, "command", "netscape -remote 'openURL(%u,new-window)'");

	node = xmlNewChild (url_handlers, NULL, "url_handler", NULL);
	xmlNewChild (node, NULL, "name", "Lynx (xterm)");
	xmlNewChild (node, NULL, "command", "xterm -e lynx %u");

	/* recently opened documents */
	recently_opened_documents = xmlNewChild (root, NULL, "recently_opened_documents", NULL);
	xmlNewProp (recently_opened_documents, "size", "5");

	/* browser type */
	browser_type = xmlNewChild (root, NULL, "browser_type", NULL);
	xmlNewProp (browser_type, "type", "Mozilla v1.0, v1.1, v1.2, v1.3 and Netscape 6+");

	/* middle mouse click */
	middle_mouse = xmlNewChild (root, NULL, "middle_mouse", NULL);
	xmlNewProp (middle_mouse, "use", "yes");

	/* dialog options */
	dialog_options = xmlNewChild (root, NULL, "dialog_options", NULL);

	url_dropzone = xmlNewChild (dialog_options, NULL, "url_dropzone", NULL);
	xmlNewProp (url_dropzone, "sticky", "no");
	xmlNewProp (url_dropzone, "always_on_top", "no");

	node = xmlNewChild (url_dropzone, NULL, "size", NULL);
	xmlNewProp (node, "use", "no");
	xmlNewProp (node, "width", "230");
	xmlNewProp (node, "height", "29");

	node = xmlNewChild (url_dropzone, NULL, "position", NULL);
	xmlNewProp (node, "use", "no");
	xmlNewProp (node, "x", "10");
	xmlNewProp (node, "y", "10");

	/* sort order */
	sort_order = xmlNewChild (root, NULL, "sort_order", NULL);
	xmlNewProp (sort_order, "type", "Unsorted");
}


static void set_property (config *c, char *xpath, char *property, char *value)
{
	xmlXPathContextPtr path_ctx = xmlXPathNewContext (c->doc);
	xmlXPathObjectPtr path = xmlXPathEvalExpression (xpath, path_ctx);
	xmlNodePtr node;

	if (path->type != XPATH_NODESET)
	{
		return;
	}

	if (path->nodesetval == NULL)
	{
		return;
	}

	if (path->nodesetval->nodeNr == 0)
	{
		xmlNodePtr root = xmlDocGetRootElement (c->doc);

		xmlNodePtr browser_type = xmlNewChild (root, NULL, strrchr (xpath, ':') + 1, NULL);
		xmlNewProp (browser_type, property, value);

		return;
	}

	node = path->nodesetval->nodeTab[0];
	if (node == NULL)
	{
		return;
	}

	xmlSetProp (node, property, value);

	xmlXPathFreeObject (path);
	xmlXPathFreeContext (path_ctx);
}


static char *get_property (config *c, char *xpath)
{
	xmlXPathContextPtr path_ctx = xmlXPathNewContext (c->doc);
	xmlXPathObjectPtr path = xmlXPathEvalExpression (xpath, path_ctx);

	static char *value = NULL;

	if (path->type != XPATH_STRING)
	{
		return NULL;
	}

	if (value != NULL)
	{
		free (value);
	}

	value = strdup (path->stringval);

	xmlXPathFreeObject (path);
	xmlXPathFreeContext (path_ctx);

	return value;
}

